package pacman3d.util;

import java.util.Hashtable;

/**
 * <p>Diese Klasse ist abstrakte Basisklasse f&uuml;r Klassen, welche
 * Objekte im Sinne von &quot;Objektpooling&quot; verwalten, wie z.B.
 * <code>pacman3d.util.CellHandler</code>, <code>pacman3d.util.ItemHandler</code>
 * oder <code>pacman3d.util.ResourceHandler</code>.
 * F&uuml;r den Umgang mit den zu verwaltenden Objekten
 * werden Methoden zum Hinzuf&uuml;gen, Serialisieren und Deserialisieren des
 * Objektpooles zur Verf&uuml;gung gestellt.</p>
 *
 * <p>Diese Klasse erf&uuml;llt unter Anderem die folgenden Aufgaben:</p>
 * <ul>
 *   <li>Serialisierung und Deserialisierung der ihr anvertrauten Objekte
 *   in bzw. aus einem XML-Datenstrom.</li>
 *   <li>Reduzierung des Hauptspeicherbedarfs durch Vermeidung von Duplikaten
 *   im Speicher, unabh&auml;ngig davon, wie viele Klassen ein
 *   Objekt verwenden.</li>
 *   <li>Reduzierung der Exportgr&ouml;&szlig; bei XML-Serialisierung durch
 *   Vermeidung von Duplikaten, unabh&auml;ngig davon, wie viele Klassen ein
 *   Objekt verwenden.</li>
 *   <li>...</li>
 * </ul>
 *
 * <p>Diese Klasse ist Bestandteil der Anwendung
 * <a href="http://www.stormzone.de/uni/pacman3d/" target="_blank">
 * Pacman&nbsp;3D</a>, welche im Wintersemester 2001/02 (Oktober 2001
 * bis M&auml;rz 2002) im Rahmen des
 * <a href="http://www.agc.fhg.de/uniGoethe/lehre/ws0102/praktikum.php" target="_blank">
 * Praktikum Computergrafik mit VRML und Java 3D</a> der
 * <a href="http://www.agc.fhg.de" target="_blank">Professur f&uuml;r Grafische
 * Datenverarbeitung</a> am
 * <a href="http://www.informatik.uni-frankfurt.de" target="_blank">Fachbereich
 * Informatik und Biologie</a> der <a href="http://www.uni-frankfurt.de" target="_blank">
 * Johann Wolfgang Goethe-Universit&auml;t Frankfurt</a> entwickelt worden ist.
 * Die Anwendung inklusive Quellcode, Screenshots, Autorinformationen und
 * umfangreichen Informationsmaterial kann
 * unter <a href="http://www.stormzone.de/uni/pacman3d/" target="_blank">
 * www.stormzone.de/uni/pacman3d/</a> bezogen werden.</p>
 *
 * @author              Fabian Wleklinski
 *                      (<a href="mailto:Fabian@Wleklinski.de">Fabian@Wleklinski.de</a>)
 *                      (Labyrinth-Gruppe)<br>
 * @version             $Date: 2002/04/02 12:57:37 $<br>
 */
public abstract class TypeHandler implements XMLSerializeable {

	/**
	 * Abbildung von Objektnamen auf die jeweiligen Objekte. Diese Abbildung
	 * wird z.B. verwendet, um anhand eines gegebenen Namens ein konkretes
	 * Objekt zur&uuml;ckgeben zu koennen. Die verwalteten Objekte sind vom
	 * Typ <code>TypeHandlerItem</code>, bzw. eines Nachfahren davon.
	 *
	 * @see pacman3d.util.TypeHandlerItem
	 */
	private Hashtable m_oNameInstanceMapping;

	/**
	 * Abbildung von Objekten auf ihre Namen. Diese Abbildung wird z.B. verwendet,
	 * um anhand eines gegebenen Objektes den Namen desselben
	 * zur&uuml;ckgeben zu koennen. Die verwalteten Objekte sind vom Typ
	 * <code>TypeHandlerItem</code>, bzw. eines Nachfahren davon.
	 *
	 * @see pacman3d.util.TypeHandlerItem
	 */
	private Hashtable m_oInstanceNameMapping;

	/**
	 * Definiert ob die Objekte, welche verwaltet werden, prinzipiell von
	 * mehr als einer Klasse zugleich verwendet werden - oder nicht. Falls
	 * Letzteres der Fall ist, werden Objekte geklont (<code>Object.clone()</code>),
	 * bevor sie zur&uuml;ckgegeben werden.
	 *
	 * @see Object.clone()
	 */
	private boolean m_bTypeIsShareable = true;

	/**
	 * Bei der XML Serialisierung/Deserialisierung
	 * der Name des XML Elementes in dem XML Datenstrom, unterhalb dessen
	 * jeweils eine Typinformation kodiert ist, z.B. &quot;ressource&quot;
	 * (<code>&lt;ressource/&gt;</code>). Elemente dieses Namens werden
	 * unterhalb eines Elementes mit Namen <code>m_sElementsName</code>
	 * erwartet.
	 *
	 * @see m_sElementsName
	 */
	private String m_sElementName;

	/**
	 * Bei der XML Serialisierung/Deserialisierung
	 * der Name des XML Elementes in dem XML Datenstrom, unterhalb dessen
	 * die Gesamtheit aller Typinformation kodiert sind, z.B. &quot;ressources&quot;
	 * (<code>&lt;ressources/&gt;</code>). Unterhalb dieses Elementes
	 * werden Elemente mit Namen <code>m_sElementName</code> erwartet.
	 *
	 * @see m_sElementName
	 */
	private String m_sElementsName;

	/**
	 * Konstruktor des <code>TypeHandler</code>. Diese Klasse kann nicht
	 * direkt instanziert werden, da sie abstrakt ist, und ihr wesentliche
	 * Funktionalit&auml;t fehlt, welche erweiternde Klassen je nach
	 * Anwendung implementieren m&uuml;ssen.
	 *
	 * @param bTypeIsShareable      Definiert ob die Objekte, welche
	 *                              verwaltet werden, prinzipiell von
	 *                              mehr als einer Klasse zugleich verwendet
	 *                              werden - oder nicht. Falls Letzteres der
	 *                              Fall ist, werden Objekte geklont
	 *                              (<code>Object.clone()</code>),
	 *                              bevor sie zur&uuml;ckgegeben werden.
	 * @param sElementsName         Bei der XML Serialisierung/Deserialisierung
	 *                              der Name des XML Elementes in dem
	 *                              XML Datenstrom, unterhalb dessen die
	 *                              Gesamtheit aller Typinformation kodiert
	 *                              sind, z.B. &quot;ressources&quot;
	 *                              (<code>&lt;ressources/&gt;</code>).
	 *                              Unterhalb dieses Elementes
	 *                              werden Elemente mit Namen
	 *                              <code>sElementName</code> erwartet.
	 * @param sElementName          Bei der XML Serialisierung/Deserialisierung
	 *                              der Name des XML Elementes in dem
	 *                              XML Datenstrom, unterhalb dessen
	 *                              jeweils eine Typinformation kodiert ist,
	 *                              z.B. &quot;ressource&quot;
	 *                              (<code>&lt;ressource/&gt;</code>).
	 *                              Elemente dieses Namens werden unterhalb
	 *                              eines Elementes mit Namen
	 *                              <code>m_sElementsName</code> erwartet.
	 * @see java.lang.Object#clone()
	 */
	protected TypeHandler( boolean bTypeIsShareable, String sElementsName,
		String sElementName ) {

		m_bTypeIsShareable      = bTypeIsShareable;
		m_sElementsName         = sElementsName;
		m_sElementName          = sElementName;
		m_oNameInstanceMapping  = new Hashtable();
		m_oInstanceNameMapping  = new Hashtable();
	}


	/**
	 * Re-initialisiert den Objektpool, d.h. entfernt alle verwalteten
	 * Objekte.
	 */
	protected void reset() {
		m_oInstanceNameMapping.clear();
		m_oNameInstanceMapping.clear();
	}

	/**
	 * Legt das gegebene Objekt (<code>oObject</code>) unter einem neuen,
	 * eindeutigen Namen im Objektpool ab.
	 *
	 * @param oObject       Das abzulegende Objekt.
	 * @return              Der Name, unter dem das Objekt abgelegt wurde.
	 */
	protected String addInstance( Object oObject ) {
		if (oObject == null) {
			Debug.out( getClass().getName(),
				"cannot add null type!",
				Debug.LEVEL_WARNING );
			return "";
		}

		// Testen, ob dieser Typ bereits existiert, und dabei ggf.
		// den Namen ermitteln, unter dem er verfuegbar ist
		String sTypeName = getTypeName( oObject );

		if (sTypeName == null) {

			// eine neue Instanz von TypeHandlerItem erzeugen
			TypeHandlerItem oTypeHandlerItem = new
				TypeHandlerItem( oObject );
			sTypeName = oTypeHandlerItem.getName();

			// das neue Objekt ablegen
			m_oNameInstanceMapping.put( sTypeName,
				oTypeHandlerItem );
			m_oInstanceNameMapping.put( oObject,
				oTypeHandlerItem.getName() );
		}
		return sTypeName;
	}

	/**
	 * Legt das gegebene Objekt (<code>oObject</code>) unter dem gegebenen
	 * Namen (<code>sName</code>) im Objektpool ab.
	 *
	 * @param oObject       Das abzulegende Objekt.
	 * @param sName         Der Name, unter dem das Objekt abgelegt werden
	 *                      soll.
	 */
	protected void addInstance( Object oObject, String sName ) {
		TypeHandlerItem oTypeHandlerItem = new
			TypeHandlerItem( oObject, sName );

		addType( oTypeHandlerItem );
	}

	/**
	 * Legt das gegebene, in einer Instanz von <code>TypeHandlerItem</code>
	 * gekapselte Objekt (<code>oTypeHandlerItem</code>), unter einem neuen,
	 * eindeutigen Namen im Objektpool ab. Der erzeugte Name f&uuml;r das
	 * abgelegte Objekt kann <code>oTypeHandlerItem</code> entnommen
	 * werden.
	 *
	 * @param oTypeHandlerItem      Das abzulegende und gekapselte Objekt.
	 * @see TypeHandlerItem#getName()
	 */
	protected void addType( TypeHandlerItem oTypeHandlerItem ) {
		// testen ob der Typ bereits im Objektpool enthalten ist!

		// wenn der Typ noch keinen Namen besitzt, einen Namen
		// besorgen!
		String sName = oTypeHandlerItem.getName();
		if (sName == null) {
			sName = TypeHandlerItem.createTypeName( oTypeHandlerItem );
		} else {
			if (sName.equals("")) {
				sName = TypeHandlerItem.createTypeName( oTypeHandlerItem );
			}
		}
		m_oNameInstanceMapping.put( sName, oTypeHandlerItem );
		m_oInstanceNameMapping.put( oTypeHandlerItem.getInstance(), sName );
	}

	/**
	 * Deserialisiert alle Daten des <code>TypeHandler</code> aus der
	 * gegebenen XML-Struktur (<code>oElement</code>). Der
	 * <code>TypeHandler</code> wendet dazu die Methode <code>loadTypeFromDOM()</code>
	 * auf jedes serialisierte Objekt an.
	 * Die serialisierten Daten m&uuml;ssen zuvor mittels
	 * <code>saveToDOM()</code> serialisiert worden sein.
	 *
	 * @param oElement  Das XML Element, welches die zu
	 *                  deserialisierenden Daten enth&auml;lt.
	 * @exception   XMLMissingAttributeException Wird geworfen wenn ein
	 *              benoetigtes XML-Attribut
	 *              nicht vorhanden ist, oder keinen Wert enthaelt
	 *              (<code>&lt;author name=""/&gt;</code>).
	 * @exception   ClassNotFoundException Wird geworfen wenn der Versuch
	 *              eine Java-Klasse zu instanzieren fehlschlaegt. Die
	 *              Instanzierung von Klassen ist ntig, wenn in den
	 *              XML-Daten Referenzen auf Java-Klassen vorhanden sind
	 *              (<code>&lt;celltype class="pacman3d.labyrinth.cells.CellFloor"&gt;</code>).
	 * @see loadTypeFromDOM(org.w3c.dom.Element)
	 * @see saveToDOM(org.w3c.dom.Document)
	 */
	public void loadFromDOM( org.w3c.dom.Element oElement ) throws
		java.lang.ClassNotFoundException, XMLMissingAttributeException {

		/* get a list of all elements */
		org.w3c.dom.NodeList oTypeElements =
			oElement.getElementsByTagName( m_sElementName );

		int iNumberOfTypes = oTypeElements.getLength();

		/* iterate trough all elements */
		for (int iType=0; iType<iNumberOfTypes; iType++) {
			org.w3c.dom.Node oCurrentNode =
				oTypeElements.item( iType );

			if (oCurrentNode.getNodeType() == org.w3c.dom.Node.ELEMENT_NODE) {
				org.w3c.dom.Element oCurrentType =
					(org.w3c.dom.Element) oCurrentNode;

				loadTypeFromDOM( oCurrentType );


			} else {
				Debug.out( getClass().getName(),
					"not an XML element!", Debug.LEVEL_CRITICAL );
			}
		}
	}

	/**
	 * Deserialisiert alle Daten eines einzelnen Objektes aus der
	 * gegebenen XML-Struktur (<code>oElement</code>). Der
	 * <code>TypeHandler</code> wendet diese Methode auf serialisierte
	 * Objekte an, wenn er deserialisiert wird (<code>loadFromDOM()</code>).
	 * Die serialisierten Daten m&uuml;ssen zuvor mittels
	 * <code>saveToDOM()</code> serialisiert worden sein. Diese Methode ist
	 * abstrakt und muss daher von abgeleiteten Klassen je nach Anwendung
	 * &uuml;berschrieben werden.
	 *
	 * @param oElement  Das XML Element, welches die zu
	 *                  deserialisierenden Daten enth&auml;lt.
	 * @exception   XMLMissingAttributeException Wird geworfen wenn ein
	 *              benoetigtes XML-Attribut
	 *              nicht vorhanden ist, oder keinen Wert enthaelt
	 *              (<code>&lt;author name=""/&gt;</code>).
	 * @exception   ClassNotFoundException Wird geworfen wenn der Versuch
	 *              eine Java-Klasse zu instanzieren fehlschlaegt. Die
	 *              Instanzierung von Klassen ist ntig, wenn in den
	 *              XML-Daten Referenzen auf Java-Klassen vorhanden sind
	 *              (<code>&lt;celltype class="pacman3d.labyrinth.cells.CellFloor"&gt;</code>).
	 * @see saveToDOM(org.w3c.dom.Document)
	 */
	protected abstract void loadTypeFromDOM( org.w3c.dom.Element oElement )
		throws XMLMissingAttributeException, ClassNotFoundException;

	/**
	 * Sucht im Objektpool nach dem &uuml;bergebenen Objekt und gibt ggf.
	 * dessen Namen zur&uuml;ck, sofern es gefunden wurde, bzw. <code>null</code>
	 * anderenfalls.
	 *
	 * @param oObject       Das Objekt, dessen Name ermittelt werden soll.
	 * @return              Der Name des Objektes.
	 */
	protected String getTypeName( Object oObject ) {
		String sResult = null;
		if (m_bTypeIsShareable) {
			sResult = (String) m_oInstanceNameMapping.get( oObject );
		} else {
			java.util.Iterator oInstances =
				m_oNameInstanceMapping.entrySet().iterator();
			while (oInstances.hasNext()) {
				java.util.Map.Entry oResourceEntry =
					(java.util.Map.Entry) oInstances.next();

				TypeHandlerItem oTypeHandlerItem =
					(TypeHandlerItem) oResourceEntry.getValue();

				if (oTypeHandlerItem.getInstance().equals( oObject )) {
					sResult = (String) oResourceEntry.getKey();
					break;
				}
			}
		}
		return sResult;
	}

	/**
	 * Serialisiert alle Daten des <code>TypeHandler</code> in eine XML
	 * Struktur, die zur&uuml;ckgegeben wird. XML-Elemente und
	 * -Attribute werden mittels des gegebenen Dokumentes (<code>oDocument</code>)
	 * erzeugt. Der <code>TypeHandler</code> wendet zur Serialisierung die
	 * Methode <code>saveTypeElementToDOM()</code> auf jedes Objekt an.
	 * Die serialisierten Daten k&ouml;nnen sp&auml;ter wieder mittels
	 * <code>loadFromDOM()</code> deserialisiert werden.
	 *
	 * @param oDocument     Das XML-Dokument, welches zur Erstellung der
	 *                      XML-Elemente und -Attribute verwenden werden soll.
	 * @see saveTypeElementToDOM(TypeHandlerItem,org.w3c.dom.Document)
	 * @see loadFromDOM(org.w3c.dom.Element)
	 */
	public org.w3c.dom.Element saveToDOM( org.w3c.dom.Document oDocument ) {
		org.w3c.dom.Element oTypesElement =
			oDocument.createElement( m_sElementsName );

		// durch alle bekannten Ressourcen iterieren
		java.util.Iterator oInstances = m_oNameInstanceMapping.entrySet().iterator();

		while (oInstances.hasNext()) {
			java.util.Map.Entry oResourceEntry =
				(java.util.Map.Entry) oInstances.next();

			TypeHandlerItem oTypeHandlerItem =
				(TypeHandlerItem) oResourceEntry.getValue();

			org.w3c.dom.Element oResourceElement =
				saveTypeElementToDOM( oTypeHandlerItem, oDocument );

			if (oResourceElement != null) {
				oTypesElement.appendChild( oResourceElement );
			}
		}
		return oTypesElement;
	}

	/**
	 * Serialisiert ein einzelnes Objekt in eine XML-Struktur und gibt
	 * diese zur&uuml;ck. XML-Elemente und -Attribute werden mittels des
	 * gegebenen Dokumentes (<code>oDocument</code>) erzeugt. Der
	 * <code>TypeHandler</code> wendet zur Serialisierung die
	 * Methode <code>saveTypeElementToDOM()</code> auf jedes Objekt an.
	 * Die serialisierten Daten k&ouml;nnen sp&auml;ter wieder mittels
	 * <code>loadFromDOM()</code> deserialisiert werden. Diese Methode ist
	 * abstrakt und muss daher von abgeleiteten Klassen je nach Anwendung
	 * &uuml;berschrieben werden.
	 *
	 * @param oDocument             Das XML-Dokument, welches zur Erstellung
	 *                              der XML-Elemente und -Attribute verwenden
	 *                              werden soll.
	 * @param oTypeHandlerItem      Das in einer Instanz von <code>TypeHandlerItem</code>
	 *                              verpackte Objekt, welches serialsiert
	 *                              werden soll.
	 * @see loadFromDOM(org.w3c.dom.Element)
	 * @see saveToDOM(org.w3c.dom.Document)
	 */
	protected abstract org.w3c.dom.Element saveTypeElementToDOM(
		TypeHandlerItem oTypeHandlerItem, org.w3c.dom.Document oDocument );

	/**
	 * Erzeugt ein Objekt der angegebenen Klasse (<code>oClass</code>) und
	 * gibt diese zur&uuml;ck. Es k&ouml;nnen nur Klassen instanziert werden,
	 * welche &uuml;ber einen parameterlosen Konstruktor verf&uuml;gen.
	 *
	 * @param oClass        Die Klasse von der ein Objekt erzeugt
	 *                      werden soll.
	 * @return              Ein neues Objekt der Klasse <code>oClass</code>.
	 */
	protected Object createInstanceOf( Class oClass ) {
		Object oObject = null;

		try {
			Debug.out( getClass().getName(),
				"Creating instance of class '" +
				oClass.getName() + "'" );

			if (oClass != null) {
				oObject = oClass.newInstance();
			}
		} catch ( Exception oException ) {
			Debug.out( getClass().getName(),
				"Error while creating instance" +
				" of class '" + oClass.getName() +
				"'", Debug.LEVEL_CRITICAL );
		}

		return oObject;
	}

	/**
	 * Gibt das Objekt mit dem Namen <code>sName</code> zur&uuml;ck. Je
	 * nach Anwendung wird das Objekt an sich, oder eine Kopie davon
	 * zur&uuml;ckgegeben. Wird ein Objekt mit Namen <code>sName</code>
	 * nicht gefunden, wird <code>null</code> zur&uuml;ckgegeben.
	 *
	 * @param sName         Der Name des Objekt, welches zur&uuml;ckgegeben
	 *                      werden soll.
	 * @result              Das Objekt mit Namen <code>sName</code>.
	 */
	protected Object getInstanceOf( String sName ) {
		TypeHandlerItem oTypeHandlerItem = (TypeHandlerItem)
			m_oNameInstanceMapping.get( sName );
		if (oTypeHandlerItem == null) {
			return null;
		} else {
			return ((TypeHandlerItem) oTypeHandlerItem.clone()).getInstance();
		}
	}

	/**
	 * Gibt das Objekt mit dem Namen <code>sName</code> zur&uuml;ck,
	 * und sichert zu, dass es sich dabei um eine Instanz der Klasse
	 * <code>oClass</code>, oder eines ihrer Nachfahren handelt. Je
	 * nach Anwendung wird das Objekt an sich, oder eine Kopie davon
	 * zur&uuml;ckgegeben. Wird ein Objekt mit Namen <code>sName</code>
	 * nicht gefunden, oder ist es nicht eine Instanz der Klasse
	 * <code>oClass</code> oder eines ihrer Nachfahren, so wird
	 * <code>null</code> zur&uuml;ckgegeben.
	 *
	 * @param sName         Der Name des Objekt, welches zur&uuml;ckgegeben
	 *                      werden soll.
	 * @param oClass        Die Java-Klasse, von dem die zur&uuml;ckgegebene
	 *                      Instanz (abgeleitet) sein soll.
	 * @result              Das Objekt mit Namen <code>sName</code> von der
	 *                      Klasse <code>oClass</code> oder eines ihrer
	 *                      Nachfahren.
	 */
	protected Object getInstanceOf( String sName, java.lang.Class oClass ) {
		Object oResult = getInstanceOf( sName );

		if (oClass.isInstance( oResult )) {
			return oResult;
		} else {
			return null;
		}
	}
}