package pacman3d.util;

import pacman3d.util.XMLSerializeable;
import java.util.Hashtable;

/**
 * <p>Diese Klasse verwaltet Ressourcen verschiedener Arten, wie zum Beispiel
 * Texturen in Form von Grafikdateien (<code>.gif</code>, <code>.jpg</code>, ...),
 * Audiodateien, Textdateien, etc. F&uuml;r den Umgang mit diesen Ressourcen
 * werden Methoden zum Hinzuf&uuml;gen, Serialisieren und Deserialisieren einer
 * Ressourcensammlung zur Verf&uuml;gung gestellt.
 * Die zu verwaltenden Ressourcen k&ouml;nnen
 * im Dateisystem oder auf einem entfernten Server liegen, m&uuml;ssen allerdings
 * mittels einer URI adressierbar sein.</p>
 *
 * <p>In der Architektur von Pacman&nbsp;3D besteht ein Level aus einer
 * dreidimensionalen Struktur von Spielfeldern, die &quot;Zellen&quot; genannt
 * werden, und jeweils Instanzen von <code>pacman3d.labyrinth.cells.Cell</code>
 * (bzw. eines Nachfahren) sind. Alle Zellen werden von der Klasse
 * <code>pacman3d.util.CellHandler</code> verwaltet. In jeder Zelle k&ouml;nnen
 * beliebig viele Objekte wie z.B. Punkte-Pillen oder Erste-Hilfe-Koffer
 * existieren, die &quot;Items&quot; genannt werden, und jeweils Instanzen von
 * <code>pacman3d.labyrinth.items.Item</code> (bzw. eines Nachfahren) sind.
 * Alle Items werden von der Klasse <code>pacman3d.util.ItemHandler</code>
 * verwaltet. Zellen und Items, sowie andere Objekte des Spiels (Pacmans,
 * Monster, ...) verwenden Ressourcen in Form von Grafik- oder Audiodateien,
 * oder anderen Medientypen. Alle diese Ressourcen werden von der Klasse
 * <code>pacman3d.util.ResourceHandler</code> verwaltet.
 *
 * <p>Diese Klasse erf&uuml;llt unter Anderem die folgenden Aufgaben:</p>
 * <ul>
 *   <li>Serialisierung und Deserialisierung der ihr anvertrauten Ressourcen
 *   in bzw. aus einem XML-Datenstrom.</li>
 *   <li>Reduzierung des Hauptspeicherbedarfs durch Vermeidung von Duplikaten
 *   im Speicher, unabh&auml;ngig davon, wie viele Klassen eine
 *   Ressource verwenden.</li>
 *   <li>Reduzierung der Exportgr&ouml;&szlig; bei XML-Serialisierung durch
 *   Vermeidung von Duplikaten, unabh&auml;ngig davon, wie viele Klassen eine
 *   Ressource verwenden.</li>
 *   <li>...</li>
 * </ul>
 *
 * <p>Es kann zu jeder Zeit nur eine einzige Instanz von
 * <code>ResourceHandler</code> geben. Es existiert die statische Factory-
 * Methode <code>getInstance</code>, um eine Referenz auf die Instanz zu
 * bekommen.</p>
 *
 * <p>Diese Klasse ist Bestandteil der Anwendung
 * <a href="http://www.stormzone.de/uni/pacman3d/" target="_blank">
 * Pacman&nbsp;3D</a>, welche im Wintersemester 2001/02 (Oktober 2001
 * bis M&auml;rz 2002) im Rahmen des
 * <a href="http://www.agc.fhg.de/uniGoethe/lehre/ws0102/praktikum.php" target="_blank">
 * Praktikum Computergrafik mit VRML und Java 3D</a> der
 * <a href="http://www.agc.fhg.de" target="_blank">Professur f&uuml;r Grafische
 * Datenverarbeitung</a> am
 * <a href="http://www.informatik.uni-frankfurt.de" target="_blank">Fachbereich
 * Informatik und Biologie</a> der <a href="http://www.uni-frankfurt.de" target="_blank">
 * Johann Wolfgang Goethe-Universit&auml;t Frankfurt</a> entwickelt worden ist.
 * Die Anwendung inklusive Quellcode, Screenshots, Autorinformationen und
 * umfangreichen Informationsmaterial kann
 * unter <a href="http://www.stormzone.de/uni/pacman3d/" target="_blank">
 * www.stormzone.de/uni/pacman3d/</a> bezogen werden.</p>
 *
 * @see                 pacman3d.labyrinth.cells.Cell
 * @see                 pacman3d.labyrinth.cells.Item
 * @see                 pacman3d.util.CellHandler
 * @see                 pacman3d.util.ItemHandler
 * @see                 pacman3d.util.ResourceHandler
 * @author              Fabian Wleklinski
 *                      (<a href="mailto:Fabian@Wleklinski.de">Fabian@Wleklinski.de</a>)
 *                      (Labyrinth-Gruppe)<br>
 * @version             $Date: 2002/04/02 12:57:37 $<br>
 */
public class ResourceHandler extends TypeHandler implements XMLSerializeable {

	/**
	 * Die einzige Instanz des <code>ResourceHandler</code>, die es geben
	 * darf (Factory-Konzept).
	 * @see getInstance()
	 */
	private static ResourceHandler m_oInstance = null;

	/**
	 * Konstruktor - sollte nicht direkt aufgerufen werden. Statt dessen
	 * <code>getInstance()</code> benutzen (Factory-Konzept).
	 *
	 * @see getInstance()
	 */
	private ResourceHandler() {
		super( true, "resources", "resource" );
	}

	/**
	 * <p>Factory des <code>ResourceHandler</code>. Diese Methode ist zu
	 * verwenden um an eine Instanz des <code>ResourceHandler</code> zu
	 * gelangen. Gem&auml;&szlig; des Factory-Konzeptes kann zu jedem
	 * Zeitpunkt der Programmausf&uuml;hrung nur eine einzige Instanz des
	 * <code>ResourceHandler</code> existieren.</p>
	 * <p>Beispiel:</p>
	 * <pre>
	 * ResourceHandler oResourceHandler = ResourceHandler.getInstance();
	 * oResourceHandler.addTexture( ... );
	 * </pre>
	 *
	 * @return Eine Instanz des <code>ResourceHandler</code>.
	 */
	public static ResourceHandler getInstance() {
		if (m_oInstance == null) {
			m_oInstance = new ResourceHandler();
		}

		return m_oInstance;
	}

	/**
	 * F&uuml;gt eine Textur hinzu, und erzeugt daf&uuml;r selbstst&auml;ndig
	 * einen neuen, eindeutigen Bezeichner.
	 * @param oTexture      Die hinzuzuf&uuml;gende Textur.
	 * @return              Der Bezeichner, unter dem die Textur abgelegt
	 *                      wurde.
	 */
	public String addTexture( javax.media.j3d.Texture oTexture ) {
		return addInstance( oTexture );
	}

	/**
	 * Fuegt eine neue Textur hinzu, und legt diese unter dem gegebenen
	 * Namen (<code>sName</code>) ab.
	 * @param oTexture      Die hinzuzuf&uuml;gende Textur.
	 * @param sName         Der Name, unter dem die Textur abgelegt werden
	 *                      soll.
	 */
	public void addTexture( javax.media.j3d.Texture2D oTexture, String sName ) {
		ResourceHandlerItem oResourceHandlerItem = new
			ResourceHandlerItem( oTexture, sName );
		addType( oResourceHandlerItem );
	}

	/**
	 * Fuegt eine neue Textur einschlie&szlig;lich ihrer bin&auml;ren
	 * Repr&auml;sentation hinzu, und legt diese unter dem gegebenen
	 * Namen (<code>sName</code>) ab. Der Inhalt der Parameter <code>abData</code>,
	 * <code>sMimeType</code> und <code>sFileName</code> wird f&uuml;r die
	 * Serialisierung des <code>ResourceHandler</code>
	 * (<code>saveToDOM()</code>)ben&ouml;tigt.
	 *
	 * @param oTexture      Die hinzuzuf&uuml;gende Textur.
	 * @param sName         Der Name, unter dem die Textur abgelegt werden
	 *                      soll.
	 * @param abData        Die bin&auml;re Repr&auml;sentation der Textur,
	 *                      d.h. der Inhalt der zur Textur geh&ouml;renden
	 *                      Grafikdatei.
	 * @param sMimeType     Der Mimetyp der Textur, z.B. &quot;image/gif&quot;.
	 * @param sFileName     Der originale Dateiname der zur Textur
	 *                      geh&ouml;rigen Grafikdatei, z.B.
	 *                      <code>texture_wall_1.gif</code>.
	 * @see saveToDOM(org.w3c.dom.Document)
	 */
	public void addTexture( javax.media.j3d.Texture2D oTexture,
		String sName, byte[] abData, String sMimeType, String sFileName ) {

		ResourceHandlerItem oResourceHandlerItem = new
			ResourceHandlerItem( oTexture, sName, abData,
			sMimeType, sFileName );
		addType( oResourceHandlerItem );
	}

	/**
	 * Fuegt eine neue Textur einschlie&szlig;lich der URL ihrer
	 * bin&auml;ren Repr&auml;sentation hinzu, und legt diese unter dem
	 * gegebenen Namen (<code>sName</code>) ab. Der Inhalt der Parameter <code>oURL</code>,
	 * <code>sMimeType</code> und <code>iFileSize</code> wird f&uuml;r die
	 * Serialisierung des <code>ResourceHandler</code>
	 * (<code>saveToDOM()</code>)ben&ouml;tigt.
	 *
	 * @param oTexture      Die hinzuzuf&uuml;gende Textur.
	 * @param sName         Der Name, unter dem die Textur abgelegt werden
	 *                      soll.
	 * @param oURL          Die URL der bin&auml;ren Repr&auml;sentation der
	 *                      Textur, d.h. die URL der zur Textur geh&ouml;renden
	 *                      Grafikdatei.
	 * @param sMimeType     Der Mimetyp der Textur, z.B. &quot;image/gif&quot;.
	 * @param iFileSize     Die Dateigr&ouml;&szlig;e der zur Textur
	 *                      geh&ouml;rigen Grafikdatei in Byte.
	 * @see saveToDOM(org.w3c.dom.Document)
	 */
	public void addTexture( javax.media.j3d.Texture2D oTexture,
		String sName, java.net.URL oURL, String sMimeType, int iFileSize ) {

		ResourceHandlerItem oResourceHandlerItem = new
			ResourceHandlerItem( oTexture, sName, oURL,
			iFileSize, sMimeType );
		addType( oResourceHandlerItem );
	}

	/**
	 * Fuegt eine neue Textur einschlie&szlig;lich der URL ihrer
	 * bin&auml;ren Repr&auml;sentation hinzu, und erzeugt daf&uuml;r
	 * selbstst&auml;ndig einen neuen,
	 * eindeutigen Bezeichner. Der Inhalt der Parameter <code>oURL</code>,
	 * <code>sMimeType</code> und <code>iFileSize</code> wird f&uuml;r die
	 * Serialisierung des <code>ResourceHandler</code>
	 * (<code>saveToDOM()</code>)ben&ouml;tigt.
	 *
	 * @param oTexture      Die hinzuzuf&uuml;gende Textur.
	 * @param oURL          Die URL der bin&auml;ren Repr&auml;sentation der
	 *                      Textur, d.h. die URL der zur Textur geh&ouml;renden
	 *                      Grafikdatei.
	 * @param sMimeType     Der Mimetyp der Textur, z.B. &quot;image/gif&quot;.
	 * @param iFileSize     Die Dateigr&ouml;&szlig;e der zur Textur
	 *                      geh&ouml;rigen Grafikdatei in Byte.
	 * @return              Der Bezeichner, unter dem die Textur abgelegt
	 *                      wurde.
	 * @see saveToDOM(org.w3c.dom.Document)
	 */
	public String addTexture( javax.media.j3d.Texture2D oTexture,
		java.net.URL oURL, String sMimeType, int iFileSize ) {

		ResourceHandlerItem oResourceHandlerItem = new
			ResourceHandlerItem( oTexture, oURL,
			iFileSize, sMimeType );
		addType( oResourceHandlerItem );
		return oResourceHandlerItem.getName();
	}

	/**
	 * Kodiert die &uuml;bergebenen, bin&auml;ren Daten mittels
	 * base64-Kodierung in alphanumerische Symbole (6-Bit), d.h.
	 * bildet sie auf die Menge der 64 Symbole {a..zA..Z0..9_/} ab. Mittels
	 * dieser Kodierung k&ouml;nnen Bin&auml;rdaten wie z.B. Grafik- und
	 * Audiodateien in XML-Datenstr&ouml;e existieren. Da die
	 * 6-bittigen Symbole zu ihrer Speicherung dennoch 8 Bit ben&ouml;tigen,
	 * nimmt bei diesem Vorgang die Gr&ouml;&szlig;e der Daten um etwa ein
	 * Drittel zu.
	 *
	 * @param aiByteStream  Die zu kodierenden Bin&auml;rdaten.
	 * @return              Die kodierten Bin&auml;rdaten.
	 * @see decodeData(char[])
	 * @see encodeDataToString(byte[])
	 */
	public char[] encodeData( byte[] aiByteStream ) {
		return org.apache.xmlrpc.Base64.encode( aiByteStream );
	}

	/**
	 * Kodiert die &uuml;bergebenen, bin&auml;ren Daten mittels
	 * base64-Kodierung in alphanumerische Symbole (6-Bit), d.h.
	 * bildet sie auf die Menge der 64 Symbole {a..zA..Z0..9_/} ab. Mittels
	 * dieser Kodierung k&ouml;nnen Bin&auml;rdaten wie z.B. Grafik- und
	 * Audiodateien in XML-Datenstr&ouml;e existieren. Da die
	 * 6-bittigen Symbole zu ihrer Speicherung dennoch 8 Bit ben&ouml;tigen,
	 * nimmt bei diesem Vorgang die Gr&ouml;&szlig;e der Daten um etwa ein
	 * Drittel zu.
	 *
	 * @param aiByteStream  Die zu kodierenden Bin&auml;rdaten.
	 * @return              Die kodierten Bin&auml;rdaten.
	 * @see decodeData(char[])
	 * @see encodeData(byte[])
	 */
	public String encodeDataToString( byte[] aiByteStream ) {
		return new String( org.apache.xmlrpc.Base64.encode(
			aiByteStream ) );
	}

	/**
	 * Dekodiert die &uuml;bergebenen, base64-kodierten Daten, die z.B.
	 * mittels <code>encodeData()</code> kodiert worden sind, und gibt
	 * die (urspr&uuml;nglichen) Bin&auml;rdaten zur&uuml;ck.
	 * Bei diesem Vorgang nimmt die Gr&ouml;&szlig;e der Daten um etwa ein
	 * Viertel ab.
	 *
	 * @param acBase64Data  Die base64-kodierten Daten, die zu
	 *                      dekodieren sind.
	 * @return              Die (urspr&uuml;nglichen) Bin&auml;rdaten.
	 * @see encodeData(byte[])
	 * @see encodeDataToString(byte[])
	 */
	public byte[] decodeData( char[] acBase64Data ) {
		return org.apache.xmlrpc.Base64.decode( acBase64Data );
	}

	/**
	 * Gibt die Textur mit dem angegebenen Namen (<code>sName</code>) zur&uuml;ck. Diese Textur
	 * muss zuvor mittels <code>addTexture()</code> hinzugef&uuml;gt worden
	 * sein.
	 *
	 * @param sName         Der Name der zur&uuml;ckzugebenden Textur.
	 * @return              Die Textur mit dem angegebenen Namen.
	 * @see addTexture(javax.media.j3d.Texture)
	 * @see addTexture(javax.media.j3d.Texture2D,String)
	 * @see addTexture(javax.media.j3d.Texture2D,String,byte[],String,String)
	 * @see addTexture(javax.media.j3d.Texture2D,String,java.net.URL,String,int)
	 * @see addTexture(javax.media.j3d.Texture2D,java.net.URL,String,int)
	 */
	public javax.media.j3d.Texture2D getTexture( String sName ) {
		Object oRessource = getInstanceOf( sName );
		return (javax.media.j3d.Texture2D) oRessource;
	}

	/**
	 * Serialisiert alle Daten einer einzelnen Ressource in eine XML-Struktur,
	 * und gibt diese zur&uuml;ck. Der <code>ResourceHandler</code>
	 * wendet diese Methode auf alle Ressourcen an, wenn er serialisiert
	 * wird (<code>saveToDOM()</code>). Die serialisierten Daten k&ouml;nnen
	 * mittels <code>loadTypeFromDOM()</code> wieder deserialisiert werden.
	 *
	 * @param oTypeHandlerItem      Die in einer Instanz von
	 *                              <code>TypeHandlerItem</code> gekapselte
	 *                              Ressource, welche serialisiert werden
	 *                              soll.
	 * @param oDocument     Ein XML-Dokument zwecks Erstellung von Knoten.
	 * @see saveToDOM(org.w3c.dom.Document)
	 * @see loadTypeFromDOM(org.w3c.dom.Element)
	 */
	protected org.w3c.dom.Element saveTypeElementToDOM(
		TypeHandlerItem oTypeHandlerItem, org.w3c.dom.Document oDocument ) {

		/* testen, ob es sich um eine Instanz von ResourceHandlerItem
		handelt - anderenfalls koennen wir es nicht serialisieren. */
		if (! (oTypeHandlerItem instanceof ResourceHandlerItem)) {
			Debug.out( getClass().getName(), "cannot serialize " +
				oTypeHandlerItem, Debug.LEVEL_WARNING );
			return null;
		}

		ResourceHandlerItem oResourceHandlerItem =
			(ResourceHandlerItem) oTypeHandlerItem;

		org.w3c.dom.Element oTypeElement =
			oDocument.createElement( "resource" );

		// <resource> erzeugen
		oTypeElement.setAttribute( "name", oTypeHandlerItem.getName() );

		/* Fallunterscheidung, je nachdem welcher Typ von
		Repraesentation vorliegt */
		// Object oBinaryData = m_oBinaryData.get( sTypeName );

		/*
		java.io.ByteArrayOutputStream oByteArrayOutputStream =
			new java.io.ByteArrayOutputStream();
		com.sun.image.codec.jpeg.JPEGImageEncoder oJPEGImageEncoder =
			com.sun.image.codec.jpeg.JPEGCodec.createJPEGEncoder( oByteArrayOutputStream );
		javax.media.j3d.ImageComponent oImageComponent =
			oTexture.getImage(0);

		java.io.FileOutputStream oFileOutputStream = null;
		try {
			oFileOutputStream =
				new java.io.FileOutputStream( sTypeName + ".jpg" );
		} catch ( Exception e ) {
		}
		com.sun.image.codec.jpeg.JPEGImageEncoder oJPEGImageEncoder2 =
			com.sun.image.codec.jpeg.JPEGCodec.createJPEGEncoder( oFileOutputStream );
		*/

		// <content> erzeugen
		org.w3c.dom.Element oContentElement =
			oDocument.createElement( "content" );
		oTypeElement.appendChild( oContentElement );

		java.net.URL oURL = oResourceHandlerItem.getURL();
		byte[] abData = oResourceHandlerItem.getData();
		String sSize = (new Integer( oResourceHandlerItem.getSize() )).toString();
		String sMimeType = oResourceHandlerItem.getMimeType();
		String sFileName = oResourceHandlerItem.getFileName();

		// testen, ob die Resource durch einen Verweis eingebunden
		// werden kann. anderenfalls wird getestet, ob die Ressource
		// in die Datei eingebettet werden kann. Wenn auch das nicht
		// funktioniert, wird ein Fehler gemeldet.
		if (oURL != null) {
			oContentElement.setAttribute( "uri", oURL.toString() );
			oContentElement.setAttribute( "mode", "external" );
			oContentElement.setAttribute( "size", sSize );
			oContentElement.setAttribute( "mimetype", sMimeType );
		} else if (abData != null) {
			oContentElement.setAttribute( "encoding",
				"base64" );
			oContentElement.setAttribute( "mode",
				"embedded" );
			oContentElement.setAttribute( "size", sSize );
			oContentElement.setAttribute( "mimetype", sMimeType );
			oContentElement.setAttribute( "filename", sFileName );

			String sEncodedData = encodeDataToString( abData );

			org.w3c.dom.CDATASection oBinaryDataNode =
				oDocument.createCDATASection(
				sEncodedData );
			oContentElement.appendChild( oBinaryDataNode );

		} else {

			Debug.out( getClass().getName(), "cannot serialize resource: " +
				"no source available.", Debug.LEVEL_WARNING );

			return null;
		}

		// String sSize = (String) m_oBinaryFileSizes.get( sTypeName );
		// oContentElement.setAttribute( "size", sSize );

		// String sMimeType = (String) m_oBinaryMimeTypes.get( sTypeName );
		// oContentElement.setAttribute( "mimetype", sMimeType );

		/*
		if (oImageComponent instanceof javax.media.j3d.ImageComponent2D) {
			java.awt.image.BufferedImage oBufferedImage =
				((javax.media.j3d.ImageComponent2D) oImageComponent).getImage();

			try {
				oJPEGImageEncoder.encode( oBufferedImage );
				oJPEGImageEncoder2.encode( oBufferedImage );
			} catch ( java.io.IOException oException ) {
				Debug.out( this.getClass().getName(), oException, Debug.LEVEL_CRITICAL );
			}

			String sEncodedImage = encodeDataToString(
				oByteArrayOutputStream.toByteArray() );

			org.w3c.dom.CDATASection oBinaryDataNode = oDocument.createCDATASection(
				sEncodedImage );
			oContentElement.appendChild( oBinaryDataNode );

			// acEncodedImage
			oContentElement.setAttribute( "encoding", "base64" );
			oContentElement.setAttribute( "size", new Integer(oByteArrayOutputStream.size()).toString() );
			oContentElement.setAttribute( "filename", sTypeName + ".jpg" );
			oContentElement.setAttribute( "height", new Integer(oBufferedImage.getHeight()).toString() );
			oContentElement.setAttribute( "width", new Integer(oBufferedImage.getWidth()).toString() );
			oContentElement.setAttribute( "mimetype", "image/jpeg" );
			oContentElement.setAttribute( "mode", "embedded" );
			oTypeElement.appendChild( oContentElement );
		} else {
			Debug.out( this.getClass().getName(),
				"Cannot encode instance of " +
				oImageComponent.getClass().getName() + "!",
				Debug.LEVEL_CRITICAL );
		}
		*/

		return oTypeElement;
	}

	/**
	 * Deserialisiert alle Daten einer einzelnen Ressource aus der
	 * gegebenen XML-Struktur (<code>oTypeElement</code>). Der
	 * <code>ResourceHandler</code> wendet diese Methode auf serialisierte
	 * Ressourcen an, wenn er deserialisiert wird (<code>loadFromDOM()</code>).
	 * Die serialisierten Daten m&uuml;ssen zuvor mittels
	 * <code>saveTypeElementToDOM()</code> serialisiert worden sein.
	 *
	 * @param oTypeElement  Das XML Element, welches die zu
	 *                      deserialisierenden Daten enth&auml;lt.
	 * @exception   XMLMissingAttributeException Wird geworfen wenn ein
	 *              benoetigtes XML-Attribut
	 *              nicht vorhanden ist, oder keinen Wert enthaelt
	 *              (<code>&lt;author name=""/&gt;</code>).
	 * @see saveTypeElementToDOM(TypeHandlerItem,org.w3c.dom.Document)
	 * @see loadFromDOM(org.w3c.dom.Element)
	 */
	protected void loadTypeFromDOM( org.w3c.dom.Element oTypeElement )
		 throws XMLMissingAttributeException {

		/* get the name of the current type */
		String sName = oTypeElement.getAttribute( "name" );

		if (sName.equals( "" )) {
			throw new XMLMissingAttributeException( "name" );
		}

		/* there has to be exactly 1 element <content>
		below <ressource>: get it. */

		org.w3c.dom.NodeList oContentElements =
			oTypeElement.getElementsByTagName( "content" );

		if (oContentElements.getLength() == 0 ) {
			Debug.out( getClass().getName(),
				"<content>-element not found!", Debug.LEVEL_CRITICAL );
		} else if (oContentElements.getLength() > 1) {
			Debug.out( getClass().getName(),
				"only 1 <content>-element allowed!", Debug.LEVEL_CRITICAL );
		} else {
			org.w3c.dom.Element oContentElement =
				(org.w3c.dom.Element) oContentElements.item( 0 );

			loadContentElement( sName, oContentElement );
		}
	}

	/**
	 * Erstellt eine (neue) Ressource (z.B. eine Textur) aus der
	 * &uuml;bergebenen, bin&auml;ren Repr&auml;sentation (z.B. der Inhalt
	 * einer Grafik-, Audio- oder Textdatei), und legt diese Ressource
	 * unter dem gegebenen Namen (<code>sName</code>) ab.
	 * Der Inhalt der Parameter <code>sMimeType</code> und
	 * <code>sFileName</code> wird f&uuml;r die Serialisierung des
	 * <code>ResourceHandler</code> (<code>saveToDOM()</code>)ben&ouml;tigt.
	 *
	 * @param abResourceFile Die bin&auml;re Repr&auml;sentation der Ressource,
	 *                      d.h. der Inhalt der zur Ressource geh&ouml;renden
	 *                      Datei (z.B. Grafik-, Audio- oder Textdatei).
	 * @param sName         Der Name, unter dem die Ressource abgelegt werden
	 *                      soll.
	 * @param sMimeType     Der Mimetyp der Ressource, z.B. &quot;image/gif&quot;.
	 * @param sFileName     Der originale Dateiname der zur Ressource
	 *                      geh&ouml;rigen Datei, z.B.
	 *                      <code>texture_wall_1.gif</code>.
	 * @exception   UnknownResourceTypeException Wird geworfen, wenn die in
	 *              <code>abResourceFile</code> &uuml;bergebenen
	 *              Bin&auml;rdaten nicht als Ressource erkannt werden
	 *              k&ouml;nnen.
	 * @see saveToDOM(org.w3c.dom.Document)
	 */
	private String addResourceFromByteArray( byte[] abResourceFile,
		String sName, String sMimeType, String sFileName ) throws
		UnknownResourceTypeException {

		String sResult;

		java.awt.Image oImage =
			java.awt.Toolkit.getDefaultToolkit().createImage( abResourceFile );

		com.sun.j3d.utils.image.TextureLoader oTextureLoader =
			new com.sun.j3d.utils.image.TextureLoader( oImage, null );

		javax.media.j3d.Texture oTexture = oTextureLoader.getTexture();

		ResourceHandlerItem oResourceHandlerItem;
		if (sName != null) {
			oResourceHandlerItem = new ResourceHandlerItem(
				oTexture, sName, abResourceFile, sMimeType,
				sFileName );
			sResult = sName;
			addType( oResourceHandlerItem );
		} else {
			oResourceHandlerItem = new ResourceHandlerItem(
				oTexture, abResourceFile, sMimeType, sFileName );
			addType( oResourceHandlerItem );
			sResult = oResourceHandlerItem.getName();
		}

		return sResult;
	}

	/**
	 * Erstellt eine (neue) Ressource (z.B. eine Textur) aus der
	 * &uuml;bergebenen, bin&auml;ren Repr&auml;sentation (z.B. die URL
	 * einer Grafik-, Audio- oder Textdatei), und legt diese Ressource
	 * unter dem gegebenen Namen (<code>sName</code>) ab. Falls kein Name
	 * angegeben ist (d.h. <code>sName==null</code>), wird selbstst&auml;ndig
	 * einen neuer, eindeutiger Bezeichner erzeugt.  Der Inhalt der Parameter
	 * <code>sMimeType</code> und <code>iFileSize</code> wird f&uuml;r die
	 * Serialisierung des <code>ResourceHandler</code>
	 * (<code>saveToDOM()</code>) ben&ouml;tigt.
	 *
	 * @param oURL          Die URL der bin&auml;ren Repr&auml;sentation der
	 *                      Ressource, d.h. die URL der zur Ressource
	 *                      geh&ouml;renden Datei (z.B. Grafik-, Audio- oder
	 *                      Textdatei).
	 * @param sName         Der Name, unter dem die Ressource abgelegt werden
	 *                      soll.
	 * @param sMimeType     Der Mimetyp der Ressource, z.B. &quot;image/gif&quot;.
	 * @param sFileName     Der originale Dateiname der zur Ressource
	 *                      geh&ouml;rigen Datei, z.B.
	 *                      <code>texture_wall_1.gif</code>.
	 * @return              Der Bezeichner, unter dem die Ressource abgelegt
	 *                      wurde.
	 * @exception   UnknownResourceTypeException Wird geworfen, wenn die durch
	 *              <code>oURL</code> referenzierten Bin&auml;rdaten nicht
	 *              als Ressource erkannt werden k&ouml;nnen.
	 * @see saveToDOM(org.w3c.dom.Document)
	 */
	private String addResourceFromURL( java.net.URL oURL, String sName,
		String sMimeType, int iFileSize ) throws
		UnknownResourceTypeException {

		String sResult = "";

		// testen ob der Dateityp bekannt ist, und anderenfalls eine
		// Exception werfen
		if (oURL.getFile().toLowerCase().endsWith( ".jpg" ) ||
		    oURL.getFile().toLowerCase().endsWith( ".gif" )) {

			com.sun.j3d.utils.image.TextureLoader oTextureLoader =
				new com.sun.j3d.utils.image.TextureLoader(
				oURL, null );

			javax.media.j3d.Texture oTexture =
				oTextureLoader.getTexture();

			if (oTexture != null) {
				Debug.out( this.getClass().getName(),
					oURL.toString() + " loaded.",
					Debug.LEVEL_NOTICE );
			} else {
				Debug.out( this.getClass().getName(),
					"Could not load " +
					oURL.toString(), Debug.LEVEL_WARNING  );
			}

			if (oTexture != null) {
				ResourceHandlerItem oResourceHandlerItem;
				if (sName != null) {
					oResourceHandlerItem = new ResourceHandlerItem(
						oTexture, sName, oURL, iFileSize, sMimeType );
					sResult = sName;
					addType( oResourceHandlerItem );
				} else {
					oResourceHandlerItem = new ResourceHandlerItem(
						oTexture, oURL, iFileSize, sMimeType );
					addType( oResourceHandlerItem );
					sResult = oResourceHandlerItem.getName();
				}
			}
		} else {
			// unbekannter Dateityp => Exception werfen
			throw new UnknownResourceTypeException( oURL );
		}

		// wenn wir hier angelangt sind, scheint der Ressourcentyp
		// bekannt zu sein...

		return sResult;
	}

	/**
	 * Verarbeitet beim Lesen eines XML-Datenstromes ein XML-Element von
	 * der Art <code>&lt;content&gt;</code>, welches die Daten f&uuml;r
	 * eine einzelne Ressource repr&auml;sentiert.
	 *
	 * @exception   XMLMissingAttributeException Wird geworfen wenn beim
	 *              Einlesen von XML-Daten ein benoetigtes XML-Attribut
	 *              nicht vorhanden ist, oder keinen Wert enthaelt
	 *              (<code>&lt;author name=""/&gt;</code>).
	 * @see loadFromDOM(org.w3c.dom.Document)
	 * @see loadTypeFromDOM(org.w3c.dom.Element)
	 */
	private void loadContentElement( String sName, org.w3c.dom.Element oContentElement )
		throws XMLMissingAttributeException {

		/* get attribute values */
		String sEncoding = oContentElement.getAttribute( "encoding" );
		int iSize = new Integer( oContentElement.getAttribute(
			"size" ) ).intValue();
		String sFileName = oContentElement.getAttribute( "filename" );
		String sMimeType = oContentElement.getAttribute( "mimetype" );
		String sMode = oContentElement.getAttribute( "mode" );
		String sURI = oContentElement.getAttribute( "uri" );

		/* if one or more non-optional attributes are missing => throw
		an exception */

		if (iSize <= 0) {
			throw new XMLMissingAttributeException( "size" );
		}
		if (sMimeType.equals( "" )) {
			throw new XMLMissingAttributeException( "mimetype" );
		}
		if (sMode.equals( "" )) {
			throw new XMLMissingAttributeException( "mode" );
		}

		boolean bEmbeddedResource = sMode.equals( "embedded" );

		if (bEmbeddedResource) {
			if (sFileName.equals( "" )) {
				throw new XMLMissingAttributeException( "filename" );
			}
		} else {
			if (sURI.equals( "" )) {
				throw new XMLMissingAttributeException( "url" );
			}
		}

		Debug.out( this.getClass().getName(), "uri='" + sURI + "',size='" +
			iSize + "',mode='" + sMode + "',filename='" +
			sFileName + "',mimetype='" + sMimeType + "',encoding='" +
			sEncoding + "'", Debug.LEVEL_NOTICE );

		com.sun.j3d.utils.image.TextureLoader oTextureLoader = null;

		if (bEmbeddedResource) {
			char[] abContent = null;
			org.w3c.dom.NodeList oNodeList = oContentElement.getChildNodes();
			int i=0;
			int iNumberOfChilds = oNodeList.getLength();
			while ((i<iNumberOfChilds) && (abContent==null)) {
				if (oNodeList.item(i).getNodeType() == org.w3c.dom.Node.CDATA_SECTION_NODE) {
					abContent = ((org.w3c.dom.CDATASection) oNodeList.item(i)).getData().toCharArray();
				}
				i++;
			}

			if (abContent != null) {
				byte[] abDecodedContent = decodeData( abContent );
				abContent = null;

				try {
					addResourceFromByteArray(
						abDecodedContent, sName,
						sMimeType, sFileName );
				} catch( Exception oException ) {
				}
			}

		} else {

			if (sMimeType.equals("image/jpeg") |
			    sMimeType.equals("image/gif"))
			{
				java.net.URL oURL = null;
				try {
					oURL = new java.net.URL( sURI );
				} catch( java.net.MalformedURLException oException ) {
					Debug.out( this.getClass().getName(), oException, Debug.LEVEL_WARNING );
				}

				try {
					addResourceFromURL( oURL, sName,
						sMimeType, iSize );
				} catch( Exception oException ) {
					Debug.out( getClass().getName(), oException, Debug.LEVEL_WARNING );
				}
			} else {
			}

		}
	}
}