package pacman3d.util;

import java.util.Hashtable;

/**
 * <p>Diese Klasse verwaltet Items (d.h. Objekte, die sich innerhalb von
 * Spielfeldern befinden, wie z.B. Erste-Hilfe-Koffer oder Punkte-Pillen) wie
 * sie f&uuml;r Pacman&nbsp;3D ben&ouml;tigt werden. Jede Zelle entspricht einer
 * Instanz der Klasse <code>pacman3d.labyrinth.items.Item</code>. F&uuml;r den
 * Umgang mit diesen Items werden Methoden zum Hinzuf&uuml;gen, Serialisieren
 * und Deserialisieren einer Itemsammlung zur Verf&uuml;gung gestellt.</p>
 *
 * <p>In der Architektur von Pacman&nbsp;3D besteht ein Level aus einer
 * dreidimensionalen Struktur von Spielfeldern, die &quot;Zellen&quot; genannt
 * werden, und jeweils Instanzen von <code>pacman3d.labyrinth.cells.Cell</code>
 * (bzw. eines Nachfahren) sind. Alle Zellen werden von der Klasse
 * <code>pacman3d.util.CellHandler</code> verwaltet. In jeder Zelle k&ouml;nnen
 * beliebig viele Objekte wie z.B. Punkte-Pillen oder Erste-Hilfe-Koffer
 * existieren, die &quot;Items&quot; genannt werden, und jeweils Instanzen von
 * <code>pacman3d.labyrinth.items.Item</code> (bzw. eines Nachfahren) sind.
 * Alle Items werden von der Klasse <code>pacman3d.util.ItemHandler</code>
 * verwaltet. Zellen und Items, sowie andere Objekte des Spiels (Pacmans,
 * Monster, ...) verwenden Ressourcen in Form von Grafik- oder Audiodateien,
 * oder anderen Medientypen. Alle diese Ressourcen werden von der Klasse
 * <code>pacman3d.util.ResourceHandler</code> verwaltet.
 *
 * <p>Ein Labyrinth besteht in Pacman&nbsp;3D aus zahlreichen Zellen, von
 * denen im Allgemeinen viele Items &quot;identisch&quot; sind. Wenn z.B. in
 * jeder Zelle ein Item enthalten ist, dann besteht z.B. ein Spielfeld mit
 * 20x25x5 Zellen aus 2.500 Items, die sich im Allgemeinen auf nicht mehr als
 * f&uuml;nf bis zehn verschieden parametrisierte Items reduzieren lassen. Es
 * ist die Aufgabe des <code>ItemHandler</code>, diese Zellen zu verwalten, und
 * die angesprochene Redundanz zu vermeiden.</p>
 *
 * <p>Diese Klasse erf&uuml;llt unter Anderem die folgenden Aufgaben:</p>
 * <ul>
 *   <li>Serialisierung und Deserialisierung der ihr anvertrauten Items
 *   in bzw. aus einem XML-Datenstrom.</li>
 *   <li>Reduzierung des Hauptspeicherbedarfs durch Vermeidung von Duplikaten
 *   im Speicher, unabh&auml;ngig davon, wie viele Klassen ein Item
 *   verwenden.</li>
 *   <li>Reduzierung der Exportgr&ouml;&szlig; bei XML-Serialisierung durch
 *   Vermeidung von Duplikaten, unabh&auml;ngig davon, wie viele Klassen ein
 *   Item verwenden.</li>
 *   <li>...</li>
 * </ul>
 *
 * <p>Es kann zu jeder Zeit nur eine einzige Instanz von
 * <code>ItemHandler</code> geben. Es existiert die statische Factory-
 * Methode <code>getInstance</code>, um eine Referenz auf die Instanz zu
 * bekommen.</p>
 *
 * <p>Diese Klasse ist Bestandteil der Anwendung
 * <a href="http://www.stormzone.de/uni/pacman3d/" target="_blank">
 * Pacman&nbsp;3D</a>, welche im Wintersemester 2001/02 (Oktober 2001
 * bis M&auml;rz 2002) im Rahmen des
 * <a href="http://www.agc.fhg.de/uniGoethe/lehre/ws0102/praktikum.php" target="_blank">
 * Praktikum Computergrafik mit VRML und Java 3D</a> der
 * <a href="http://www.agc.fhg.de" target="_blank">Professur f&uuml;r Grafische
 * Datenverarbeitung</a> am
 * <a href="http://www.informatik.uni-frankfurt.de" target="_blank">Fachbereich
 * Informatik und Biologie</a> der <a href="http://www.uni-frankfurt.de" target="_blank">
 * Johann Wolfgang Goethe-Universit&auml;t Frankfurt</a> entwickelt worden ist.
 * Die Anwendung inklusive Quellcode, Screenshots, Autorinformationen und
 * umfangreichen Informationsmaterial kann
 * unter <a href="http://www.stormzone.de/uni/pacman3d/" target="_blank">
 * www.stormzone.de/uni/pacman3d/</a> bezogen werden.</p>
 *
 * @see                 pacman3d.labyrinth.cells.Cell
 * @see                 pacman3d.labyrinth.cells.Item
 * @see                 pacman3d.util.CellHandler
 * @see                 pacman3d.util.ItemHandler
 * @see                 pacman3d.util.ResourceHandler
 * @author              Fabian Wleklinski
 *                      (<a href="mailto:Fabian@Wleklinski.de">Fabian@Wleklinski.de</a>)
 *                      (Labyrinth-Gruppe)<br>
 * @version             $Date: 2002/04/02 12:57:37 $<br>
 */
public class ItemHandler extends TypeHandler implements XMLSerializeable {

	/**
	 * Die einzige Instanz des <code>ItemHandler</code>, die es geben darf
	 * (Factory-Konzept).
	 * @see getInstance()
	 */
	private static ItemHandler m_oInstance = null;

	/**
	 * Konstruktor - sollte nicht direkt aufgerufen werden. Statt dessen
	 * <code>getInstance()</code> benutzen (Factory-Konzept).
	 *
	 * @see getInstance()
	 */
	private ItemHandler() {
		super( false, LevelXmlSyntax.TAG_ITEMTYPES, LevelXmlSyntax.TAG_ITEMTYPE );
	}

	/**
	 * Re-initialisiert das Item-Repository, d.h. loescht alle verwalteten
	 * Item-Instanzen.
	 */
	public void reset() {
		super.reset();
	}

	/**
	 * <p>Factory des <code>ItemHandler</code>. Diese Methode ist zu
	 * verwenden um an eine Instanz des <code>ItemHandler</code> zu
	 * gelangen. Gem&auml;&szlig; des Factory-Konzeptes kann zu jedem
	 * Zeitpunkt der Programmausf&uuml;hrung nur eine einzige Instanz des
	 * <code>ItemHandler</code> existieren.</p>
	 * <p>Beispiel:</p>
	 * <pre>
	 * ItemHandler oItemHandler = ItemHandler.getInstance();
	 * oItemHandler.addItem( ... );
	 * </pre>
	 *
	 * @return Eine Instanz des <code>ItemHandler</code>.
	 */
	public static ItemHandler getInstance() {
		if (m_oInstance == null) {
			m_oInstance = new ItemHandler();
		}

		return m_oInstance;
	}

	/**
	 * Ermittelt den Namen des &uuml;bergebenen Item (<code>oItem</code>),
	 * und gibt diesen zur&uuml;ck, sofern das Item im Itempool enthalten
	 * ist. Anderenfalls wird <code>null</code> zur&uuml;ckgegeben.
	 *
	 * @param oItem         Das Item, dessen Name gesucht ist.
	 * @return              Der Name des Items oder <code>null</code>.
	 */
	public String getItemName( pacman3d.labyrinth.items.Item oItem ) {
		return getTypeName( oItem );
	}

	/**
	 * Gibt das Item mit dem angegebenen Namen (<code>sName</code>)
	 * zur&uuml;ck. Das Item muss zuvor mittels <code>addItem()</code>
	 * hinzugef&uuml;gt worden sein. Falls das Item nicht im Itempool
	 * enthalten ist, wird <code>null</code> zur&uuml;ckgegeben.
	 *
	 * @param sName         Der Name des zur&uuml;ckzugebenden Item.
	 * @return              Das Item mit dem angegebenen Namen bzw.
	 *                      <code>null</code>.
	 * @see addItem(pacman3d.labyrinth.items.Item)
	 */
	public pacman3d.labyrinth.items.Item getItemInstance( String sName ) {
		pacman3d.labyrinth.items.Item oResult =
			(pacman3d.labyrinth.items.Item) getInstanceOf( sName );

		// neues Item am MessageService anmelden!
		pacman3d.message.MessageService.getInstance().addMessageListener(
			oResult );

		return oResult;
	}

	/**
	 * F&uuml;gt ein Item hinzu, und erzeugt daf&uuml;r selbstst&auml;ndig
	 * einen neuen, eindeutigen Bezeichner. Falls das Item bereits abgelegt
	 * ist, wird es kein weiteres Mal hinzugef&uuml;gt, sondern der Name
	 * des abgelegten Items zur&uuml;ckgegeben.
	 *
	 * @param oItem         Das hinzuzuf&uuml;gende Item.
	 * @return              Der Bezeichner, unter dem das Item abgelegt
	 *                      ist.
	 */
	public String addItem( pacman3d.labyrinth.items.Item oItem ) {
		ItemHandlerItem oItemHandlerItem = new ItemHandlerItem( oItem );
		addType( oItemHandlerItem );
		return oItemHandlerItem.getName();
	}

	/**
	 * Serialisiert alle Daten eines einzelnen Item in eine XML-Struktur,
	 * und gibt diese zur&uuml;ck. Der <code>ItemHandler</code>
	 * wendet diese Methode auf alle Items an, wenn er serialisiert
	 * wird (<code>saveToDOM()</code>). Die serialisierten Daten k&ouml;nnen
	 * mittels <code>loadTypeFromDOM()</code> wieder deserialisiert werden.
	 *
	 * @param oTypeHandlerItem      Das in einer Instanz von
	 *                              <code>TypeHandlerItem</code> gekapselte
	 *                              Item, welches serialisiert werden soll.
	 * @param oDocument             Ein XML-Dokument zwecks Erstellung von Knoten.
	 * @see saveToDOM(org.w3c.dom.Document)
	 * @see loadTypeFromDOM(org.w3c.dom.Element)
	 */
	protected org.w3c.dom.Element saveTypeElementToDOM(
		TypeHandlerItem oTypeHandlerItem, org.w3c.dom.Document oDocument ) {

		// ein neues Element fr das Item erstellen
		org.w3c.dom.Element oItemTypeElement = oDocument.createElement(
			LevelXmlSyntax.TAG_ITEMTYPE );

		// ein neues Attribut fr den Namen des Typs erstellen und
		// hinzufgen
		oItemTypeElement.setAttribute( LevelXmlSyntax.ATTR_NAME,
			oTypeHandlerItem.getName() );

		// das Item sich serialisieren lasse
		pacman3d.labyrinth.items.Item oItem =
			(pacman3d.labyrinth.items.Item) oTypeHandlerItem.getInstance();
		org.w3c.dom.Element oItemElement = oItem.saveToDOM( oDocument );
		if (oItemElement != null) {
			oItemTypeElement.appendChild( oItemElement );
		}

		return oItemTypeElement;
	}

	/**
	 * Deserialisiert alle Daten eines einzelnen Items aus der
	 * gegebenen XML-Struktur (<code>oTypeElement</code>). Der
	 * <code>ItemHandler</code> wendet diese Methode auf serialisierte
	 * Items an, wenn er deserialisiert wird (<code>loadFromDOM()</code>).
	 * Die serialisierten Daten m&uuml;ssen zuvor mittels
	 * <code>saveTypeElementToDOM()</code> serialisiert worden sein.
	 *
	 * @param oTypeElement  Das XML Element, welches die zu
	 *                      deserialisierenden Daten enth&auml;lt.
	 * @exception   XMLMissingAttributeException Wird geworfen wenn ein
	 *              benoetigtes XML-Attribut
	 *              nicht vorhanden ist, oder keinen Wert enthaelt
	 *              (<code>&lt;author name=""/&gt;</code>).
	 * @exception   ClassNotFoundException Wird geworfen wenn der Versuch
	 *              eine Java-Klasse zu instanzieren fehlschlaegt. Die
	 *              Instanzierung von Klassen ist ntig, wenn in den
	 *              XML-Daten Referenzen auf Java-Klassen vorhanden sind
	 *              (<code>&lt;itemtype class="pacman3d.labyrinth.items.ItemPill"&gt;</code>).
	 * @see saveTypeElementToDOM(TypeHandlerItem,org.w3c.dom.Document)
	 * @see loadFromDOM(org.w3c.dom.Element)
	 */
	protected void loadTypeFromDOM( org.w3c.dom.Element oTypeElement )
		 throws XMLMissingAttributeException, ClassNotFoundException {

		/* access the class loader */
		ClassLoader oClassLoader = this.getClass().getClassLoader();

		/* get the name of the current type */
		String sCurName =
			oTypeElement.getAttribute( LevelXmlSyntax.ATTR_NAME );

		// get the <item>-Element
		org.w3c.dom.NodeList oItemElements =
			oTypeElement.getElementsByTagName(
			LevelXmlSyntax.TAG_ITEM );
		if (oItemElements.getLength() != 1) {
			Debug.out( getClass().getName(),
				"needing 1 <item/> below <itemtype/>!",
				Debug.LEVEL_CRITICAL );
			return;
		}

		org.w3c.dom.Element oItemElement =
			(org.w3c.dom.Element) oItemElements.item( 0 );

		/* get the java class name of the current type */
		String sCurJavaClass =
			oItemElement.getAttribute(
			LevelXmlSyntax.ATTR_JAVACLASS );

		if (sCurName.equals( "" )) {
			throw new XMLMissingAttributeException(
				LevelXmlSyntax.ATTR_NAME );
		}

		if (sCurJavaClass.equals( "" )) {
			throw new XMLMissingAttributeException(
				LevelXmlSyntax.ATTR_JAVACLASS );
		}

		/* get params */
		org.w3c.dom.NodeList oParamsElements =
			oItemElement.getElementsByTagName(
			LevelXmlSyntax.TAG_PARAMS );

		int iNumberOfParams = oParamsElements.getLength();
		org.w3c.dom.Element oParamsElement = null;
		if (iNumberOfParams == 1) {
			oParamsElement = (org.w3c.dom.Element) oParamsElements.item( 0 );
		}

		/* create an instance of this type */
		Class oClass = null;
		try {
			oClass = oClassLoader.loadClass( sCurJavaClass );
		} catch ( ClassNotFoundException oException ) {
			Debug.out( getClass().getName(),
				"could not load class '" +
				sCurJavaClass + "'!",
				Debug.LEVEL_CRITICAL );

			throw oException;
		}

		Object oObject = createInstanceOf( oClass );

		if (oObject != null) {
			if (oParamsElement != null) {
				((XMLSerializeable) oObject).loadFromDOM( oParamsElement );
			}

			pacman3d.util.ItemHandlerItem oItemHandlerItem =
				new pacman3d.util.ItemHandlerItem(
				(pacman3d.labyrinth.items.Item) oObject, sCurName );
			addType( oItemHandlerItem );
		}
	}
}