package pacman3d.util;

/**
 * <b>Title:</b>      	Pacman 3D - Nicht wirklich ;)</br>
 * <b>Description:</b><br>
 *
 * <p>Klasse fuer die Ausgabe von Debugmeldungen.</p>
 *
 * <b>Copyright:</b>	Copyright (c) 2002<br>
 *
 * @author              Labyrinth-Gruppe<br>
 * @version             7.1.2002<br>
 */
public class Debug {

        /**
         * Der zuletzt erzeugte Einzug als character-Feld.
         */
        private static char[] m_iIndent = null;

        /**
         * Der <code>PrintStream</code>, auf dem die Ausgabe erfolgen soll.
         * Wird er nicht explizit gesetzt, erfolgt die Ausgabe auf der
         * Konsole.
         * @see setPrintStream()
         */
        private static java.io.PrintStream m_oPrintStream = System.err;

        /**
         * Die Anzahl Zeichen pro Zeile, die ausgegeben werden sollen.
         */
        private static int m_iCharsPerLine = 80;

        /**
         * Der minimale Debuglevel, ab dem Meldungen ausgegeben werden.
         */
        private static int m_iDebugLevel = java.lang.Integer.MIN_VALUE;

        /**
         * <code>true</code> wenn bei der Ausgabe von Exceptions auch
         * ein Stacktrace angezeigt werden soll
         */
        private static boolean m_bPrintStackTrace = true;

        /**
         * Debuglevel "Kritischer Fehler" fuer das Melden von Fehlern, die
         * i.A. zum Programmabbruch fuehren, wie z.B. das Fehlen von
         * benoetigten Dateien oder Ressourcen.
         */
        public static int LEVEL_CRITICAL = 10;

        /**
         * Debuglevel "Warnung" fuer nicht-kritische Fehler, die i.A. <b>nicht</b>
         * zum Programmabbruch fuehren, wie z.B. das Fehlen von Texturdateien,
         * die nicht zwingend erforderlich sind.
         */
        public static int LEVEL_WARNING = 5;

        /**
         * Debuglevel "Notiz" fuer Meldungen die keinen Fehlern entsprechen und
         * daher auch keinen Einfluss auf die Programmausfuehrung haben, wie
         * z.B. die Ausgabe von Berechungsergebnissen. Dieser Debuglevel ist
         * der Vorgabe-Debuglevel fuer Meldungen.
         */
        public static int LEVEL_NOTICE = 0;

        /**
         * Da diese Klasse nicht instanziiert werden muss, den ererbten
         * Konstruktor verstecken.
         */
        private Debug() {
        }

        /**
         * Setzt den <code>PrintStream</code>, auf dem die Ausgabe aller
         * Meldungen erfolgen soll. Wird der <code>PrintStream</code> nicht
         * explizit definiert, erfolgt die Ausgabe auf der Konsole.
         * @param oPrintStream  Der <code>PrintStream</code>, auf dem alle
         *                      Meldungen ausgegeben werden sollen.
         */
        public static void setPrintStream( java.io.PrintStream oPrintStream ) {
                m_oPrintStream = oPrintStream;
        }

        /**
         * Setzt die Anzahl Zeichen pro Zeile, die ausgegeben werden sollen.
         */
        public static void setCharsPerLine( int iCharsPerLine ) {
                m_iCharsPerLine = iCharsPerLine;
        }

        /**
         * Setzt den Level, ab dem auf die Konsole reportet wird.
         */
        public static void setDebugLevel( int iDebugLevel ) {
                m_iDebugLevel = iDebugLevel;
        }

        /**
         * Definiert ob bei der Ausgabe von Exceptions auch
         * ein Stacktrace angezeigt werden soll.
         */
        public static void setPrintStackTrace( boolean bPrintStackTrace ) {
                m_bPrintStackTrace = bPrintStackTrace;
        }

        /**
         * Nimmt eine Debugmeldung mit einem gegebenen Klassennamen und
         * Debuglevel entgegen. Die Entscheidung, ob die Meldung ausgegeben
         * wird, wird anhand des uebergebenen Debuglevel gefaellt.
         *
         * @param sClassName    Der Klassenname der meldenden Klasse
         * @param sMessageText  Der Meldungstext
         * @param iDebugLevel   Der Debuglevel der Meldung
         * @see setDebugLevel()
         */
        public static void out( String sClassName, String sMessageText, int iDebugLevel ) {
                if (iDebugLevel >= m_iDebugLevel) {
                        printMessage( sClassName, sMessageText, iDebugLevel );
                }
        }

        /**
         * Nimmt eine Exception mit einem gegebenen Klassennamen und
         * Debuglevel entgegen. Die Entscheidung, ob die Meldung ausgegeben
         * wird, wird anhand des uebergebenen Debuglevel gefaellt.
         *
         * @param sClassName    Der Klassenname der meldenden Klasse
         * @param oException    Die auszugebende Exception
         * @param iDebugLevel   Der Debuglevel der Meldung
         * @see setDebugLevel()
         */
        public static void out( String sClassName, Exception oException, int iDebugLevel ) {
                if (iDebugLevel >= m_iDebugLevel) {
                        printMessage( sClassName, oException.toString(), iDebugLevel );

                        if (m_bPrintStackTrace) {
                                java.io.ByteArrayOutputStream oByteArrayOutputStream =
                                        new java.io.ByteArrayOutputStream();
                                java.io.PrintStream oPrintStream = new java.io.PrintStream( oByteArrayOutputStream );
                                oException.printStackTrace( oPrintStream );
                                String sStackTrace = oByteArrayOutputStream.toString();
                                m_oPrintStream.println( formatText( sStackTrace, 32, m_iCharsPerLine ) );
                        }
                }
        }

        /**
         * Nimmt eine Debugmeldung mit dem Vorgabe-Debugvalue ("Notiz") entgegen.
         *
         * @param sClassName    Der Klassenname der meldenden Klasse
         * @param sMessageText  Der Meldungstext
         * @see setDebugLevel()
         */
        public static void out( String sClassName, String sMessageText ) {
                out( sClassName, sMessageText, LEVEL_NOTICE );
        }

        /**
         * Nimmt eine Exception mit dem Vorgabe-Debugvalue ("Notiz") entgegen.
         *
         * @param sClassName    Der Klassenname der meldenden Klasse
         * @param oException    Die auszugebende Exception
         * @see setDebugLevel()
         */
        public static void out( String sClassName, Exception oException ) {
                out( sClassName, oException, LEVEL_NOTICE );
        }

        /**
         * Gibt eine Meldung unbedingt aus.
         *
         * @param sClassName    Der Klassenname der meldenden Klasse
         * @param sMessageText  Der Meldungstext
         * @param iDebugLevel   Der Debuglevel der Meldung
         */
        private static void printMessage( String sClassName, String sMessageText, int iDebugLevel ) {
                String sOutText = "";

                if (iDebugLevel == LEVEL_NOTICE) {
                        // sOutText += "NOTICE!\n";
                } else if (iDebugLevel == LEVEL_WARNING) {
                        sOutText += "########## ! WARNING ! ##########\n";
                } else if (iDebugLevel == LEVEL_CRITICAL) {
                        sOutText += "########## !!! CRITICAL !!! ##########\n";
                }

                int iMaxLength = 30;
                sClassName += ":";
                while (sClassName.length() < iMaxLength) sClassName += " ";
                if (sClassName.length() > iMaxLength) {
                        sOutText += "[..]" + sClassName.substring( sClassName.length()-iMaxLength+4, sClassName.length() );
                } else {
                        sOutText += sClassName;
                }

                sOutText += "  ";

                int i=0;
                int chars = m_iCharsPerLine - 40;
                while ((sMessageText.length()-i) > chars) {
                        sOutText += sMessageText.substring( i, i+chars ) + "\n                                ";
                        i += chars;
                }

                sOutText += sMessageText.substring( i );

                m_oPrintStream.println( sOutText );

        }

        /**
         * Formatiert den gegebenen Text fuer zeilenweise Ausgabe.
         * @param iIndent       Die Breite des Einzugs am linken Rand in Zeichen.
         * @param iLineLength   Die Anzahl von Zeichen pro Zeile, die ausgegeben
         *                      werden sollen.
         */
        private static String formatText( String sText, int iIndent, int iLineLength ) {
                java.lang.StringBuffer oStringBuffer = new java.lang.StringBuffer( sText );

                if ((m_iIndent == null) || (m_iIndent.length < iIndent)) {
                        m_iIndent = new char[ iIndent ];
                        for (int i=0; i<iIndent; i++) {
                                m_iIndent[ i ] = ' ';
                        }
                }

                /* die Laenge des gegebenen Textes merken */
                int iLen = oStringBuffer.length();
                int iChar = 0;
                for (int i=0; i<iLen; i++) {
                        if (oStringBuffer.charAt(i) == '\n') {
                                oStringBuffer.insert( i+1, m_iIndent, 0, iIndent );
                                iLen = iLen + iIndent;
                                i = i + iIndent;
                                iChar = iIndent;
                        } else if (iChar == 0) {
                                oStringBuffer.insert( i, "\n" );
                                oStringBuffer.insert( i+1, m_iIndent, 0, iIndent );
                                iLen = iLen + iIndent + 1;
                                i = i + iIndent + 1;
                                iChar = iIndent;
                        }
                        iChar++;

                        if (iChar == iLineLength) {
                                iChar = 0;
                        }
                }

                return oStringBuffer.toString();
        }
}