package pacman3d.util;

import pacman3d.labyrinth.cells.*;
import java.util.Hashtable;

/**
 * <p>Diese Klasse verwaltet Zellen (d.h. Spielfelder) wie sie f&uuml;r
 * Pacman&nbsp;3D ben&ouml;tigt werden. Jede Zelle entspricht einer Instanz
 * der Klasse <code>pacman3d.labyrinth.cells.Cell</code>. F&uuml;r den Umgang
 * mit diesen Zellen werden Methoden zum Hinzuf&uuml;gen, Serialisieren und
 * Deserialisieren einer Zellensammlung zur Verf&uuml;gung gestellt.</p>
 *
 * <p>In der Architektur von Pacman&nbsp;3D besteht ein Level aus einer
 * dreidimensionalen Struktur von Spielfeldern, die &quot;Zellen&quot; genannt
 * werden, und jeweils Instanzen von <code>pacman3d.labyrinth.cells.Cell</code>
 * (bzw. eines Nachfahren) sind. Alle Zellen werden von der Klasse
 * <code>pacman3d.util.CellHandler</code> verwaltet. In jeder Zelle k&ouml;nnen
 * beliebig viele Objekte wie z.B. Punkte-Pillen oder Erste-Hilfe-Koffer
 * existieren, die &quot;Items&quot; genannt werden, und jeweils Instanzen von
 * <code>pacman3d.labyrinth.items.Item</code> (bzw. eines Nachfahren) sind.
 * Alle Items werden von der Klasse <code>pacman3d.util.ItemHandler</code>
 * verwaltet. Zellen und Items, sowie andere Objekte des Spiels (Pacmans,
 * Monster, ...) verwenden Ressourcen in Form von Grafik- oder Audiodateien,
 * oder anderen Medientypen. Alle diese Ressourcen werden von der Klasse
 * <code>pacman3d.util.ResourceHandler</code> verwaltet.
 *
 * <p>Ein Labyrinth besteht in Pacman&nbsp;3D aus tausenden von Zellen, von
 * denen im Allgemeinen viele Zellen &quot;identisch&quot; sind. Ein Spielfeld
 * von z.B. 20x25x5 Zellen besteht aus 2.500 Zellen, die sich im Allgemeinen
 * auf nicht mehr als f&uuml;nf bis zehn verschieden parametrisierte Zellen
 * reduzieren lassen. Es ist die Aufgabe des <code>CellHandler</code>, diese
 * Zellen zu verwalten, und die angesprochene Redundanz zu vermeiden.</p>
 *
 * <p>Diese Klasse erf&uuml;llt unter Anderem die folgenden Aufgaben:</p>
 * <ul>
 *   <li>Serialisierung und Deserialisierung der ihr anvertrauten Zellen
 *   in bzw. aus einem XML-Datenstrom.</li>
 *   <li>Reduzierung des Hauptspeicherbedarfs durch Vermeidung von Duplikaten
 *   im Speicher, unabh&auml;ngig davon, wie viele Klassen eine Zelle
 *   verwenden.</li>
 *   <li>Reduzierung der Exportgr&ouml;&szlig; bei XML-Serialisierung durch
 *   Vermeidung von Duplikaten, unabh&auml;ngig davon, wie viele Klassen eine
 *   Zelle verwenden.</li>
 *   <li>...</li>
 * </ul>
 *
 * <p>Es kann zu jeder Zeit nur eine einzige Instanz von
 * <code>CellHandler</code> geben. Es existiert die statische Factory-
 * Methode <code>getInstance</code>, um eine Referenz auf die Instanz zu
 * bekommen.</p>
 *
 * <p>Diese Klasse ist Bestandteil der Anwendung
 * <a href="http://www.stormzone.de/uni/pacman3d/" target="_blank">
 * Pacman&nbsp;3D</a>, welche im Wintersemester 2001/02 (Oktober 2001
 * bis M&auml;rz 2002) im Rahmen des
 * <a href="http://www.agc.fhg.de/uniGoethe/lehre/ws0102/praktikum.php" target="_blank">
 * Praktikum Computergrafik mit VRML und Java 3D</a> der
 * <a href="http://www.agc.fhg.de" target="_blank">Professur f&uuml;r Grafische
 * Datenverarbeitung</a> am
 * <a href="http://www.informatik.uni-frankfurt.de" target="_blank">Fachbereich
 * Informatik und Biologie</a> der <a href="http://www.uni-frankfurt.de" target="_blank">
 * Johann Wolfgang Goethe-Universit&auml;t Frankfurt</a> entwickelt worden ist.
 * Die Anwendung inklusive Quellcode, Screenshots, Autorinformationen und
 * umfangreichen Informationsmaterial kann
 * unter <a href="http://www.stormzone.de/uni/pacman3d/" target="_blank">
 * www.stormzone.de/uni/pacman3d/</a> bezogen werden.</p>
 *
 * @see                 pacman3d.labyrinth.cells.Cell
 * @see                 pacman3d.labyrinth.cells.Item
 * @see                 pacman3d.util.CellHandler
 * @see                 pacman3d.util.ItemHandler
 * @see                 pacman3d.util.ResourceHandler
 * @author              Fabian Wleklinski
 *                      (<a href="mailto:Fabian@Wleklinski.de">Fabian@Wleklinski.de</a>)
 *                      (Labyrinth-Gruppe)<br>
 * @version             $Date: 2002/04/02 12:57:37 $<br>
 */
public class CellHandler extends TypeHandler implements XMLSerializeable {

	/**
	 * Die einzige Instanz des <code>CellHandler</code>, die es geben darf
	 * (Factory-Konzept).
	 * @see getInstance()
	 */
	private static CellHandler m_oInstance = null;

	/**
	 * Konstruktor - sollte nicht direkt aufgerufen werden. Statt dessen
	 * <code>getInstance()</code> benutzen (Factory-Konzept).
	 *
	 * @see getInstance()
	 */
	private CellHandler() {
		super( false, "celltypes", "celltype" );
	}

	/**
	 * <p>Factory des <code>CellHandler</code>. Diese Methode ist zu
	 * verwenden um an eine Instanz des <code>CellHandler</code> zu
	 * gelangen. Gem&auml;&szlig; des Factory-Konzeptes kann zu jedem
	 * Zeitpunkt der Programmausf&uuml;hrung nur eine einzige Instanz des
	 * <code>CellHandler</code> existieren.</p>
	 * <p>Beispiel:</p>
	 * <pre>
	 * CellHandler oCellHandler = CellHandler.getInstance();
	 * oCellHandler.addCell( ... );
	 * </pre>
	 *
	 * @return Eine Instanz des <code>CellHandler</code>.
	 */
	public static CellHandler getInstance() {
		if (m_oInstance == null) {
			m_oInstance = new CellHandler();
		}

		return m_oInstance;
	}

	/**
	 * Re-initialisiert den Zellenpool, d.h. entfernt alle verwalteten
	 * Zellen.
	 */
	public void reset() {
		super.reset();
	}

	/**
	 * Gibt die Zelle mit dem angegebenen Namen (<code>sName</code>)
	 * zur&uuml;ck. Diese Zelle muss zuvor mittels <code>addCell()</code>
	 * hinzugef&uuml;gt worden sein. Falls die Zelle nicht im Zellenpool
	 * enthalten ist, wird <code>null</code> zur&uuml;ckgegeben.
	 *
	 * @param sName         Der Name der zur&uuml;ckzugebenden Zelle.
	 * @return              Die Zelle mit dem angegebenen Namen bzw.
	 *                      <code>null</code>.
	 * @see addCell(pacman3d.labyrinth.cells.Cell)
	 * @see addCell(pacman3d.labyrinth.cells.Cell,boolean)
	 */
	public pacman3d.labyrinth.cells.Cell getCellInstance( String sName ) {
		return (pacman3d.labyrinth.cells.Cell) getInstanceOf( sName );
	}

	/**
	 * F&uuml;gt eine Zelle hinzu, und erzeugt daf&uuml;r selbstst&auml;ndig
	 * einen neuen, eindeutigen Bezeichner. Falls die Zelle bereits abgelegt
	 * ist, wird sie kein weiteres Mal hinzugef&uuml;gt, sondern der Name
	 * der abgelegten Zelle zur&uuml;ckgegeben.
	 *
	 * @param oCell         Die hinzuzuf&uuml;gende Zelle.
	 * @return              Der Bezeichner, unter dem die Zelle abgelegt
	 *                      ist.
	 */
	public String addCell( pacman3d.labyrinth.cells.Cell oCell ) {
		return addCell( oCell, false );
	}

	/**
	 * F&uuml;gt eine Zelle hinzu, und erzeugt daf&uuml;r selbstst&auml;ndig
	 * einen neuen, eindeutigen Bezeichner. Falls die Zelle bereits abgelegt
	 * ist, und der Parameter <code>bDoEnforce</code> auf <code>false</code>
	 * gesetzt ist, wird die Zelle kein weiteres Mal hinzugef&uuml;gt,
	 * sondern der Name der abgelegten Zelle zur&uuml;ckgegeben. Falls der
	 * Parameter <code>bDoEnforce</code> auf <code>true</code> gesetzt ist,
	 * wird das Hinzuf&uuml;gen der Zelle erzwungen, auch dann, wenn sie
	 * bereits abgelegt worden ist.
	 *
	 * @param oCell         Die hinzuzuf&uuml;gende Zelle.
	 * @param bDoEnforce    Wenn auf <code>true</code> gesetzt, wird nicht
	 *                      berprft, ob die Zelle evtl. schon
	 *                      hinzugef&uuml;gt wurde.
	 * @return              Der Bezeichner, unter dem die Zelle abgelegt
	 *                      ist.
	 */
	public String addCell( pacman3d.labyrinth.cells.Cell oCell, boolean bDoEnforce ) {
		String sCellTypeName = null;

		// ist die Zelle schon im Repository?
		if (! bDoEnforce) {
			sCellTypeName = getTypeName( oCell );
		}

		if (sCellTypeName == null) {
			CellHandlerItem oCellHandlerItem = new CellHandlerItem( oCell );
			addType( oCellHandlerItem );
			sCellTypeName = oCellHandlerItem.getName();
		}
		return sCellTypeName;
	}

	/**
	 * Deserialisiert alle Daten einer einzelnen Zelle aus der
	 * gegebenen XML-Struktur (<code>oCelltypeElement</code>). Der
	 * <code>CellHandler</code> wendet diese Methode auf serialisierte
	 * Zellen an, wenn er deserialisiert wird (<code>loadFromDOM()</code>).
	 * Die serialisierten Daten m&uuml;ssen zuvor mittels
	 * <code>saveTypeElementToDOM()</code> serialisiert worden sein.
	 *
	 * @param oCelltypeElement      Das XML Element, welches die zu
	 *                              deserialisierenden Daten enth&auml;lt.
	 * @exception   XMLMissingAttributeException Wird geworfen wenn ein
	 *              benoetigtes XML-Attribut
	 *              nicht vorhanden ist, oder keinen Wert enthaelt
	 *              (<code>&lt;author name=""/&gt;</code>).
	 * @exception   ClassNotFoundException Wird geworfen wenn der Versuch
	 *              eine Java-Klasse zu instanzieren fehlschlaegt. Die
	 *              Instanzierung von Klassen ist ntig, wenn in den
	 *              XML-Daten Referenzen auf Java-Klassen vorhanden sind
	 *              (<code>&lt;celltype class="pacman3d.labyrinth.cells.CellFloor"&gt;</code>).
	 * @see saveTypeElementToDOM(TypeHandlerItem,org.w3c.dom.Document)
	 * @see loadFromDOM(org.w3c.dom.Element)
	 */
	protected void loadTypeFromDOM( org.w3c.dom.Element oCelltypeElement )
		 throws XMLMissingAttributeException, ClassNotFoundException {

		/* access the class loader */
		ClassLoader oClassLoader = this.getClass().getClassLoader();

		/* get the name of the current type */
		String sCurName =
			oCelltypeElement.getAttribute( "name" );

		if (sCurName.equals( "" )) {
			throw new XMLMissingAttributeException(
				"name" );
		}

		/* get the <cell>-element */
		org.w3c.dom.NodeList oCellNodes =
			oCelltypeElement.getElementsByTagName( "cell" );

		/* test if there is exactly one <cell>-element */
		int iCellNodes = oCellNodes.getLength();
		if (iCellNodes < 1) {
			Debug.out( getClass().getName(), "no <cell>-element found. cannot load celltype.",
				Debug.LEVEL_CRITICAL );
			return;
		} else if (iCellNodes > 1) {
			Debug.out( getClass().getName(),
				(new Integer(oCellNodes.getLength())).toString() +
				" <cell>-elements found. needing only one. cannot load celltype.",
				Debug.LEVEL_CRITICAL );
			return;
		}

		org.w3c.dom.Element oCellElement = (org.w3c.dom.Element)
			oCellNodes.item( 0 );

		/* get the java class name of the current type */
		String sCurJavaClass =
			oCellElement.getAttribute( "javaclass" );

		if (sCurJavaClass.equals( "" )) {
			throw new XMLMissingAttributeException(
				"javaclass" );
		}

		/* get params */
		/*
		org.w3c.dom.NodeList oParamsElements =
			oCellElement.getElementsByTagName( "params" );

		int iNumberOfParams = oParamsElements.getLength();
		org.w3c.dom.Element oParamsElement = null;
		if (iNumberOfParams == 1) {
			oParamsElement = (org.w3c.dom.Element) oParamsElements.item( 0 );
		}
		*/

		/* create an instance of this type */
		Class oClass = null;
		try {
			oClass = oClassLoader.loadClass( sCurJavaClass );
		} catch ( ClassNotFoundException oException ) {
			Debug.out( getClass().getName(),
				"could not load class '" +
				sCurJavaClass + "'!",
				Debug.LEVEL_CRITICAL );

			throw oException;
		}

		Object oObject = createInstanceOf( oClass );

		if (oObject != null) {
			// if (oParamsElement != null) {
				Debug.out( this.getClass().getName(),
					"loading parameters...", Debug.LEVEL_NOTICE );
				// ((XMLSerializeable) oObject).loadFromDOM( oCellElement );
				((XMLSerializeable) oObject).loadFromDOM( oCellElement );
			// }

			Debug.out( getClass().getName(),
				"Adding instance of class '" +
				sCurJavaClass + "' to repository." );

			CellHandlerItem oCellHandlerItem = new
				CellHandlerItem( (pacman3d.labyrinth.cells.Cell) oObject, sCurName );
			addType( oCellHandlerItem );
		}
	}

	/**
	 * Serialisiert alle Daten einer einzelnen Zelle in eine XML-Struktur,
	 * und gibt diese zur&uuml;ck. Der <code>CellHandler</code>
	 * wendet diese Methode auf alle Zellen an, wenn er serialisiert
	 * wird (<code>saveToDOM()</code>). Die serialisierten Daten k&ouml;nnen
	 * mittels <code>loadTypeFromDOM()</code> wieder deserialisiert werden.
	 *
	 * @param oTypeHandlerItem      Die in einer Instanz von
	 *                              <code>TypeHandlerItem</code> gekapselte
	 *                              Zelle, welche serialisiert werden soll.
	 * @param oDocument     Ein XML-Dokument zwecks Erstellung von Knoten.
	 * @see saveToDOM(org.w3c.dom.Document)
	 * @see loadTypeFromDOM(org.w3c.dom.Element)
	 */
	protected org.w3c.dom.Element saveTypeElementToDOM(
		TypeHandlerItem oTypeHandlerItem, org.w3c.dom.Document oDocument ) {

		org.w3c.dom.Element oCelltypeElement = oDocument.createElement(
			"celltype" );
		oCelltypeElement.setAttribute( "name", oTypeHandlerItem.getName() );

		org.w3c.dom.Element oCellElement =
			((pacman3d.labyrinth.cells.Cell) oTypeHandlerItem.getInstance()).saveToDOM( oDocument );
		oCelltypeElement.appendChild( oCellElement );

		return oCelltypeElement;
	}
}