package pacman3d.pacman;

import java.net.*;
import java.util.*;
import java.io.*;

import javax.media.j3d.*;
import javax.vecmath.*;

import pacman3d.labyrinth.*;
import pacman3d.message.*;
import pacman3d.pacman.maintenance.*;
import pacman3d.pacman.animation.*;
import pacman3d.pacman.sound.*;
import pacman3d.util.*;
import pacman3d.labyrinth.items.*;
import pacman3d.labyrinth.cells.*;

import org.w3c.dom.*;

/**
 * <h1>Pacman</h1>
 *
 * <p>This Class holds all the logic for the player character of our game. To be
 * able to successfully use it, you should be sure, that the following points are
 * met. First of all you should be certain, that a KeyControl Object exists, with
 * the ID of this Pacman. Otherwise you would have a Pacman subject unable to
 * move, 'cause it'll never receive a movement message. (One possible exception
 * would be, that this Pacman is created as Shadow-Pacman. That is a Pacman, that
 * is only created on other clients to mirror the moves of the master Pacman. In
 * this case a KeyControl for the Masterpacman would exist, and since the shadow
 * Pacmen have the same ID like the Masterpacman, they to would receive the
 * movement messages).</p>
 *
 * <p>As you see there are two kinds of Pacmen, that can be instantiated:</p>
 * <ul>
 *   <li>the Masterpacman:  therefore you'd use the constructor without an
 *                          ID.&nbsp;</li>
 *   <li>the Shadowpacman:  created, when a constructor with an ID is used.</li>
 * </ul>
 * <p>When instantiated you should get the ID of this Pacman and create a
 * KeyControl for this one. </p>
 * <p>Copyright: Copyright (c) 2001<br>
 * </p>
 * @author <a href="mailto:bergmann@cs.uni-frankfurt.de">Frank Bergmann </a>
 * @version 2.3
 *
 */

public class Pacman implements
    pacman3d.message.MessageListener,
    pacman3d.util.Identifiable,
    pacman3d.util.XMLSerializeable,
    pacman3d.util.Introspectable  {

    /** the unique id ... used for receiving and sending messages */
    private ID m_oID;

    /** vector containing all the messages not yet processed */
    private Vector m_oMassageQueue;

    /** TransformGroup containing PacMan */
    private TransformGroup m_oPacmanGroup;

    /** Switch containing all TGs */
    private Switch m_oSW;

    /** current position in the labyrinth */
    private Point3i m_oCurrentPos;
    /** current real position in the universe */
    private Point3d m_oRealPos;
    /** current line of sight */
    private Vector3d m_oCurrentLineOfSight;
    /** real line of sight */
    private Vector3d m_oRealLineOfSight;
    /** the instance to send messages to */
    private MessageService m_oMessageService;
    /** the class holding the basic movement operations */
    private PacManMovement m_oMoves;

    /** Instanz des Labyrinthes, zum Orientieren */
    private Labyrinth m_oLabyrinth = null;

    /** Vector of Parameternames */
    private Vector m_oParameterNames;
    /** Vector of Parameterdescriptions */
    private Vector m_oParameterDescritpions;
    /** Vector of Parametertypes */
    private Vector m_oParameterTypes;

    /** object containing all atributes */
    private Admin m_oMaintenance;

    /** object used for playing sounds */
    private PacmanSound m_oSound;

    /** should initial transformation be applied? */
    private boolean m_bdoTransform = true;

    /** is a movement operation in progress */
    private boolean m_bMovementStarted=false;

    /** the operation to be continued */
    private Message m_oCurrentOp;
    /** the current movement to be resumed */
    private MovementMessage m_oMove= null;

    /** the number of ticks still needed to complete the operation */
    private int m_nNumberLeft;

    /** the step, when to tell Labyrinth that a new cell is entered */
    private int m_nStep;

    /** number of ticks needed to complete a movement from cell to cell */
    private int m_nMovementCost = 60;

    /** number of ticks needed to complete a 90 degree rotation  */
    private int m_nRotationCost = 30;

    /** currently selected model */
    private int m_nCurrentModel;

    /** the id for the score frame/panel ... needed to send the updated score */
    private ID m_oScoreID=null;
    /** the id for the score frame/panel ... so i can send the selected item */
    private ID m_oInventoryID=null;

    /** this vector holds the currently selected pacman models. With this, it
     *  is possible to run the death animation sequence ...
     */

    private Vector m_oPacmanModels;

    /** this boolean indicates whether gravitation influences
     *  pacman or not.
     */
    private boolean m_bIgnoreGravity = true;

    private Point3d m_oStartPostition = null;
    private Point3i m_oEndPosition = null;
    private boolean m_bIsInitialized = false;
    private Vector3d m_oInitialLineOfSight;
    private boolean m_bIsMaster;
	private static Vector m_oModels = null;
	private static Vector m_oModelDescriptions = null;
	private int m_nDeathCount = 0;
	private String m_sSelectedModel = "Moriya";
	private boolean m_bIsDemoMode = false;
	/** Vector containing all executes moves .. .needed for demo recording */
	private Vector m_oMadeMoves = new Vector();

    /** constructor initializing the PacMan. A Pacman created by this
     *  constructor can do virtually nothing. All it is good for is for
     *  creating J3D Nodes. But that is exactly its task. It'll be used
     *  by the intro, so to show the available models.
     */
    public Pacman() {

	// get IDs
	m_oID=new ID();

	// initial line of sight as encoded in objects
	m_oInitialLineOfSight = new Vector3d(0,0,-1);

	// initialize message handling
	m_oMassageQueue = new Vector();
	m_oMessageService = MessageService.getInstance();

	// initialize static model list (if not done already!)
	getModelList();

	// set looks of pacman
	m_oPacmanGroup = createPacMan("Moriya");
	m_oMaintenance = new Admin(this);
	m_oSW.setWhichChild(1);

	//set booleans
	m_bdoTransform = true;
	m_bIsMaster = true;
	m_bIsInitialized = false;

	//misc
	m_oMoves = new PacManMovement(m_oPacmanGroup);
	initParameters();

	// send first message, indicaten number of lifes left
	sendScoreMessage();
    }

	/** constructor initializing the PacMan. A Pacman created by this
     *  constructor will be a master pacman ... in other words it will interact
     *  with items & monsters.
     *  @param oLabyrinth the labyrinth the pacman is moving in
	 */
	public Pacman(Labyrinth oLabyrinth) {
       this(oLabyrinth, "Moriya",true);
    }
	/** constructor initializing the PacMan. A Pacman created by this
     *  constructor will act as shadow-pacman. In other words, it will
     *  react only to Movement Messages (or Die-Messages) but it won't
     *  take items, or slay monsters
     *  @param oID the ID, that the Pacman shall receive
     *  @param oLabyrinth the labyrinth the pacman is moving in
	 */
	public Pacman(ID oID, Labyrinth oLabyrinth) {
       this(oID, oLabyrinth, "Moriya",true);
    }

    /** constructor initializing the PacMan. A Pacman created by this
     *  constructor will be a master pacman ... in other words it will interact
     *  with items & monsters.
     *  @param oLabyrinth the labyrinth the pacman is moving in
	 *  @param sPacmanModel the string of the model to be loaded the String should
	 *         be one of the Strings from getModelList.
	 *  @param isFriendly if <b>true</b> this pacman doesn't attack other pacman
	 *         otherwise he'll treat them like he would treat monsters :)
	 */
    public Pacman(Labyrinth oLabyrinth, String sPacmanModel, boolean isFriendly) {

	// get IDs
	m_oID=new ID();
	m_oLabyrinth = oLabyrinth;
	m_oScoreID= m_oLabyrinth.getPanelScoreID();
	m_oInventoryID = m_oLabyrinth.getPanelInventoryID();

	// initial line of sight as encoded in objects
	m_oInitialLineOfSight = new Vector3d(0,0,-1);

	// initialize message handling
	m_oMassageQueue = new Vector();
	m_oMessageService = MessageService.getInstance();

	// initialize static model list (if not done already!)
	getModelList();

	// set looks of pacman
	m_sSelectedModel = (String)m_oModels.get(m_oModelDescriptions.indexOf(sPacmanModel));
	if (m_sSelectedModel == null || m_sSelectedModel.equals(""))
		 m_sSelectedModel = "Moriya";
	m_oPacmanGroup = createPacMan(m_sSelectedModel);
	m_sSelectedModel = sPacmanModel;
	m_oMaintenance = new Admin(this);
	m_oSW.setWhichChild(1);

	//set booleans
	m_bIgnoreGravity = !(oLabyrinth.hasGravity());
	m_bdoTransform = true;
	m_bIsMaster = true;
	m_bIsInitialized = false;
	m_oMaintenance.setFriendly(isFriendly);

	//misc
	m_oMoves = new PacManMovement(m_oPacmanGroup);
	initParameters();

	// send first message, indicaten number of lifes left
	sendScoreMessage();
    }
    /** constructor initializing the PacMan. A Pacman created by this
     *  constructor will act as shadow-pacman. In other words, it will
     *  react only to Movement Messages (or Die-Messages) but it won't
     *  take items, or slay monsters
     *  @param oID the ID, that the Pacman shall receive
     *  @param oLabyrinth the labyrinth the pacman is moving in
	 *  @param sPacmanModel the string of the model to be loaded the String should
	 *         be one of the Strings from getModelList.
	 *  @param isFriendly if <b>true</b> this pacman doesn't attack other pacman
	 *         otherwise he'll treat them like he would treat monsters :)
     *
     */
    public Pacman(ID oID,Labyrinth oLabyrinth,
		String sPacmanModel, boolean isFriendly) {

	// get IDs
	m_oID=oID;
	m_oLabyrinth = oLabyrinth;
	m_oScoreID= m_oLabyrinth.getPanelScoreID();
	m_oInventoryID = m_oLabyrinth.getPanelInventoryID();

	// initial line of sight as encoded in objects
	m_oInitialLineOfSight = new Vector3d(0,0,-1);

	// initialize message handling
	m_oMassageQueue = new Vector();
	m_oMessageService = MessageService.getInstance();

	// initialize static model list (if not done already!)
	getModelList();

	// set looks of pacman
	m_sSelectedModel = (String)m_oModels.get(m_oModelDescriptions.indexOf(sPacmanModel));
	if (m_sSelectedModel == null || m_sSelectedModel.equals(""))
		 m_sSelectedModel = "Moriya";
	m_oPacmanGroup = createPacMan(m_sSelectedModel);
	m_sSelectedModel = sPacmanModel;

	m_oMaintenance = new Admin(this);
	m_oSW.setWhichChild(1);

	//set booleans
	m_bIgnoreGravity = !(oLabyrinth.hasGravity());
	m_bdoTransform = true;
	m_bIsMaster=false;
	m_bIsInitialized = false;
	m_oMaintenance.setFriendly(isFriendly);

	//misc
	m_oMoves = new PacManMovement(m_oPacmanGroup);
	initParameters();

	// send first message, indicaten number of lifes left
	sendScoreMessage();
    }
    /**
     * initializes the parameters
     */
    private void initParameters() {
	m_oParameterNames= new Vector();
	m_oParameterTypes=new Vector();
	m_oParameterDescritpions=new Vector();
	m_oParameterNames.add("score");
	m_oParameterTypes.add(new Integer(0));
	m_oParameterDescritpions.add("add a basis score ... ");
	m_oParameterNames.add("name");
	m_oParameterTypes.add(new String());
	m_oParameterDescritpions.add("select a name for the pacman");
	m_oParameterNames.add("friendly");
	m_oParameterTypes.add(new Boolean(true));
	m_oParameterDescritpions.add("indicates whether to play friendly or not");
    }
    /** adds Message to the Message queue
     *  @param oMessage the message to be added
     */
    public void getMessage(Message oMessage) {

	if (m_bIsMaster && !(m_bIsDemoMode)) {
		Vector oNewMessageQueue = new Vector();
		for (Iterator i = m_oMassageQueue.iterator(); i.hasNext(); ) {
			Message oTempMessage = (Message)i.next();
			Object oTemp =oTempMessage.getContent();
			if (!(oTemp instanceof MovementMessage)) {
			oNewMessageQueue.add(oTempMessage);
			}
		}
		oNewMessageQueue.add(oMessage);
		m_oMassageQueue=oNewMessageQueue;
	} else {

		m_oMassageQueue.add(oMessage);
	}

	// Debug.out (this.getClass().getName(), "getMessage: Message received", Debug.LEVEL_NOTICE);

    }


    /**
     * <p>This is the main function for all operations regarding this pacman.
     * It operates as follows. First will be decided whether an operation is
     * in progress or not.
     *
     * <ul>
     * <li>If so this operation will be resumed. </li>
     * <li>If not, another operation will be taken out of the MessageQueue.
     * In the case that there would be no new operation to be processed, this
     * function will end.</li>
     * </ul></p>
     *
     * @param nTicks the number of ticks available for operations
     */
    public void makeLoop (int nTicks)  {
	int nTickAmount = nTicks;

	boolean bRunAgain = true;
	if (m_nDeathCount > 0) {
		m_nDeathCount-=nTickAmount;
		bRunAgain = false;
		if (m_nDeathCount <= 0) {
			setNewStartPoint();
			showDeathAnimation();
			m_oMaintenance.setInvincible(15);
			if (m_bIsMaster) slayMonster();
		}
	}
	    m_oMaintenance.checkModes();
	while (bRunAgain) {

	    /*
	     * first decide whether an operation is in progress ...
	     */
	    if (m_oMove != null) {

		/**
		 * the operation to be resumed is a Movement Operation
		 */
		/**
		 * resume rotation of pacman
		 */

		if (m_oMove.getAction().equalsIgnoreCase("rotate")) {

		    /** handleRotation */
		    handleRotation(m_oMove);
		    nTickAmount--;

		} else {

		    nTickAmount-=handleMovement(m_oMove, nTickAmount);

		}
		if (nTickAmount<= 1) bRunAgain=false;
	    } else {

		/*
		 * there is no operation to be resumed. So a new Message will be
		 * processed.
		 */
		if (m_oMassageQueue.isEmpty()) {
		    if (m_bIgnoreGravity == false) {
					    doFall();
		    }

					// since we've got nothing to do, maybe there is something,
					// we could take with us

		    Cell oCell = m_oLabyrinth.getCell(m_oCurrentPos);
		    if (m_bIsMaster) {
				doTakeFrom(oCell);
				slayMonster();
		    }

		    return;

		} else {

		    Point3i oStartPos = m_oLabyrinth.getPacmanPos(m_oID);
		    m_oCurrentOp = (Message) m_oMassageQueue.get(0);
            if (m_bIsMaster) {
				try {

					Message oTempMessage = new Message(m_oID,m_oCurrentOp.getContent(),m_oID);
					m_oMessageService.sendMessage(oTempMessage,m_oMessageService.PROPAGATE_NETWORK);
					m_oMadeMoves.add(oTempMessage);

				}
				catch (Exception ex) {

					Debug.out(this.getClass().getName(),
						"Error propagating message to myself" ,Debug.LEVEL_NOTICE );
					Debug.out(this.getClass().getName(),ex);

				}
            }
		    Object oContent = m_oCurrentOp.getContent();

					if ((oContent instanceof MovementMessage) ||
					    (oContent instanceof FallingContent) ) {

					    // first get the movement operation to be started
					    if (oContent instanceof MovementMessage)
			    m_oMove = (MovementMessage)oContent;
					    else if (oContent instanceof FallingContent)
						    m_oMove = ((FallingContent)oContent).getMove();

					    // decide whether a rotation or a move should be
					    // performed

					    if (m_oMove.getAction().equalsIgnoreCase("move")) {

						    m_oEndPosition=getNextPosition(oStartPos,
							   m_oMove.getDirection());
						    Cell oTemp = m_oLabyrinth.getCell(m_oEndPosition);

			    if (oTemp.isWalkable()) {

				if (m_oMove.isFast()) {
									m_nNumberLeft =
										(int)Math.ceil(m_nMovementCost/2);
									m_nStep =
										(int)Math.ceil(m_nMovementCost/4);
				} else {
									m_nNumberLeft = m_nMovementCost;
									m_nStep =
										(int)Math.ceil(m_nMovementCost/2);
				}

				// get current possition as written in labyrinth
				m_oStartPostition = convertToPoint(oStartPos);
				// set cell to walk to
								m_oSound.playSound(1);

			    } else {

								m_oCurrentOp=null;
								m_oMove = null;
								m_oSound.playSound(0);
								if (m_bIgnoreGravity == false) {
													doFall();
								}
								/** well if we are at it ... we could look, whether
								 *  or not to kill a monster or two :)
								 */
								if (m_bIsMaster) slayMonster();
			    }
			} else {

			    m_oMoves.setRotateX(Math.PI / (2*m_nRotationCost));
			    m_oMoves.setRotateY(Math.PI / (2*m_nRotationCost));
			    m_oMoves.setRotateZ(Math.PI / (2*m_nRotationCost));

			    m_oSound.playSound(1);
			    boolean bTemp=false;
			    if (m_oMove.getDirection().equalsIgnoreCase("left")) {
								bTemp = m_oMoves.LEFT_ROTATION;
			    } else if(m_oMove.getDirection().equalsIgnoreCase("right")) {
								bTemp = m_oMoves.RIGHT_ROTATION;
			    }

						    m_nNumberLeft = m_nRotationCost;
			    m_oMoves.setLineOfSightRotation(
							    this.getLineOfSight(),
							    getNextLineOfSight(this.getLineOfSight(),
									       m_oMove.getDirection()),bTemp);

			}

		    } else if (oContent instanceof Integer){
				setParameter("invincible", (Integer)oContent);
		    } else if (oContent instanceof String){
					    if (m_bIsMaster) {
						    if( ((String) oContent).equalsIgnoreCase("info")) {

								Debug.out(this.getClass().getName(),
									  ".makeLoop: LOS: " + m_oCurrentLineOfSight +
									  "\n POS: " + m_oCurrentPos +
									  "\n realPos: " +  m_oRealPos+
									  "\n other: " + m_oMaintenance,
									  Debug.LEVEL_NOTICE);

								Cell oCellTemp = m_oLabyrinth.getCell(m_oCurrentPos);
								Debug.out(this.getClass().getName(),
									  "CurrentCell.hasitems(): " + oCellTemp.hasItems() +
									  "\n items: " + oCellTemp.getItemsVector(),
									  Debug.LEVEL_NOTICE);


							} else if( ((String) oContent).equalsIgnoreCase("select next model")) {

								m_oSW.setWhichChild(
							    (m_oSW.getWhichChild()+1)% m_oSW.numChildren());

							} else if( ((String) oContent).equalsIgnoreCase("last item")) {

								m_oMaintenance.InventoryLast();
								sendInventoryMessage();

						    } else if( ((String) oContent).equalsIgnoreCase("next item")) {

								m_oMaintenance.InventoryNext();
								sendInventoryMessage();

						    } else if( ((String) oContent).equalsIgnoreCase("use item")) {

								m_oMaintenance.ItemUse();
								sendInventoryMessage();
								sendScoreMessage();

							} else if( ((String) oContent).equalsIgnoreCase("kill Pacman")) {

								Debug.out(this.getClass().getName(),
									"pacman death animation called",
									Debug.LEVEL_NOTICE);

								showDeathAnimation ();
							} else if( ((String) oContent).equalsIgnoreCase("request_score_content_message")) {

								sendGameScoreMessage();
							}else if( ((String) oContent).equalsIgnoreCase("request_score_content_message_and_quit")) {

								sendGameScoreMessage();
								m_oLabyrinth.endGame(m_oID,
										m_oMaintenance.getPoints().intValue() );

							}

					    }
					    if( ((String) oContent).equalsIgnoreCase("die")) {
							Debug.out(this.getClass().getName(),
								"die message received from " + m_oCurrentOp.getSender()
								+ " actual DeathCount: " + m_nDeathCount
								+ " bool: " + (m_nDeathCount <= 0));
						    if ((!m_oMaintenance.isInvincible()) && (m_nDeathCount <= 0)) {
								boolean bAnotherLifeLeft = m_oMaintenance.gotAnotherLife() ;
								if (m_bIsMaster) sendScoreMessage();
								if (bAnotherLifeLeft || m_oLabyrinth.isMultiplayerGame()) {

									// ok we've got another life left ... so all i
									// have to do is to set up a new position, and
									// enable invulnerability for 10 seconds

									m_nDeathCount = 1000;

								    if (m_bIsMaster) {
										try {
											m_oLabyrinth.getCamera().SetQuake(100, 5, 5, 5, .7, 1, 1.5);
										}
										catch (Exception ex) {
											Debug.out(this.getClass().getName(),"couldn't start quake");
										}
										m_oMaintenance.die();
									}


									showDeathAnimation();



								} else {

									// gosh ... game over ... that's it for this
									// time ... all that is left is to tell the
									// labyrinth, that i can't go on

									m_oMaintenance.die();
									showDeathAnimation();

									m_oLabyrinth.endGame(m_oID,
										m_oMaintenance.getPoints().intValue() );

								}
						    }
						}
						else if (((String) oContent).equalsIgnoreCase("endGame")) {

							Debug.out(this.getClass().getName(),"game ended");

							m_oLabyrinth.endGame(m_oID,
									m_oMaintenance.getPoints().intValue() );

						} else if (((String) oContent).equalsIgnoreCase("hidePacman")) {
							Debug.out(this.getClass().getName(),"hidePacman");
							setVisible(false);
						} else if (((String) oContent).equalsIgnoreCase("showPacman")) {
							Debug.out(this.getClass().getName(),"showPacman");
							setVisible(true);
						}

					    m_oCurrentOp = null;
					    m_oMove=null;
					}
					m_oMassageQueue.remove(0);

				}
				if (nTickAmount<= 1) bRunAgain=false;
	    }
	}
    }
    /** updates the visibility status of PacMan
     *  @param bIsVisible if <b>true</b> pacman is visible otherwise not
     */
    public void setVisible (boolean bIsVisible) {
		if (bIsVisible) {
			m_oSW.setWhichChild(m_nCurrentModel);
		} else {
			if (m_oSW.getWhichChild() > 0) {
			m_nCurrentModel = m_oSW.getWhichChild();
			m_oSW.setWhichChild(0);
			}
		}
    }

    /** returns Java3D Node containing PacMan <b> this function should
     *  be called only once right after creaing PacMan </b>
     *
     *  @return the J3DNode containing this pacman
     */
    public javax.media.j3d.Node getJ3DNode () {
	    return m_oPacmanGroup;
    }

    /** explicitly sets the position <b>should be called after
     *  initalizing pacman </b>
     *
     *  @param oPoint the position where pacman will spawn
     */
    public void setPacmanPos(Point3i oPoint) {
	/*m_oCurrentPos = new Point3i(nX,nY,nZ);
	m_oRealPos= new Point3d(nX,nY,nZ);
	m_oMoves.initialTransformation(new Vector3d(
						    m_oCurrentPos.x ,
						    m_oCurrentPos.y,
						    m_oCurrentPos.z),1);*/
		m_oCurrentPos = oPoint;
		m_oRealPos= new Point3d((double)oPoint.x, (double)oPoint.y, (double)oPoint.z);
		m_oMoves.initialTransformation(new Vector3d(
								m_oCurrentPos.x ,
								m_oCurrentPos.y,
								m_oCurrentPos.z),1);
    }

    /** explicitly sets the position <b>should be called after
     *  initalizing pacman </b>
     *
     * @param oPoint the position, pacman will move to
     */
    public void movePacman(Point3d oPoint) {
	    m_oMoves.setPosition(new Vector3d(oPoint.x ,oPoint.y,oPoint.z));
    }

    /** explicitly sets the position <b>should be called after
     *  initalizing pacman </b>
     *
     *  @param nX the x position of Pacman in the labyrinth
     *  @param nY the y position of Pacman in the labyrinth
     *  @param nZ the z position of Pacman in the labyrinth
     */
    public void setPacmanPos(int nX, int nY, int nZ) {
	setPacmanPos (new Point3i(nX, nY, nZ));
    }
    /** returns the current position of this pacman
     *
     *  @return current descrete position of the pacman
     */
    public Point3i getPacmanPos () {
	return m_oCurrentPos;
    }

    /** returns the real position of this pacman
     *
     *  @return real position of the pacman
     */
    public Point3d getRealPacmanPos () {
	return m_oRealPos;
    }

    /** returns line of sight
     *
     *  @return Vector3D containing the direction, that pacman looks to
     */
    public Vector3d getLineOfSight() {
	return m_oCurrentLineOfSight;
    }

    /** returns line of sight
     *
     *  @return Vector3D containing the real direction, that pacman looks to
     */
    public Vector3d getRealLineOfSight() {
	return m_oRealLineOfSight;
    }

    /**
     * sets line of sight <b> should be called initially </b>
     *
     * @param oCurrentLineOfSight the line of sight to be set
     */
    public void setLineOfSight(Vector3d oCurrentLineOfSight) {
	if (!m_bIsInitialized) {
	    /** gotta rotate the first time, i enter this one, so
	     *  the pacman won't look in the wrong direction. As
	     *  it is, i just rotate pacman to the left. In worst -
	     *  case i'll rotate pacman 3 times, which wouldn't be
	     *  too much.
	     *
	     * since the case can arise, that i wont get a valid
	     * line of sight, i kinda should break up, or else i'll
	     * be stuck in the loop.
	     */

			boolean bVectorReached = m_oInitialLineOfSight.equals(oCurrentLineOfSight);

			if ((   Math.abs(Math.floor(oCurrentLineOfSight.x)) +
				Math.abs(Math.floor(oCurrentLineOfSight.y)) +
				Math.abs(Math.floor(oCurrentLineOfSight.z))) != 1.0) {

				Debug.out(this.getClass().getName(),
					  "the new Line of Sight: " + oCurrentLineOfSight +
					  " is not valid. won't rotate.",
					  Debug.LEVEL_WARNING);
				m_bIsInitialized= true;
				return;
			}

			MovementMessage oInitialMove =new MovementMessage("rotate","left");
			m_oMoves.setRotateX(Math.PI / (2));
			m_oMoves.setRotateY(Math.PI / (2));
			m_oMoves.setRotateZ(Math.PI / (2));
			Vector3d oNewVector = m_oInitialLineOfSight;

			while (!bVectorReached) {
				oNewVector = getLeftVector(oNewVector);
				//Debug.out(this.getClass().getName(),
				//	  "init: " + m_oInitialLineOfSight +
				//	  " new: " +oNewVector + "end: " + oCurrentLineOfSight);

				m_oMoves.setLineOfSightRotation(
							m_oInitialLineOfSight,
							oNewVector,
							m_oMoves.LEFT_ROTATION);

				m_oMoves.processMessage(oInitialMove);
				bVectorReached = oNewVector.equals(oCurrentLineOfSight);
			}
		m_bIsInitialized=true;
		m_oCurrentLineOfSight=oCurrentLineOfSight;
	    m_oRealLineOfSight=oCurrentLineOfSight;
	} else {
			m_oCurrentLineOfSight=oCurrentLineOfSight;
			if (m_oRealLineOfSight==null)
				m_oRealLineOfSight=oCurrentLineOfSight;
	}
    }

    /**
     * get invincibility status
     *
     * @return <b>true</b> if pacman is incincible <b>false</b> otherwise
     */
    public boolean isInvincible () {
	return m_oMaintenance.isInvincible();
    }

    /**
     * this function will create the group holding PacMan
     *
     * @param sPacmanModel the model to load
     *
     * @return the TransformGroup containing the pacman
     *
     */
    private TransformGroup createPacMan(String sPacmanModel) {
	m_oPacmanGroup=null;
	m_oSW = new Switch();
	m_oSW.setCapability(Switch.ALLOW_SWITCH_READ);
	m_oSW.setCapability(Switch.ALLOW_SWITCH_WRITE);
	m_oSW.setCapability(Switch.ALLOW_CHILDREN_READ);
	m_oSW.setCapability(Switch.ALLOW_CHILDREN_WRITE);
	m_oSW.addChild(new PacmanObjTrans("light"));
	Debug.out(this.getClass().getName(),"createPacMan: created swith");

	m_oPacmanModels = new Vector();

	if (sPacmanModel.equalsIgnoreCase("PacImage")) {
	    TransformGroup oModel1 = new PacImage ();
	    m_oPacmanModels.add(oModel1);
	    m_oSW.addChild(oModel1);
	} else if (sPacmanModel.equalsIgnoreCase("PacImageVersion2")) {
	    TransformGroup oModel2 =
				new pacimage2version(new Color3f(0.7f, 0.5f, 0.0f));
					m_oPacmanModels.add(oModel2);
					m_oSW.addChild(oModel2);
	} else if(sPacmanModel.equalsIgnoreCase("standard")) {
	    TransformGroup oModel3 = new PacmanObjTrans();
	    m_oPacmanModels.add(oModel3);
	    m_oSW.addChild(oModel3 );
	} else {
	    TransformGroup oModel4 = new PacmanObjTrans(sPacmanModel);
	    m_oPacmanModels.add(oModel4);
	    m_oSW.addChild(oModel4);
	}
	m_oSW.setWhichChild(1);
	Debug.out(this.getClass().getName(),"createPacMan: added Models: " + m_oPacmanModels.size());

	TransformGroup oTG = new TransformGroup ();
	oTG.setCapability(TransformGroup.ALLOW_TRANSFORM_READ);
	oTG.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
	oTG.addChild(m_oSW);
	try {
		Vector oSoundVector = new Vector();
		oSoundVector.add(new URL("file:./media/sound/ALAdmin.wav"));
		oSoundVector.add(new URL("file:./media/sound/flesh1.wav"));
		oSoundVector.add(new URL("file:./media/sound/flesh2.wav"));
		oSoundVector.add(new URL("file:./media/sound/s_health.au"));
		oSoundVector.add(new URL("file:./media/sound/energy.au"));
	    m_oSound = new PacmanSound(oSoundVector);
	    oTG.addChild(m_oSound);
	}
	catch (Exception ex) {
		Debug.out(this.getClass().getName(),
		      "Couldn't add sound",
		      Debug.LEVEL_WARNING);

		Debug.out(this.getClass().getName(),ex,Debug.LEVEL_WARNING);
	}
	Debug.out(this.getClass().getName(),"createPacMan: added sound");

	Debug.out(this.getClass().getName(),"createPacMan: final tg: " + oTG);
	return oTG;
    }

    /**
     *
     * @return the ID of this pacman
     */
    public ID getID() {
	return m_oID;
    }

    /**
     * this function will create a model and return it for the intro
     * @param sPacmanModel model to be shown
     * @return TransformGroup containing the selected pacman
     */
    public TransformGroup getModelPreview (String sPacmanModel) {
	return createPacMan(sPacmanModel);
    }

    /**
     * get vector of all available models
     * @return vector of all available models
     */
    public static Vector getModelList() {
		if (m_oModelDescriptions==null || m_oModels==null) {
			m_oModelDescriptions = new Vector();
			m_oModels = new Vector();
			File oFile;
			//m_oModelDescriptions.add("PacImage");
			m_oModels.add("PacImageVersion2");
			m_oModelDescriptions.add("J3D Primitiv Pacman");
			try {
				oFile = new File("./models/pacman/body.wrl");
				if (oFile.canRead()) {
					m_oModels.add("standard");
					m_oModelDescriptions.add("Standard Pacman");
				}
				oFile = new File("./models/pacman/body_blue.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_blue.wrl");
					m_oModelDescriptions.add("Standard Pacman (blue)");
				}
				oFile = new File("./models/pacman/body_red.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_red.wrl");
					m_oModelDescriptions.add("Standard Pacman (red)");
				}
				oFile = new File("./models/pacman/body_green.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_green.wrl");
					m_oModelDescriptions.add("Standard Pacman (green)");
				}
				oFile = new File("./models/pacman/body_s.wrl");
				if (oFile.canRead()) {
					//m_oModels.add("./models/pacman/body_low.wrl");
					m_oModels.add("./models/pacman/body_s.wrl");
					m_oModelDescriptions.add("Samurai Pacman");
				}
				oFile = new File("./models/pacman/body_s_blue.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_s_blue.wrl");
					m_oModelDescriptions.add("Samurai Pacman (blue)");
				}
				oFile = new File("./models/pacman/body_s_green.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_s_green.wrl");
					m_oModelDescriptions.add("Samurai Pacman (green)");
				}
				oFile = new File("./models/pacman/body_s_red.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_s_red.wrl");
					m_oModelDescriptions.add("Samurai Pacman (red)");
				}
				oFile = new File("./models/pacman/body_s_low.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_s_low.wrl");
					m_oModelDescriptions.add("Samurai Pacman (lowres)");
				}
				oFile = new File("./models/pacman/body_m.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_m.wrl");
					m_oModelDescriptions.add("Mafia Pacman");
				}
				oFile = new File("./models/pacman/body_m_blue.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_m_blue.wrl");
					m_oModelDescriptions.add("Mafia Pacman (blue)");
				}
				oFile = new File("./models/pacman/body_m_green.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_m_green.wrl");
					m_oModelDescriptions.add("Mafia Pacman (green)");
				}
				oFile = new File("./models/pacman/body_m_red.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_m_red.wrl");
					m_oModelDescriptions.add("Mafia Pacman (red)");
				}
				oFile = new File("./models/pacman/body_m_low.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_m_low.wrl");
					m_oModelDescriptions.add("Mafia Pacman (lowres)");
				}
				oFile = new File("./models/pacman/body_b.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_b.wrl");
					m_oModelDescriptions.add("Female Pacman");
				}
				oFile = new File("./models/pacman/body_b_blue.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_b_blue.wrl");
					m_oModelDescriptions.add("Female Pacman (blue)");
				}
				oFile = new File("./models/pacman/body_b_green.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_b_green.wrl");
					m_oModelDescriptions.add("Female Pacman (green)");
				}
				oFile = new File("./models/pacman/body_b_red.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_b_red.wrl");
					m_oModelDescriptions.add("Female Pacman (red)");
				}
				oFile = new File("./models/pacman/body_b_low.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_b_low.wrl");
					m_oModelDescriptions.add("Female Pacman (lowres)");
				}
				oFile = new File("./models/pacman/body_k.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_k.wrl");
					m_oModelDescriptions.add("Army Pacman");
				}
				oFile = new File("./models/pacman/body_k_blue.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_k_blue.wrl");
					m_oModelDescriptions.add("Army Pacman (blue)");
				}
				oFile = new File("./models/pacman/body_k_green.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_k_green.wrl");
					m_oModelDescriptions.add("Army Pacman (green)");
				}
				oFile = new File("./models/pacman/body_k_red.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_k_red.wrl");
					m_oModelDescriptions.add("Army Pacman (red)");
				}
				oFile = new File("./models/pacman/body_k_low.wrl");
				if (oFile.canRead()) {
					m_oModels.add("./models/pacman/body_k_low.wrl");
					m_oModelDescriptions.add("Army Pacman (lowres)");
				}
				oFile = new File("./models/pacman/Moriya/kopf.wrl");
				if (oFile.canRead()) {
					m_oModels.add("Moriya");
				    m_oModelDescriptions.add("Moriya");
				}

			}
			catch (Exception ex) {
				Debug.out("Pacman","Error while initializing models:");
				Debug.out("Pacman",ex);
			}

			return m_oModelDescriptions;
		}else {
			return m_oModelDescriptions;
		}
    }

    /**
     * <p>right now this code won't do anything good ... later it will be
     * possible to initialize PacMan with parameters given in XML format</p>
     *
     * @param oElement      xml-element containing initparameters
     */
    public void loadFromDOM( org.w3c.dom.Element oElement ) {
		if (oElement != null) {
			//first read the metadates
			// ... on the other hand ... why the heck should i care :)

			// so i better start reading the messages
			Element oModel = (Element)(oElement.getElementsByTagName("meta").item(0));
			String sModel = oModel.getAttribute("pacmanmodel");
			Debug.out(this.getClass().getName(),"model to select: " + sModel);
			m_sSelectedModel = sModel;

			// sadly i can't select another model here ... dunno why ... so it
			// will be better, to let the level decide which model to load ...

				//this.createPacMan(sModel);


			NodeList oMessageList = oElement.getElementsByTagName("message");
			this.setDemoMode(true);

			if (oMessageList.getLength() > 0){
				for (int i = 0; i < oMessageList.getLength(); i++) {
					Element oMessage = (Element)(oMessageList.item(i));
					if (oMessage != null) {
						// gotta add message to current message list

						String sMessageType = oMessage.getAttribute("type");
						String sMessageValue = oMessage.getAttribute("value");

						if (sMessageType.equalsIgnoreCase("move") ||
							sMessageType.equalsIgnoreCase("rotate")) {
								this.getMessage(new Message(m_oID,
							    new MovementMessage(sMessageType,
								    sMessageValue,false),m_oID));
						} else {
							this.getMessage(new Message(m_oID,
							    sMessageType,m_oID));
						}
						Debug.out(this.getClass().getName(), "found: type: " + sMessageType +
							      " value: " + sMessageValue);
					}
				}
				m_oMadeMoves = m_oMassageQueue;
			}
		} else {
			Debug.out(this.getClass().getName(), "the given demofile contained no entries.");
		}

    }

    /**
     * <p><b>not yet implemented</b></p>
     *
     * @param oDoc  the document to be saved to
     * @return      the xml-node to be saved
     */
    public org.w3c.dom.Element saveToDOM(org.w3c.dom.Document oDoc) {
		Element oDemo = oDoc.createElement("demomode");
		Element oMeta = oDoc.createElement("meta");
		oMeta.setAttribute("pacmanmodel", this.m_sSelectedModel);
		oMeta.setAttribute("author",m_oMaintenance.getName());
		oMeta.setAttribute("recorddate",new Date().toString());
		oMeta.setAttribute("title",
			"dunno ... maybe this attribut isn't needed after all");
		Element oMessages = oDoc.createElement("messages");
		for (int i = 0; i < m_oMadeMoves.size(); i++) {
			Object oContent = ((Message) m_oMadeMoves.get(i)).getContent();
			Element oMessage =  oDoc.createElement("message");
			if (oContent instanceof MovementMessage) {
				oMessage.setAttribute("type",
					((MovementMessage) oContent).getAction());
				oMessage.setAttribute("value",
					((MovementMessage) oContent).getDirection());
				oMessages.appendChild(oMessage);
			} else if (oContent instanceof String) {
				oMessage.setAttribute("type",
					((String) oContent));
				oMessage.setAttribute("value",
					" ");
				oMessages.appendChild(oMessage);
			}
		}
		oDemo.appendChild(oMeta);
		oDemo.appendChild(oMessages);
	    return oDemo;
    }

    /**
     * gets the next position
     *
     * @param ocurrentPos current position of pacman
     * @param sDirection the direction in which the next position shall
     *        be calculated
     * @return the next cell in the given direction
     */
    private Point3i getNextPosition(Point3i ocurrentPos,String sDirection) {

		Point3i oNextPoint = new Point3i(ocurrentPos);
	if(sDirection.equalsIgnoreCase("forward")) {
	    oNextPoint.x=ocurrentPos.x + (int)m_oCurrentLineOfSight.x;
	    oNextPoint.y=ocurrentPos.y + (int)m_oCurrentLineOfSight.y;
	    oNextPoint.z=ocurrentPos.z + (int)m_oCurrentLineOfSight.z;
	} else if(sDirection.equalsIgnoreCase("back")) {
	    oNextPoint.x=ocurrentPos.x - (int)m_oCurrentLineOfSight.x;
	    oNextPoint.y=ocurrentPos.y - (int)m_oCurrentLineOfSight.y;
	    oNextPoint.z=ocurrentPos.z - (int)m_oCurrentLineOfSight.z;
	} else if(sDirection.equalsIgnoreCase("left")) {
	    oNextPoint.x=ocurrentPos.x + (int)m_oCurrentLineOfSight.z;
	    oNextPoint.y=ocurrentPos.y + (int)m_oCurrentLineOfSight.y;
	    oNextPoint.z=ocurrentPos.z - (int)m_oCurrentLineOfSight.x;
	} else if(sDirection.equalsIgnoreCase("right")) {
	    oNextPoint.x=ocurrentPos.x - (int)m_oCurrentLineOfSight.z;
	    oNextPoint.y=ocurrentPos.y - (int)m_oCurrentLineOfSight.y;
	    oNextPoint.z=ocurrentPos.z + (int)m_oCurrentLineOfSight.x;
	} else if(sDirection.equalsIgnoreCase("up")) {
	    oNextPoint.x=ocurrentPos.x ;
	    oNextPoint.y=ocurrentPos.y +1;
	    oNextPoint.z=ocurrentPos.z ;
	} else if(sDirection.equalsIgnoreCase("down")) {
	    oNextPoint.x=ocurrentPos.x ;
	    oNextPoint.y=ocurrentPos.y -1;
	    oNextPoint.z=ocurrentPos.z ;
	}
	return oNextPoint;
    }
    /**
     * returns the new line of sight ...
     *
     * @param ocurrentLineOfSight the current line of sight
     * @param sDirection the direction, where to move to
     *
     * @return the next line of sight
     */
    private Vector3d getNextLineOfSight(Vector3d ocurrentLineOfSight,String sDirection) {
	Vector3d oNextLineOfSight=new Vector3d();

	if(sDirection.equalsIgnoreCase("up")) {
	    // sorry ... no imagination :)
	    oNextLineOfSight=ocurrentLineOfSight;
	} else if(sDirection.equalsIgnoreCase("down")) {
	    // sorry ... no imagination :)
	    oNextLineOfSight=ocurrentLineOfSight;
	} else if(sDirection.equalsIgnoreCase("left")) {
	    oNextLineOfSight=getLeftVector(ocurrentLineOfSight);
	} else if(sDirection.equalsIgnoreCase("right")) {
	    oNextLineOfSight=getRightVector(ocurrentLineOfSight);
	}
	return oNextLineOfSight;
    }
    /**
     * this function returns to a given vector a vector
     * containing a left rotation
     *
     * @param oVec the current vector
     * @return the next vector (left of current)
     */
    private Vector3d getLeftVector (Vector3d oVec) {
	Vector3d oLeft=new Vector3d();
	if (oVec.x==1 && oVec.y==0 && oVec.z==0) {
	    oLeft  = new Vector3d(0,0,-1);
	}else if (oVec.x==0 && oVec.y==0 && oVec.z==-1) {
	    oLeft  = new Vector3d(-1,0,0);
	}else if (oVec.x==-1 && oVec.y==0 && oVec.z==0) {
	    oLeft  = new Vector3d(0,0,1);
	}else if (oVec.x==0 && oVec.y==0 && oVec.z==1) {
	    oLeft  = new Vector3d(1,0,0);
	}
	return oLeft;
    }
    /**
     * this function returns to a given vector a vector
     * containing a right rotation
     *
     * @param oVec the current vector
     * @return the next vector (right of current)
     */
    private Vector3d getRightVector (Vector3d oVec) {
	Vector3d oRight=new Vector3d();
	if (oVec.x==1 && oVec.y==0 && oVec.z==0) {
	    oRight  = new Vector3d(0,0,1);
	}else if (oVec.x==0 && oVec.y==0 && oVec.z==1) {
	    oRight  = new Vector3d(-1,0,0);
	}else if (oVec.x==-1 && oVec.y==0 && oVec.z==0) {
	    oRight  = new Vector3d(0,0,-1);
	}else if (oVec.x==0 && oVec.y==0 && oVec.z==-1) {
	    oRight  = new Vector3d(1,0,0);
	}
	return oRight;
    }

    /** this function allows to set any parameter available to the pacman
     *
     * @param sParametername the parameter to be set
     * @param oParametervalue the value to be set
     */
    public void setParameter ( String sParametername, Object oParametervalue) {
		if (sParametername.equalsIgnoreCase("score")) {
			if (oParametervalue instanceof Integer) {
			m_oMaintenance.setPoints((Integer)oParametervalue);
			} else if (oParametervalue instanceof ItemValue) {
			m_oMaintenance.setPoints(
						 (Integer)((ItemValue)oParametervalue).getValue());
			}
		} else if (sParametername.equalsIgnoreCase("lives")) {
			if (oParametervalue instanceof Integer) {
			m_oMaintenance.addLives((Integer)oParametervalue);
			} else if (oParametervalue instanceof ItemValue) {
			m_oMaintenance.addLives(
						(Integer)((ItemValue)oParametervalue).getValue());
			}
		} else if (sParametername.equalsIgnoreCase("invincible")) {
			if (oParametervalue instanceof Integer) {
				Debug.out(this.getClass().getName(),"enabling invicibility for: "
					+ ((Integer)oParametervalue).intValue());
			m_oMaintenance.setInvincible(((Integer)oParametervalue).intValue());
			} else if (oParametervalue instanceof ItemValue) {
				Debug.out(this.getClass().getName(),"enabling invicibility for: "
					+ ((Integer)((ItemValue)oParametervalue).getValue()).intValue());
			m_oMaintenance.setInvincible(
							 ((Integer)((ItemValue)oParametervalue).getValue()).intValue());
			}

		} else if (sParametername.equalsIgnoreCase("name")) {
			m_oMaintenance.setName((String)oParametervalue);
		} else if (sParametername.equalsIgnoreCase("friendly")) {
			m_oMaintenance.setFriendly(((Boolean)oParametervalue).booleanValue());
		} else if (sParametername.equalsIgnoreCase("model")) {
			createPacMan((String)oParametervalue);
		}
    }
    /** returns any parameter
     *
     * @param sParametername the parameter to be returned
     * @return the parametervalue that was asked vor
     */
    public Object getParameter (String sParametername){
	Object oResult=new Object();
	if (sParametername.equalsIgnoreCase("score")) {
	    return m_oMaintenance.getPoints();
	}else if (sParametername.equalsIgnoreCase("name")) {
	    return m_oMaintenance.getName();
	}else if (sParametername.equalsIgnoreCase("friendly")) {
	    return new Boolean(m_oMaintenance.isFriendly());
	}else if (sParametername.equalsIgnoreCase("modelname")) {
	    return m_sSelectedModel;
	}
	return oResult;
    }
    /** returns number of parameters available for the pacman
     *
     * @return number of parameters available for the pacman
     */
    public int getParameterCount () {
	return m_oParameterNames.size();
    }
    /** returns name of parameter at the given position
     *
     * @param nPosition the position of the parameter
     * @return the name of the parameter
     */
    public String getParameterName (int nPosition) {
	return (String)m_oParameterNames.get(nPosition);
    }
    /** returns the type of the parameter at the given position
     *
     * @param nPosition the position of the parameter
     * @return the parameter type of that parameter
     */
    public Object getParameterType (int nPosition) {
	return m_oParameterTypes.get(nPosition);
    }
    /** this method will set the number of ticks needed to complete a Movement
     *  operation
     *
     * @param nCost the new cost of a movement operation
     */
    public void setMovementCost (int nCost) {
	m_nMovementCost=nCost;
    }
    /** this method will set the number of ticks needed to complete a Rotate
     *  operation
     *
     * @param nCost the new cost of a rotation operation
     */

    public void setRotationCost (int nCost) {
	m_nRotationCost=nCost;
    }

    /** this function returns the description for the given parameter
     *
     * @param nPos the position of the parameter
     * @return the description of the parameter
     */
    public String getParameterDescription( int nPos ) {
	return (String)m_oParameterNames.get(nPos);
    }

    /**
     * this function checks, whether or not pacman stands over a chasm
     * if so, it'll let him fall
     */
    private void doFall() {
	Point3i oTemp = m_oLabyrinth.getPacmanPos(m_oID);
	boolean bWalkable = m_oLabyrinth.getCell(
						 oTemp.x,
						 oTemp.y-1,
						 oTemp.z).isWalkable();
	if (bWalkable && m_oMassageQueue.isEmpty()) {
	    Message oMessage = new Message (
					    m_oID,new FallingContent(
								     new MovementMessage("move","down",false)),
					    m_oID);
	    m_oMassageQueue.insertElementAt(oMessage,0);
	}

    }

    /** sends the Message to the PostOffice (Labyrinth)
     *
     * @param oMessage message to be send
     */
    private void sendMessage (Message oMessage) {

		if (m_oMessageService != null) {

			try {

			m_oMessageService.sendMessage(oMessage);

			}
			catch (Exception ex) {

			Debug.out(this.getClass().getName(),
				  "Error while sending Message: " + oMessage ,Debug.LEVEL_NOTICE );
			Debug.out(this.getClass().getName(),ex);

			}
		}
    }

    /** sends the Message to the PostOffice (Labyrinth)
     *
     * @param oMessage message to be send
     */
    private void sendLocalMessage (Message oMessage) {

	if (m_oMessageService != null) {

	    try {

		m_oMessageService.sendMessage(oMessage,
					m_oMessageService.PROPAGATE_LOCAL);

	    }
	    catch (Exception ex) {

		Debug.out(this.getClass().getName(),
			  "Error while sending Message: " + oMessage ,Debug.LEVEL_NOTICE );
		Debug.out(this.getClass().getName(),ex);

	    }
	}
    }

    /**
     * this function wil examine the given cell. If there are any takable items
     * in it ... pacman will take them all. If there are non-takable items
     * pacman will just execute them right there
     *
     * @param oCell Cell to be examined
     */
    private void doTakeFrom(Cell oCell) {
		if (oCell.hasItems()) {

			Vector oTemp = oCell.getItemsVector();
			Iterator oTempIt = oTemp.iterator();
			while (oTempIt.hasNext()) {

				Item oItem = (Item) oTempIt.next();

				if (oItem.isVisible()) {
					if (oItem.isTakable()) {
						m_oMaintenance.addItem(oItem);
						oItem.take(this);
						sendInventoryMessage();
						m_oSound.playSound(3);

					} else {

						oItem.execute(this);
						sendScoreMessage();
						m_oSound.playSound(4);

					}
				}
			}
		}
    }

    /**
     * sends the current score to the score-panel
     */
    public void sendScoreMessage() {
	if (m_oScoreID == null)  {
		if (m_oLabyrinth != null)
		    m_oScoreID=m_oLabyrinth.getPanelScoreID();
	}
	if (m_oScoreID != null) {

	    sendLocalMessage(new Message (m_oID,
				     new ScoreUpdateContent(m_oMaintenance.getPoints(),
					 m_oMaintenance.getLives(),m_oMaintenance.getKillCount()),
				     m_oScoreID));

	}
    }
    /**
     * sends the current inventory state to the inventory-panel
     */
    public void sendInventoryMessage() {
	if (m_oInventoryID == null) m_oInventoryID=m_oLabyrinth.getPanelInventoryID();
	if (m_oInventoryID != null) {
			sendLocalMessage(new Message (m_oID,
			    m_oMaintenance.getMessage(),m_oInventoryID));
	}
    }

    /**
     * this function will convert a vector3d or Point3i to a
     * point3d
     *
     * @param oTemp the object to be converted
     * @return the converted Point3d
     */
    private Point3d convertToPoint(Object oTemp) {
	if (oTemp instanceof Vector3d)
	    return ( new Point3d(((Vector3d)oTemp).x,
				 ((Vector3d)oTemp).y,
				 ((Vector3d)oTemp).z));
	else if (oTemp instanceof Point3i)
	    return ( new Point3d(((Point3i)oTemp).x,
				 ((Point3i)oTemp).y,
				 ((Point3i)oTemp).z));
	Debug.out(this.getClass().getName(),
		  "oups ... convertion failed",
		  Debug.LEVEL_WARNING);
	return null;
    }
    /**
     * in order to make the makeLoop method more readable i thought it a good
     * idea to source out the code for actually rotating the pacman. that is
     * done in this method.
     *
     * @param oMove the movement-message to resume
     */
    private void handleRotation(MovementMessage oMove) {

	if (m_nNumberLeft == 1) {
	    Vector3d oNewLineOfSight =
		getNextLineOfSight( this.getLineOfSight(),
				    oMove.getDirection());
	    this.setLineOfSight(oNewLineOfSight);
	    m_oMoves.processMessage(oMove);
	    m_oRealLineOfSight=m_oMoves.getEndLineOfSight();
	    m_oCurrentLineOfSight=m_oMoves.getEndLineOfSight();
	    m_oCurrentOp = null;
	    m_oMove = null;
	} else {
	    m_oMoves.processMessage(oMove);
	    m_oRealLineOfSight=m_oMoves.getLineOfSight();
	}
	m_nNumberLeft--;
    }
    /**
     * this method will resume the current movement operation. Instead of
     * walking along the pre-calculated positions like before this version will
     * calculate the position to walk to just before moving. Whether this will
     * improve the speed of pacman or rather smoothen the walk will be seen (-.-')
     *
     * @param oMove currewnt movement operation
     * @param nTickAmount ticks we've got to spend
     * @return the ticks spend
     */
    private int handleMovement(MovementMessage oMove, int nTickAmount) {

	int nTicksUsed = 0;
    //Debug.out(this.getClass().getName(),"handleMove: " + oMove + " master: " + m_bIsMaster+
	//	" my ID equals ActivePacmanID?: " + (this.getID().equals(m_oLabyrinth.getActivePacmanID())));
	/* calculate next position where the pacman should be*/
	int nMove = Math.min(m_nNumberLeft,nTickAmount);
	m_nNumberLeft-=nMove;
	nTicksUsed+=nMove;

	Vector3d oCalculatedNextPos = new Vector3d();
	oCalculatedNextPos.interpolate(m_oStartPostition,
				       convertToPoint(m_oEndPosition),
				       ((double)((double)(m_nMovementCost-m_nNumberLeft)*(1./(double)m_nMovementCost))));

	//Debug.out(this.getClass().getName(),"new position is: " + oCalculatedNextPos);

	// next cell entered
	if (m_nNumberLeft <= m_nStep) {
	    m_oCurrentPos = m_oEndPosition;
	    m_oSound.playSound(2);
	    try {
		//Debug.out(this.getClass().getName(),"giving new position to lab");
		m_oLabyrinth.setPacmanPos(m_oID,m_oEndPosition);

	    } catch (Exception ex) {

		Debug.out(this.getClass().getName(),
			  ex,
			  Debug.LEVEL_CRITICAL );
	    }



	}

	m_oRealPos = convertToPoint(oCalculatedNextPos);
	movePacman(m_oRealPos);
	if (m_nNumberLeft  <=0) {
	    m_oCurrentOp = null;
	    m_oMove = null;
	    m_oStartPostition=m_oRealPos;
            /* since we now officially entered the cell we can
	       look after the items *smile smile* */

	    Cell oCell = m_oLabyrinth.getCell(m_oCurrentPos);
	    if (m_bIsMaster) doTakeFrom(oCell);


	    /** well if we are at it ... we could look, whether
	     *  or not to kill a monster or two :)
	     */
	    if (m_bIsMaster) slayMonster();
	}

	/**
	 *  in the case that pacman is influenced by
	 *  gravity this last check will create a downward
	 *  movement if the user hasn't already chose another
	 *  op.
	 */

	if (m_bIgnoreGravity == false) {
	    doFall();
	}
	return nTicksUsed;
    }
    /**
     * this function will try to kill a monster / pacman at the current
     * position.
     */
    private void slayMonster(){

		if (m_oMaintenance.isInvincible()) {

			/**
			 * first I'm looking for any monsters at the position of this
			 * pacman. if any is found pacman tries to kill it by sending
			 * a DEATH-message
			 */

			ID[] aAllMonsters = m_oLabyrinth.getAllMonsterIds();
			for (int i = 0; i < aAllMonsters.length; i++) {
				if (m_oCurrentPos.equals(
							 m_oLabyrinth.getMonsterPos(aAllMonsters[i]))) {
					sendMessage(new Message (m_oID,
								 "die",
								 aAllMonsters[i]));
					//Debug.out(this.getClass().getName(),
					//      "killed monster");
					m_oMaintenance.setPoints(new Integer(500));
					sendScoreMessage();

				}

			}
			/**
			 *  if pacman is not in coop - mode we've got to try to kill
			 *  other pacman as well
			 */
			if (!(m_oMaintenance.isFriendly())) {

			    ID[] aAllPacmen = m_oLabyrinth.getAllPacmanIds();
				for (int i = 0; i < aAllPacmen.length; i++) {
					if ((m_oID != aAllPacmen[i]) &&
					    (m_oCurrentPos.equals(
								  m_oLabyrinth.getPacmanPos(aAllPacmen[i])))) {

					    sendMessage(new Message (m_oID,
								 "die",
								 aAllPacmen[i]));
						//Debug.out(this.getClass().getName(),
						//	  "killed pacman");
						m_oMaintenance.setPoints(new Integer(500));
						sendScoreMessage();
					}
				}
			}
		}
    }
	/**
	 * this functions shows the death animation sequence
	 */
	private void showDeathAnimation () {
		Iterator oIT = m_oPacmanModels.iterator();
		while (oIT.hasNext()) {
			Object oTempModel = oIT.next();
			if (oTempModel instanceof PacmanObjTrans) {
				((PacmanObjTrans)oTempModel).showDeathAnimation();
			} else if (oTempModel instanceof PacImage) {
				((PacImage)oTempModel).showDeathAnimation();
			} else if (oTempModel instanceof pacimage2version) {
				((pacimage2version)oTempModel).showDeathAnimation();
			}
		}
	}
	/**
	 * en- or disables the animation sequence that will allow the pacman
	 * to swing his katana.
	 *
	 * @param bStartAnimation   if <b>true</b> animation sequence is started
	 *                          if <b>false</b> sequence is stopped
	 */
	public void enableFightAnimation (boolean bStartAnimation) {
		Iterator oIT = m_oPacmanModels.iterator();
		while (oIT.hasNext()) {
			Object oTempModel = oIT.next();
			if (oTempModel instanceof PacmanObjTrans) {
				((PacmanObjTrans)oTempModel).enableInvincibility(bStartAnimation);
			}
		}
	}
	/**
	 *  set the new position after dyin'
	 */
	private void setNewStartPoint() {
		Position newPos =
			m_oLabyrinth.getNextPacmanStartPosition(
							m_oID);

		try {
		m_oLabyrinth.setPacmanPos(m_oID,
					  newPos.getPosition());
		}
		catch (Exception ex) {
			Debug.out(this.getClass().getName(),
			  "could not set position to: "
			  + newPos.getPosition(),
			  Debug.LEVEL_CRITICAL);
		}
		m_oCurrentPos =newPos.getPosition();
		m_oRealPos=convertToPoint(newPos.getPosition());

		movePacman(m_oRealPos);
		setPacmanPos(m_oCurrentPos);

		m_oCurrentOp=null;
		m_oMove=null;

		m_oInitialLineOfSight = m_oCurrentLineOfSight;
		m_bIsInitialized = false;
		setLineOfSight(newPos.getLineOfSight());
		m_oMassageQueue = new Vector();

	}
	/** sends the score message to all games */
	private void sendGameScoreMessage() {
		if (m_oScoreID != null) {

			sendMessage(new Message (m_oID,
						 new ScoreUpdateContent(m_oMaintenance.getPoints(),
						 m_oMaintenance.getLives(),m_oMaintenance.getKillCount()),
						 Message.RECEIVER_ALL_GAMES));

		}
	}
	/**
	 * with this method it is possible to influence the mode with which
	 * messages are handled.
	 *
	 * @param bIsDemoMode   if <b>true</b> all incoming messages are accepted,
	 *                      if <b>false</b> only new messages are accepted
	 */
	public void setDemoMode(boolean bIsDemoMode) {
		m_bIsDemoMode = bIsDemoMode;
	}
}
