package pacman3d.pacman;
import javax.media.j3d.*;
import javax.vecmath.*;
import java.util.Vector;
import pacman3d.message.*;
import pacman3d.util.*;

/**
 * Title:        Praktikum Java3D und VRML<br>
 * Description:  <p>this class will privide basic movement operations like
 *               <ul>
 *               <li> rotateLeft </li>
 *               <li> rotateRight</li>
 *               <li> rotateUp</li>
 *               <li> rotateDown</li>
 *               </ul>
 *               </p>
 *               <p>an update was made, 'cause the concept so far wasn't
 *               compatible with the "LineOfSight" and "Tick" concept</p>
 * last update:  11.12.2001<br>
 * @author <a href="mailto:bergmann@cs.uni-frankfurt.de">Frank Bergmann </a>
 * @version 1.0
 */

public class PacManMovement {

	/** constant indicating that pacman will rotate left */
    public final static boolean LEFT_ROTATION = false;
	/** constant indicating that pacman will rotate right */
    public final static boolean RIGHT_ROTATION = true;

    /** TransformGroup do be changed */
    private TransformGroup m_oPacman;
    /** the transformation desired */
    private Transform3D m_oTransform3D;

    private final static double TWO_PI = (2.0 * Math.PI);
    /** X Rotation amount */
    private double m_dRotateX = Math.PI / 2.0;
    /** Y Rotation amount */
    private double m_dRotateY = Math.PI / 2.0;
    /** ZRotation amount */
    private double m_dRotateZ = Math.PI / 2.0;

    /** define standard movement rate */
    private double m_dMoveRate = 1.0;
    /** set standard speed factor */
    private double m_dSpeed = NORMAL_SPEED;

    /** start position for line of sight */
    private Vector3d m_oStart;
    /** end position for line of sight */
    private Vector3d m_oEnd;
	/** is there a right rotation? */
	private boolean m_bDirection;

    protected static final double FAST_SPEED = 2.0;
    protected static final double NORMAL_SPEED = 1.0;
    protected static final double SLOW_SPEED = 0.5;

    /** constructor initializing the class with the TransformGroup containing
     *  pacman
	 *
	 *  @param oPacman the pacman this class is for
	 */
    public PacManMovement(TransformGroup oPacman) {
        m_oPacman=oPacman;
        m_oTransform3D = new Transform3D();
    }

    /** this function moves Pacman to the desired location and resizes it
	 *
	 * @param oInitPos initial position of pacman
	 * @param dScale scale pacman while loading
	 */
    public void initialTransformation (Vector3d oInitPos, double dScale) {
        m_oPacman.getTransform(m_oTransform3D);
        m_oTransform3D.setTranslation(oInitPos);
        m_oTransform3D.setScale(dScale);
        m_oPacman.setTransform(m_oTransform3D);
    }
	/**
	 * set the position of pacman
	 * @param oPos the position, where the pacman should be
	 */
    public void setPosition (Vector3d oPos) {
        m_oPacman.getTransform(m_oTransform3D);
        m_oTransform3D.setTranslation(oPos);
        m_oPacman.setTransform(m_oTransform3D);
    }


	/**
	 * rotate right
	 */
    public  void rotRight() {
        doRotateY(getRotateRight());
    }

	/**
	 * rotate up - not used for this version of pacman
	 */
    public  void rotUp() {
        doRotateX(getRotateUp());
    }

	/**
	 * rotate left
	 */
    public  void rotLeft() {
        doRotateY(getRotateLeft());
    }

	/**
	 * rotate down - not used for this version of pacman
	 */
    public  void rotDown() {
        doRotateX(getRotateDown());
    }


    /** rotate TG in Y direction
	 *
	 * @param dRadians the degree of rotation
	 */
    protected void doRotateY(double dRadians) {
        m_oPacman.getTransform(m_oTransform3D);
        Transform3D toMove = new Transform3D();
        toMove.rotY(dRadians);
        m_oTransform3D.mul(toMove);
        m_oPacman.setTransform(m_oTransform3D);
        updateLineOfSight(dRadians);
    }

    /** rotate TG in X direction
	 *
	 * @param dRadians the degree of rotation
	 */
    protected void doRotateX(double dRadians) {
        m_oPacman.getTransform(m_oTransform3D);
        Transform3D toMove = new Transform3D();
        toMove.rotX(dRadians);
        m_oTransform3D.mul(toMove);
        m_oPacman.setTransform(m_oTransform3D);
        updateLineOfSight(dRadians);
    }

    /** rotate TG in Z direction
	 *
	 * @param dRadians the degree of rotation
	 */
    protected void doRotateZ(double dRadians) {
        m_oPacman.getTransform(m_oTransform3D);
        Transform3D toMove = new Transform3D();
        toMove.rotZ(dRadians);
        m_oTransform3D.mul(toMove);
        m_oPacman.setTransform(m_oTransform3D);
        updateLineOfSight(dRadians);
    }


	/**
	 *
	 * @return the current movement rate
	 */
    protected double getMovementRate() {
        return m_dMoveRate * m_dSpeed;
    }

	/**
	 *
	 * @return the upward rotation speed
	 */
    protected double getRotateUp() {
        return m_dRotateY * m_dSpeed;
    }

	/**
	 *
	 * @return the downward rotation speed
	 */
    protected double getRotateDown() {
        return -m_dRotateY * m_dSpeed;
    }

	/**
	 *
	 * @return the leftward rotation speed
	 */
    protected double getRotateLeft() {
        return m_dRotateY * m_dSpeed ;
    }

	/**
	 *
	 * @return the rotation speed toward the right side
	 */
    protected double getRotateRight() {
        return -m_dRotateY * m_dSpeed;
    }

	/**
	 * set the degrees to rotate
	 * @param dRadians degrees
	 */
    public void setRotateX(double dRadians) {
        m_dRotateX = dRadians;
    }
	/**
	 * set the degrees to rotate
	 * @param dRadians degrees
	 */
    public void setRotateY(double dRadians) {
        m_dRotateY = dRadians;
    }
	/**
	 * set the degrees to rotate
	 * @param dRadians degrees
	 */
    public void setRotateZ(double dRadians) {
        m_dRotateZ = dRadians;
    }

	/**
	 * set the speed
	 * @param dMeters the speed
	 */
    public void setMovementRate(double dMeters) {
        m_dMoveRate = dMeters; // Travel rate in meters/frame
    }
	/**
	 * process the MovementMessage, right now only the rotation is handled by
	 * this function.
	 * @param oMove MovementMessage containing the rotation
	 */
    public void processMessage (MovementMessage oMove) {
        if (oMove.getAction().equalsIgnoreCase("rotate")) {
            if(oMove.getDirection().equalsIgnoreCase("up")) {
                rotUp();
            } else if(oMove.getDirection().equalsIgnoreCase("down")) {
                rotDown();
            } else if(oMove.getDirection().equalsIgnoreCase("left")) {
                rotLeft();
            } else if(oMove.getDirection().equalsIgnoreCase("right")) {
                rotRight();
            }
        }
    }
    /**
     * initializes line of sight calculation
     * @param oStart the line of sight to be started from
     * @param oEnd the desired line of sight
	 * @param bDirection is there a right rotation, or not?
     */
    public void setLineOfSightRotation(Vector3d oStart,
		Vector3d oEnd,boolean bDirection) {
        m_oStart=oStart;
        m_oEnd=oEnd;
		m_bDirection=bDirection;
    }

	/**
	 * this function kinda caused the most problems. the Java3D inbuild
	 * interpolate functions just implements (1-alpha)*this + alpha*t1.
	 * That is fine, if one does want to rotate a Vector toward the left
	 * but if a rotation toward the ride is desired this function fails.
	 * (so we had a rotation of 270 degrees toward the left each time we
	 * wanted a rotation of 90 degrees toward the right, which looked kinda
	 * strange). Finding a new function involved some hard thinking. But
	 * finally i made it. (And proud i am)
	 *
	 * @param   dAmount this is a scalar > 0  and < 1. It is calculated by
	 *          dividing 2 Pi by the number of steps needed to complete a 90
	 *          degree rotation
	 */
    private void updateLineOfSight (double dAmount) {
		if (m_oStart != null && m_oEnd != null) {
			if (m_bDirection==this.RIGHT_ROTATION) {

				//(1-alpha)*this - alpha*t1.
				m_oStart.x=(((1-dAmount)*m_oStart.x) - (dAmount*m_oEnd.x));
				m_oStart.y=(((1-dAmount)*m_oStart.y) - (dAmount*m_oEnd.y));
				m_oStart.z=(((1-dAmount)*m_oStart.z) - (dAmount*m_oEnd.z));
				m_oStart.normalize();

			} else {
				//(1-alpha)*this + alpha*t1.
				m_oStart.x=(((1-dAmount)*m_oStart.x) + (dAmount*m_oEnd.x));
				m_oStart.y=(((1-dAmount)*m_oStart.y) + (dAmount*m_oEnd.y));
				m_oStart.z=(((1-dAmount)*m_oStart.z) + (dAmount*m_oEnd.z));
				m_oStart.normalize();

			}
		}
    }
	/**
	 * this function will return the current calculated line of sight. this
	 * function is used to tell the camera the real vector3d where pacman is
	 * looking too.
	 *
	 * @return current real line of sight
	 */
    public Vector3d getLineOfSight() {
		return m_oStart;

    }
	/**
	 * same as getLineOfSight() only this function only tells about the descete
	 * vectors needed for the labyrinth.
	 *
	 * @return current line of sight
	 */
    public Vector3d getEndLineOfSight() {
		return m_oEnd;
    }
}