package pacman3d.pacman;
import java.awt.event.*;
import pacman3d.util.Debug;
import pacman3d.message.*;
/**
 * Title:        Praktikum Java3D und VRML<br>
 *
 * <b>Description:</b>  <p>this class will work as KeyListener and will allow basic
 *               interaction with PacMan and Camera. <br> If the keys do apply
 *               to PacMan, a message will be send to the labyrinth and hopefully
 *               will be relayed to the pacman allowing him to move the next Tick</p>
 *               <p>this update was made since the massage class provided a constructor
 *               and so sending a message will take less method calls :)</p>
 *
 * Copyright:    Copyright (c) 2001<br>
 *
 * last update: 02.02.2002<br>
 * @author <a href="mailto:bergmann@cs.uni-frankfurt.de">Frank Bergmann </a>
 * @version 2.0
 */

public class KeyControl implements KeyListener {

    /** since this class will send messages it needs to have an ID */
    private ID m_oUID;
    /** ID of the PacMan that will receive the movement messages */
    private ID m_oPacmanID;
    /** ID of the Camera that will receive the messages */
    private ID m_oCameraID;
    /** the instance to send messages to */
    private MessageService m_oMessageService;

    /** default key - forward <br> this key can be changed with setForwardKey()*/
    private int m_nForwardKey = KeyEvent.VK_UP;
    /** default key - back <br> this key can be changed with setBackKey()*/
    private int m_nBackKey = KeyEvent.VK_DOWN;
    /** default key - left <br> this key can be changed with setLeftKey()*/
    private int m_nLeftKey = KeyEvent.VK_LEFT;
    /** default key - right <br> this key can be changed with setRightKey()*/
    private int m_nRightKey = KeyEvent.VK_RIGHT;
    /** default key - F1*/
    private int m_nEgoViewKey = KeyEvent.VK_F1;
    /** default key - F2*/
    private int m_nTrackerViewKey = KeyEvent.VK_F2;
    /** default key - F3 */
    private int m_nBirdViewKey = KeyEvent.VK_F3;
    /** default key - F4 */
    private int m_nIsoViewKey = KeyEvent.VK_F4;
    /** default key - space */
    private int m_nToggleStyleKey = KeyEvent.VK_SPACE;
    /** default key - NumPad 4 */
    private int m_nCameraLeftKey = KeyEvent.VK_NUMPAD4;
    /** default key - NumPad 6 */
    private int m_nCameraRightKey = KeyEvent.VK_NUMPAD6;
    /** default key - NumPad 8 */
    private int m_nCameraUpKey = KeyEvent.VK_NUMPAD8;
    /** default key - NumPad 2 */
    private int m_nCameraDownKey = KeyEvent.VK_NUMPAD2;
    /** default key - NumPad 0 */
    private int m_nCameraResetKey = KeyEvent.VK_NUMPAD0;
    /** default key - NumPad 0 */
    private int m_nCameraZoomInKey = KeyEvent.VK_NUMPAD1;
    /** default key - NumPad 0 */
    private int m_nCameraZoomOutKey = KeyEvent.VK_NUMPAD3;
    /** key to end the game */
    private int m_nEscape = KeyEvent.VK_ESCAPE;
    /** key to force pacman to give status information */
    private int m_nInfoKey = KeyEvent.VK_ENTER;
    /** key to select another model */
    private int m_nNextModel = KeyEvent.VK_C;

    /** key to select the next item */
    private int m_nNextItem = KeyEvent.VK_A;
    /** key to select the last item */
    private int m_nLastItem = KeyEvent.VK_S;
    /** key to use the selected item */
    private int m_nUseItem = KeyEvent.VK_D;
    /** key to kill Pacman instantly ... used for demo purposes only */
    private int m_nKillPacman = KeyEvent.VK_T;

	private int m_nHide = KeyEvent.VK_K;
	private int m_nShow = KeyEvent.VK_L;

    /** boolean deciding whether pacman will run, or not */
    private boolean m_bIsFast;

    /** constructor - initializing the KeyControl
     *
     * @param oUID      the ID of the KeyListener, needed since a message
     *                  needs a senderID
     * @param oPacmanID the ID of the pacman, that shall receive the messages.
     * @param oCamera   the ID of the Camera that shall be controlled by this
     *                  Handler
     */
    public KeyControl(ID oUID,ID oPacmanID, ID oCamera) {
        m_oUID=oUID;
        m_oPacmanID = oPacmanID;
        m_oCameraID = oCamera;
        m_oMessageService = MessageService.getInstance();
    }
    /** this function desides which movement operation is desired
     *
     * @param oEvent the event to be processed
     */
    protected void processKeyEvent(KeyEvent oEvent) {
        int nKeycode = oEvent.getKeyCode();
        //keycodes for camera group

        if(nKeycode==m_nEgoViewKey
            || nKeycode==m_nTrackerViewKey
            || nKeycode==m_nBirdViewKey
            || nKeycode==m_nIsoViewKey
            || nKeycode==m_nToggleStyleKey
            || nKeycode==m_nCameraUpKey
            || nKeycode==m_nCameraDownKey
            || nKeycode==m_nCameraRightKey
            || nKeycode==m_nCameraLeftKey
            || nKeycode==m_nCameraResetKey
            || nKeycode==m_nCameraZoomInKey
            || nKeycode==m_nCameraZoomOutKey ) {

            changeView(nKeycode);
            return;

        } else if (nKeycode == m_nNextModel) {

            sendMessage(new Message (m_oUID,"select next model",m_oPacmanID));

        } else if (nKeycode == m_nHide) {

            sendMessage(new Message (m_oUID,"hidePacman",m_oPacmanID));

        } else if (nKeycode == m_nShow) {

            sendMessage(new Message (m_oUID,"showPacman",m_oPacmanID));

        } else if (nKeycode == m_nKillPacman) {

            sendMessage(new Message (m_oUID,"kill Pacman",m_oPacmanID));

        } else if (nKeycode == m_nEscape) {

            sendMessage(new Message (m_oUID,"endGame",m_oPacmanID));

        } else if  (nKeycode == m_nInfoKey) {

            sendMessage(new Message (m_oUID,"info",m_oPacmanID));

        }
        // if shift is pressed RUN
        if(oEvent.isShiftDown())

            m_bIsFast=true;

        else // else just walk

            m_bIsFast=false;

        if( oEvent.isAltDown() )         // if alt is pressed rotate

            moveRotation(nKeycode);

        else if (oEvent.isControlDown())

            moveUpDown(nKeycode);

        else {                            // else just move

            moveDirection(nKeycode);
            processItemKeys(nKeycode);
        }

    }

    /**
     * this function will send the keys concerning the items to the pacman
     *
     * @param nKeycode the KeyCode
     */
    private void processItemKeys(int nKeycode) {

        if(nKeycode == m_nLastItem) {

            sendMessage(new Message (m_oUID,"last item",m_oPacmanID));

        } else if(nKeycode == m_nNextItem) {

            sendMessage(new Message (m_oUID,"next item",m_oPacmanID));

        } else if(nKeycode == m_nUseItem ) {

            sendMessage(new Message (m_oUID,"use item",m_oPacmanID));

        }

    }
    /** this method handles keys pressed while Control is pressed
     *
     * @param nKeycode the KeyCode to be handled
     */
    private void moveUpDown(int nKeycode) {

        if(nKeycode == m_nForwardKey)

             sendMessage(new Message (m_oUID,
                             new MovementMessage("move","up",m_bIsFast),
                             m_oPacmanID));

        else if(nKeycode == m_nBackKey)

             sendMessage(new Message (m_oUID,
                             new MovementMessage("move","down",m_bIsFast),
                             m_oPacmanID));

        else if(nKeycode == m_nLeftKey)

             sendMessage(new Message (m_oUID,
                             new MovementMessage("move","left",m_bIsFast),
                             m_oPacmanID));

        else if(nKeycode == m_nRightKey)

             sendMessage(new Message (m_oUID,
                             new MovementMessage("move","right",m_bIsFast),
                             m_oPacmanID));

    }

    /** this method handles keys pressed without a modifier being pressed
     *
     * @param nKeycode the KeyCode to be handled
     */
    private void moveDirection(int nKeycode) {

        if(nKeycode == m_nForwardKey)

            sendMessage(new Message (m_oUID,
                            new MovementMessage("move","forward",m_bIsFast),
                            m_oPacmanID));

        else if(nKeycode == m_nBackKey)

             sendMessage(new Message (m_oUID,
                             new MovementMessage("move","back",m_bIsFast),
                             m_oPacmanID));

        else if(nKeycode == m_nLeftKey)

             sendMessage(new Message (m_oUID,
                             new MovementMessage("rotate","left",m_bIsFast),
                             m_oPacmanID));

        else if(nKeycode == m_nRightKey)

             sendMessage(new Message (m_oUID,
                             new MovementMessage("rotate","right",m_bIsFast),
                             m_oPacmanID));

    }

    /** this method handles keys pressed while Alt is pressed
     *
     * @param nKeycode the KeyCode to be handled
     */
    protected void moveRotation(int nKeycode) {

        if(nKeycode == m_nForwardKey)

             sendMessage(new Message (m_oUID,
                             new MovementMessage("rotate","up",m_bIsFast),
                             m_oPacmanID));

        else if(nKeycode == m_nBackKey)

             sendMessage(new Message (m_oUID,
                             new MovementMessage("rotate","down",m_bIsFast),
                             m_oPacmanID));

        else if(nKeycode == m_nLeftKey)

             sendMessage(new Message (m_oUID,
                             new MovementMessage("rotate","left",m_bIsFast),
                             m_oPacmanID));

        else if(nKeycode == m_nRightKey)

             sendMessage(new Message (m_oUID,
                             new MovementMessage("rotate","right",m_bIsFast),
                             m_oPacmanID));

    }
    /** this method will send all messages for the camera group
     *
     * @param nKeyCode the KeyCode to be handled
     */
    private void changeView(int nKeyCode) {

        if (nKeyCode==m_nIsoViewKey) {

            sendMessage(new Message(m_oUID,"iso",m_oCameraID));

        } else if (nKeyCode==m_nBirdViewKey) {

            sendMessage(new Message(m_oUID,"bird",m_oCameraID));

        } else if (nKeyCode==m_nEgoViewKey) {

            sendMessage(new Message(m_oUID,"ego",m_oCameraID));

        } else if (nKeyCode==m_nTrackerViewKey) {

            sendMessage(new Message(m_oUID,"tracker",m_oCameraID));

        } else if (nKeyCode==m_nToggleStyleKey) {

            sendMessage(new Message(m_oUID,"toggle style",m_oCameraID));

        } else if (nKeyCode==m_nCameraUpKey) {

            sendMessage(new Message(m_oUID,"camup",m_oCameraID));

        } else if (nKeyCode==m_nCameraDownKey) {

            sendMessage(new Message(m_oUID,"camdown",m_oCameraID));

        } else if (nKeyCode==m_nCameraLeftKey) {

            sendMessage(new Message(m_oUID,"camleft",m_oCameraID));

        } else if (nKeyCode==m_nCameraRightKey) {

            sendMessage(new Message(m_oUID,"camright",m_oCameraID));

        } else if (nKeyCode==m_nCameraResetKey) {

            sendMessage(new Message(m_oUID,"camreset",m_oCameraID));

        } else if (nKeyCode==m_nCameraZoomInKey) {

            sendMessage(new Message(m_oUID,"camzoomin",m_oCameraID));

        } else if (nKeyCode==m_nCameraZoomOutKey) {

            sendMessage(new Message(m_oUID,"camzoomout",m_oCameraID));

        }
    }

    /** redefine forward key
     *
     * @param nKey the new Key to be recognized as Forward
     */
    public void setForwardKey(int nKey) {

        m_nForwardKey = nKey;

    }
    /** redefine back key
     *
     * @param nKey the new Key to be recognized as Backward
     */
    public void setBackKey(int nKey) {

        m_nBackKey = nKey;

    }
    /** redefine left key
     *
     * @param nKey the new Key to be recognized as Left
     */
    public void setLeftKey(int nKey) {

        m_nLeftKey = nKey;

    }
    /** redefine right key
     *
     * @param nKey the new Key to be recognized as Right
     */
    public void setRightKey(int nKey) {

        m_nRightKey = nKey;

    }
    /** redefine EgoView key
     *
     * @param nKey the new Key to switch to the EgoView
     */
    public void setEgoViewKey(int nKey) {

        m_nEgoViewKey = nKey;

    }
    /** redefine TrackerView key
     *
     * @param nKey the new Key to switch to the TrackerView
     */
    public void setTrackerViewKey(int nKey) {

        m_nTrackerViewKey = nKey;

    }
    /** redefine BirdView key
     *
     * @param nKey the new Key to switch to the BirdView
     */
    public void setBirdViewKey(int nKey) {

        m_nBirdViewKey = nKey;

    }
    /** redefine IsoView key
     *
     * @param nKey the new Key to switch to the IsoView
     */
    public void setIsoViewKey(int nKey) {

        m_nIsoViewKey = nKey;

    }
    /** redefine ToggleStyle key
     *
     * @param nKey the new Key to toggle the style
     */
    public void setToggleStyleKey(int nKey) {

        m_nToggleStyleKey = nKey;

    }

    /** sets key to quit the game
     *
     * @param nKey the new Key to escape the game
     */
    public void setEscapeKey(int nKey) {

        m_nEscape = nKey;

    }

    /** Handle the key pressed event
     *
     * @param e event :)
     */
    public void keyPressed(KeyEvent e) {

        //processKeyEvent(e);

    }
    /** Handle the key typed event
     *
     * @param e event :)
     */
    public void keyTyped(KeyEvent e) {
        //processKeyEvent(e);
    }

    /** Handle the key released event
     *
     * @param e event :)
     */
    public void keyReleased(KeyEvent e) {
        processKeyEvent(e);
    }

    /** sends the Message to the PostOffice (Labyrinth)
     *
     * @param oMessage the message to be send
     */
    private void sendMessage (Message oMessage) {

        if (m_oMessageService != null) {

            try {

                m_oMessageService.sendMessage(oMessage,m_oMessageService.PROPAGATE_LOCAL);

            }
            catch (Exception ex) {

                Debug.out(this.getClass().getName(),
                    "Error while sending Message: " + oMessage ,Debug.LEVEL_NOTICE );
                Debug.out(this.getClass().getName(),ex);

            }
        }
    }
}