package pacman3d.net;

import java.io.*;
import java.net.*;
import java.util.*;

/**
 * <b>Title:</b> Java3D-Praktikum - Pacman3D</br>
 * <b>Description:</b><br>
 *
 * NetworkService-Implementierung f&uuml;r TCP-Server
 * <br>
 *
 * <b>Copyright:</b>	Copyright (c) 2002<br>
 *
 * @author              Netz+Intro<br>
 */
public final class TCPServer extends TCPClient implements Runnable {

	/*
	 * 	Server-spezifische variablen
	 */

	// optionen fuer properties-datei
	private int maxClients = DEFAULT_MAX_CLIENTS;

	private final static int DEFAULT_MAX_CLIENTS = 32;

	//serverSocket
	private ServerSocket serverSocket;

	//socket-liste zum verteilen der nachrichten (enthaelt Statuscode)
	private ClientList clientList;

	//beendet run() in abgeleiteter klasse, wenn wert auf false
	private boolean mainloop = true;

	private final String DEFAULT_CONFIG_FILE = "./config/TCPServer.config";

	/**
	 * Erzeugt TCPServer.
	 * 
	 * @param maxClients maximale Anzahl an Clientverbindungen
	 * @param host HostIP als String
	 * @param port Port des Host
	 */
	public TCPServer(int maxClients, String host, int port) {
		loadProperties(DEFAULT_CONFIG_FILE);
		this.host = host;
		checkProperties();
		clientList = new ClientList(maxClients);
	}

	/**
	 * Erzeugt TCPServer.
	 * 
	 */
	public TCPServer() {
		loadProperties(DEFAULT_CONFIG_FILE);
		checkProperties();
		clientList = new ClientList(maxClients);
	}

	/**
	 * Erzeugt TCPServer.
	 * 
	 * @param host HostIP als String
	 */
	public TCPServer(String host) {
		loadProperties(DEFAULT_CONFIG_FILE);
		this.host = host;
		checkProperties();
		clientList = new ClientList(maxClients);
	}

	/**
	 * Verr&auml;t, ob Server oder Client
	 * 
	 * @return Server (true) oder Client (false)
	 */
	public boolean isServer() {
		return true;
	}

	/**
	 * Starte Thread und Netzwerkaktivitaet dieser Klasse.
	 */
	public void startNetwork() {
		new Thread(this).start();
	}

	/**
	 * Beendet Netzwerkaktivitaet, schliesst TCPServer-Instanz.
	 */
	public void closeNetwork() {
		mainloop = false;
		try {
			if (serverSocket != null)
				serverSocket.close();
		} catch (IOException ex) {
			log(this, "IOException in closeNetwork().");
		}

		storeProperties(DEFAULT_CONFIG_FILE);
	}

	/*
	 * 			Einstellungsverwaltung
	 */

	/**
	 * Laden der Grundeinstellungen
	 * 
	 * @param file Dateiname der Konfigurationsdatei
	 */
	private void loadProperties(String file) {
		log(this, "loading properties from " + file + "...");

		Properties options = new Properties();
		File configFile = new File(file);
		try {
			options.load(new FileInputStream(configFile));
		} catch (IOException ex) {
			log(this, "cannot load properties file. setting default values...");
		}

		String prop_host = options.getProperty("host", "127.0.0.1");
		String prop_timeout =
			options.getProperty("timeout", String.valueOf(DEFAULT_TIMEOUT));
		String prop_maxclients =
			options.getProperty("maxclients", String.valueOf(DEFAULT_MAX_CLIENTS));

		log(this, "host = " + prop_host);
		//todo: syntaxcheck host
		host = prop_host;

		try {
			timeout = Integer.valueOf(prop_timeout).intValue();
			if (timeout < 0)
				timeout = DEFAULT_TIMEOUT;
		} catch (Exception ex) {
			timeout = DEFAULT_TIMEOUT;
		}

		try {
			maxClients = Integer.valueOf(prop_maxclients).intValue();
			if (maxClients < 0)
				maxClients = DEFAULT_MAX_CLIENTS;
		} catch (Exception ex) {
			maxClients = DEFAULT_MAX_CLIENTS;
		}

		log(this, "timeout = " + timeout);
		log(this, "maxClients = " + maxClients);

	}

	/**
	 * Speichern der letzten Einstellungen in einer Konfigurationsdatei
	 * 
	 * @param file Dateiname der Konfigurationsdatei
	 */
	private void storeProperties(String file) {
		Properties options = new Properties();
		File configFile = new File(file);
		options.setProperty("host", host);
		options.setProperty("timeout", String.valueOf(timeout));
		options.setProperty("maxclients", String.valueOf(maxClients));
		try {
			log(this, "storing properties...");
			log(this, "host = " + host);
			log(this, "timeout = " + timeout);
			log(this, "maxclients = " + maxClients);
			options.store(new FileOutputStream(configFile), DEFAULT_CONFIG_FILE);
		} catch (IOException ex) {
			log(this, "cannot store properties file.");
		}
	}

	/**
	 * U&uuml;berpr&uuml;fung der aktuellen Parameter vor Beginn
	 */
	private void checkProperties() {
		//checking host
		localHost = null;
		try {
			localHost = InetAddress.getByName(host);
			log(this, "checkProperties: localhost = " + localHost);
		} catch (UnknownHostException ex) {
			//cba:18.2.02
			sendStatusCode(NetworkListener.STATUS_CRASHED);
			closeNetwork();
		}
	}

	/*
	 * 		server stuff
	 */

	/**
	 * Server-Mainloop: Wartet auf Verbindungen und leitet diese an <i>TCPServerThread</i>
	 * weiter. Initiiert handshaking etc.
	 */
	public void run() {

		// setting up server
		try {
			serverSocket = new ServerSocket(port, 0, localHost);
			log(this, "run(): server listening on " + serverSocket);
			sendStatusCode(NetworkListener.STATUS_CONNECTED);
			Socket socket = null;
			while (mainloop) {
				try {
					socket = serverSocket.accept();
					socket.setSoTimeout(timeout);
					log(this, "run(): connection from " + socket);

					if (clientList.addClient(socket, STATUS_HANDSHAKE, socket.hashCode())) {
						log(this, "run(): client accepted in list.");
						TCPServerThread t = new TCPServerThread(this, socket);
						new Thread(t).start();
					} else {
						log(this, "run(): client not accepted. removing from list");
						removeConnection(socket);
					}

				} catch (Exception ex) {
					log(this, ex + " in mainloop.");
					removeConnection(socket);
				}
			}

		} catch (Exception ex) {
			log(this, ex + " while creating ServerSocket");
			sendStatusCode(NetworkListener.STATUS_CRASHED);
		}

	}

	/**
	 * Testet Handshake-Aufforderung von Client auf Korrektheit.
	 * 
	 * @param socket Client-Socket
	 */
	protected synchronized void checkHandshake(Socket socket) throws Exception {

		DataInputStream dataIn = new DataInputStream(socket.getInputStream());

		log(this, "checkHandshake(): handshake message from client...");

		// version-string
		int size = dataIn.readInt();
		byte[] bytes = new byte[size];
		dataIn.readFully(bytes);
		String version = new String(bytes);
		// id
		int id = dataIn.readInt();

		log(this, "checkHandshake(): version = " + version + " : id = " + id);
		log(this, "checkHandshake(): writing \"okay\" to client.");
		DataOutputStream dataOut = new DataOutputStream(socket.getOutputStream());
		// status
		dataOut.writeInt(STATUS_HANDSHAKE);
		//version
		String s_version = "TCPServer for Pacman3D";
		dataOut.writeInt(s_version.length());
		dataOut.write(s_version.getBytes());
		log(this, "checkHandshake(): write: version = " + s_version);

		// client-test (alpha-status)
		if (version.equals("TCPClient for Pacman3D")) {
			//typ
			dataOut.writeInt(HANDSHAKE_OK);

			// id
			dataOut.writeInt(socket.hashCode());
			log(
				this,
				"checkHandshake(): write: typ = HANDSHAKE_OK, id = " + socket.hashCode());
			log(this, "checkHandshake(): currentStatus: handshake=>data");
			// accept client in list
			clientList.setStatus(socket, STATUS_DATA);
			dataOut.flush();
		} else {
			//typ
			dataOut.writeInt(HANDSHAKE_DENIED);
			log(this, "checkHandshake(): denied message. removing conenction.");
			dataOut.flush();
			removeConnection(socket);
		}
	}

	/**
	 * Entfernt das Client-Socket nach Exception
	 * 
	 * @param socket Client-Socket
	 */
	protected void removeConnection(Socket socket) {
		log(this, "removeConnection(): removing socket " + socket + " from list");
		clientList.removeClient(socket);
		try {
			if (socket != null)
				socket.close();
		} catch (IOException ex) {
			log(this, "removeConnection(): cannot close socket (null).");
		}
	}

	/**
	 * Sendet eine Nachricht an alle Clients. Methode wird von
	 * MessageService-Objekt aufgerufen.
	 * 
	 * @param message NetworkMessage zum Senden
	 */
	public void send(NetworkMessage message) {
		send(message, null);
	}

	/**
	 * Sendet eine Nachricht an alle Clients bis auf den in <code>oSocket</code>
	 * spezifizierten Client. Methode wird von MessageService-Objekt aufgerufen.
	 * 
	 * @param message NetworkMessage zum Senden
	 * @param oSocket Client-Socket
	 */
	public void send(NetworkMessage message, Socket oSocket) {
		if (message != null) {
			byte[] bytes = encodeMessage(message);
			if (bytes.length > 0)
				sendBytes(bytes, oSocket);
		} else {
			log(this, "send(): message is null.");
		}
	}

	/**
	 * Low-level-Routine zum Senden von byte[]-Messaged an alle Clients
	 * mit Ausnahme des Sockets, der in <code>oSocket</code> angegeben
	 * ist.
	 * 
	 * @param bytes Array zum Senden
	 * @param oSocket Client-Socket, welches ausgeschlossen werden soll
	 */
	protected synchronized void sendBytes(byte[] bytes, Socket oSocket) {
		DataOutputStream dataOut = null;
		Socket[] sockets = clientList.getSockets();
		for (int i = 0; i < sockets.length; i++) {
			if ((sockets[i] != null)
				&& (sockets[i] != oSocket)
				&& clientList.getStatus(sockets[i]) == STATUS_DATA) {
				try {
					log(this, "sendBytes(): write " + bytes.length + " bytes to " + sockets[i]);
					dataOut = new DataOutputStream(sockets[i].getOutputStream());
					dataOut.writeInt(clientList.getStatus(sockets[i]));
					dataOut.writeInt(bytes.length);
					dataOut.write(bytes);
					dataOut.flush();
				} catch (Exception ex) {
					log(this, ex + " in sendBytes().");
					removeConnection(sockets[i]);
				}

			}
		}
	}

	/**
	 * Low-level-Routine zum Senden von byte[]-Messaged an alle Clients
	 * 
	 * @param bytes bytes-Array zum Senden
	 */
	protected synchronized void sendBytes(byte[] bytes) {
		sendBytes(bytes, null);
	}

} // end of class TCPServer