package pacman3d.net;

import java.io.*;
import java.net.*;
import java.util.*;

import pacman3d.util.Debug;

/**
 * <b>Title:</b> Java3D-Praktikum - Pacman3D</br>
 * <b>Description:</b><br>
 *
 * NetworkService-Implementierung f&uuml;r TCP-Client
 * <br>
 *
 * <b>Copyright:</b>	Copyright (c) 2002<br>
 *
 * @author              Netz+Intro<br>
 */
public class TCPClient implements Runnable, NetworkService {

	/*
	 * 	Globale Variablen fuer Client und Server
	 */

	// verwaltet nachrichtenempfaenger
	private Vector listenerList = new Vector();
	// maximale anzahl von nachrichtenempfaengern
	private final static int maxNetworkListeners = 255;

	// fester port
	protected final static int port = 10001;

	// interner status fuer server-client kommunikation
	protected static final int STATUS_OFFLINE = 0;
	protected static final int STATUS_HANDSHAKE = 1;
	protected static final int STATUS_DATA = 2;
	// interner staus fuer handshake protokoll
	protected static final int HANDSHAKE_OK = 10;
	protected static final int HANDSHAKE_DENIED = 11;

	// optionen fuer properties-datei
	protected String host = null;
	protected int timeout = 0;
	protected InetAddress localHost;
	private int reconnectMax = 10;

	protected final static int DEFAULT_RECONNECT_MAX = 10;
	protected final static int DEFAULT_TIMEOUT = 0;

	/*
	 * 	Client-spezifische variablen
	 */

	// Socket zum Server
	private Socket socket;

	// ID vom Handshake
	private int currentID = -1;

	// Aktueller Status der Verbindung (s.o)
	private int currentStatus = STATUS_OFFLINE;

	// Beendet run() in abgeleiteter klasse, wenn wert auf false
	private boolean mainloop = true;

	// Konfigurationsdatei
	private final String DEFAULT_CONFIG_FILE = "./config/TCPClient.config";

	// debug
	protected static final boolean debug = false;

	/*
	 * 				Schnittstelle
	 */

	/**
	 * Leitet Meldungen an Debug-Klasse weiter.
	 * 
	 * @param obj betroffene Klasse
	 * @param string Meldungstext
	 */
	protected static void log(Object obj, String string) {
		if (debug)
			Debug.out(obj.getClass().getName(), string, Debug.LEVEL_NOTICE);
	}

	/**
	 * Erzeugt TCPClient.
	 */
	public TCPClient() {
		loadProperties(DEFAULT_CONFIG_FILE);
	}

	/**
	 * Erzeugt TCPClient.
	 * 
	 * @param host HostIP als String
	 */
	public TCPClient(String host) {
		loadProperties(DEFAULT_CONFIG_FILE);
		this.host = host;
	}

	/**
	 * Starte Thread und Netzwerkaktivitaet dieser Klasse.
	 */
	public void startNetwork() {
		new Thread(this).start();
	}

	/**
	 * Methode, zum Setzen des Host-Parameters
	 * 
	 * @param host HostIP
	 */
	public void setHost(String host) {
		this.host = host;
		checkProperties();
	}

	/**
	 * Verr&auml;t, ob Server oder Client
	 * 
	 * @return Server (true) oder Client (false)
	 */
	public boolean isServer() {
		return false;
	}

	/**
	 * Liefert HostIP.
	 * 
	 * @return HostIP
	 */
	public String getHost() {
		return host;
	}

	/**
	 * Beendet Netzwerkaktivitaet, schliesst TCPClient-Instanz
	 */
	public void closeNetwork() {
		currentStatus = STATUS_OFFLINE;
		mainloop = false;
		try {
			if (socket != null)
				socket.close();
		} catch (IOException ex) {
			log(this, "IOException in closeNetwork().");
		}
		storeProperties(DEFAULT_CONFIG_FILE);
	}

	/*
	 * 			Einstellungsverwaltung
	 */

	/**
	 * Laden der Grundeinstellungen
	 * 
	 * @param file Dateiname der Konfigurationsdatei
	 */
	private void loadProperties(String file) {
		log(this, "loading properties from " + file + "...");

		Properties options = new Properties();
		File configFile = new File(file);
		try {
			options.load(new FileInputStream(configFile));
		} catch (IOException ex) {
			log(this, "cannot load properties file. setting default values...");
		}

		String prop_host = options.getProperty("host", "127.0.0.1");
		String prop_timeout =
			options.getProperty("timeout", String.valueOf(DEFAULT_TIMEOUT));
		String prop_reconnect =
			options.getProperty("reconnect", String.valueOf(DEFAULT_RECONNECT_MAX));

		log(this, "host = " + prop_host);
		//todo: syntaxcheck host
		host = prop_host;

		try {
			timeout = Integer.valueOf(prop_timeout).intValue();
			if (timeout < 0)
				timeout = DEFAULT_TIMEOUT;
		} catch (Exception ex) {
			timeout = DEFAULT_TIMEOUT;
		}
		try {
			reconnectMax = Integer.valueOf(prop_reconnect).intValue();
			if (reconnectMax < 0)
				reconnectMax = DEFAULT_RECONNECT_MAX;
		} catch (Exception ex) {
			reconnectMax = DEFAULT_RECONNECT_MAX;
		}

		log(this, "timeout = " + timeout);
		log(this, "reconnect = " + reconnectMax);

	}

	/**
	 * Speichern der letzten Einstellungen in einer Konfigurationsdatei
	 * 
	 * @param file Dateiname der Konfigurationsdatei
	 */
	private void storeProperties(String file) {
		Properties options = new Properties();
		File configFile = new File(file);
		options.setProperty("host", host);
		options.setProperty("timeout", String.valueOf(timeout));
		options.setProperty("reconnect", String.valueOf(reconnectMax));

		try {
			log(this, "storing properties...");
			log(this, "host = " + host);
			log(this, "timeout = " + timeout);
			options.store(new FileOutputStream(configFile), DEFAULT_CONFIG_FILE);
		} catch (IOException ex) {
			log(this, "cannot store properties file.");
		}
	}

	/**
	 * U&uuml;berpr&uuml;fung der aktuellen Parameter vor Beginn
	 */
	private void checkProperties() {
		//checking host
		InetAddress localHost = null;
		try {
			localHost = InetAddress.getByName(host);
		} catch (UnknownHostException ex) {
			sendStatusCode(NetworkListener.STATUS_CRASHED);
			closeNetwork();
		}
	}

	/*
	 * 		NetworkListener - support
	 */

	/**
	 * Registriert einen NetworkListener bei der Netzwerkschnittstelle.
	 * Nach der Registrierung kann jeder Abonnent Nachrichten erhalten
	 * 
	 * @param listener NetworkListener-Instanz
	 */
	public void addNetworkListener(NetworkListener listener) {
		if ((listener != null)
			&& (!listenerList.contains(listener))
			&& (listenerList.size() < maxNetworkListeners))
			listenerList.add(listener);
		else
			log(this, "addNetworkListener(): listener null or already in list.");
	}

	/**
	 * Entfernt einen NetworkListener von der Netzwerkschnittstelle.
	 * 
	 * @param listener NetworkListener-Instanz	 
	 */
	public void removeNetworkListener(NetworkListener listener) {
		if ((listener != null) && (listenerList.contains(listener)))
			listenerList.remove(listener);
	}

	/**
	 * Verteilt byte-Array-Nachricht an angemeldete NetworkListener.
	 * 
	 * @param message zu verteilende Nachricht
	 */
	protected synchronized void sendToNetworkListener(NetworkMessage message) {
		if (message != null) {
			//cba: testing 21.2.02
			log(this, "sendToNetworkListener(): called.");
			// Schicke Nachricht an NetworkListener
			Enumeration enum = listenerList.elements();
			while (enum.hasMoreElements()) {
				NetworkListener listener = (NetworkListener) enum.nextElement();
				if (listener != null) {
					listener.networkMessageReceived(message);
				}
			}
		}
	}

	/**
	 * Sendet Status-Informationen zu NetzworkListenern, z.B. nach Abbruch der Verbindung
	 * 
	 * @param code Statuscode
	 * @see <i>NetworkListener</i>
	 */
	protected synchronized void sendStatusCode(int code) {

		Enumeration enum = listenerList.elements();
		while (enum.hasMoreElements()) {
			NetworkListener listener = (NetworkListener) enum.nextElement();
			if (listener != null) {
				listener.networkStatus(code);
			}
		}

	}

	/*
	 * 			converting byte[] / NetworkMessage
	 */

	/**
	 * Wandelt byte-Array in NetworkMessage-Objekt um.
	 * 
	 * @param bytes byte-Array der Nachricht
	 * @return NetworkMessage-Objekt aus byte-Array
	 */
	protected NetworkMessage decodeMessage(byte[] bytes) {
		ByteArrayInputStream byteIn = new ByteArrayInputStream(bytes);
		try {
			ObjectInputStream objIn = new ObjectInputStream(byteIn);
			NetworkMessage obj = (NetworkMessage) objIn.readObject();
			return obj;
		} catch (Exception ex) {
			log(this, ex + " in decodeMessage().");
		}
		return null;
	}

	/**
	 * Wandelt ein NetworkMessage-Objekt in einen byte-Array um.
	 * 
	 * @param message NetworkMessage-Objekt
	 * @return byte-Array der Nachricht
	 */
	protected byte[] encodeMessage(NetworkMessage message) {
		ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
		try {
			ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
			objOut.writeObject(message);
			objOut.flush();
			return byteOut.toByteArray();
		} catch (Exception ex) {
			log(this, ex + " in encodeMessage()");
		}
		return null;
	}

	/*
	 * 				client stuff
	 */

	/**
	 * Schliesst &uuml;bergebenes Socket.
	 * 
	 * @param socket zu schliessendes Socket
	 */
	protected void closeSocket(Socket socket) {
		if (socket != null) {
			try {
				socket.close();
			} catch (IOException ex) {
			}
		}
	}

	/**
	 * Sendet Handshake-Info an Server
	 * 
	 * @param socket Zielsocket
	 */
	private synchronized void requestHandshake(Socket socket) {
		try {
			// requesting handshake
			log(this, "requesting handshake...");
			DataOutputStream dataOut = new DataOutputStream(socket.getOutputStream());
			// status-info
			dataOut.writeInt(STATUS_HANDSHAKE);
			// version
			String version = "TCPClient for Pacman3D";
			dataOut.writeInt(version.length());
			dataOut.write(version.getBytes());
			// currentID
			dataOut.writeInt(currentID);
			dataOut.flush();
			log(
				this,
				"requestHandshake(): write: version = " + version + " : id = " + currentID);
		} catch (Exception ex) {
			log(this, ex + " in requestHandshake(). status => offline");
			currentStatus = STATUS_OFFLINE;
		}

	}

	/**
	 * &Uuml;berpr&uuml;fe erhaltene Handshake-Informationen.<br>
	 * Falls empfangener Status != STATUS_DATA, dann handle entsprechend.
	 * 
	 * @param dataIn DataInputStream des betreffenden Sockets
	 */
	private synchronized void checkHandshake(DataInputStream dataIn)
		throws Exception {

		log(this, "checkHandshake(): handshake message from server...");
		// version
		int size = dataIn.readInt();
		byte[] bytes = new byte[size];
		dataIn.readFully(bytes);
		String version = new String(bytes);
		// typ
		int typ = dataIn.readInt();
		log(this, "checkHandshake(): read: version = " + version + " : typ = " + typ);

		// falls
		if (typ == HANDSHAKE_OK) {
			// currentID
			int id = dataIn.readInt();
			currentID = id;
			currentStatus = STATUS_DATA;
			log(this, "checkHandshake(): read: id = " + currentID);
			log(this, "checkHandshake(): currentStatus: handshake=>data");
			sendStatusCode(NetworkListener.STATUS_CONNECTED);
		} else if (typ == HANDSHAKE_DENIED) {
			log(this, "checkHandshake(): ACCESS DENIED");
			sendStatusCode(NetworkListener.STATUS_DENIED);
			currentStatus = STATUS_OFFLINE;
		}
	}

	/**
	 * Client-Mainloop: Wartet auf Nachrichten vom Server und leitet diese weiter,
	 * initiiert handshaking etc.
	 */
	public void run() {

		int reconnectCount = 0;

		while (mainloop) {

			// abbruchbedingung
			if (reconnectCount >= reconnectMax) {
				break;
			}

			// offline. trying to connect
			currentStatus = STATUS_OFFLINE;
			DataInputStream dataIn = null;
			try {
				log(this, "run(): connecting...");
				socket = new Socket(host, port);
				socket.setSoTimeout(timeout);
				dataIn = new DataInputStream(socket.getInputStream());
				log(this, "run(): connected (" + socket + ")");
				currentStatus = STATUS_HANDSHAKE;
			} catch (BindException ex) {
				//cannot connect to socket
				log(this, "BindException in mainloop.");
				sendStatusCode(NetworkListener.STATUS_RECONNECT);
				reconnectCount++;
			} catch (IOException ex) {
				// error. ie socket closed
				log(this, "IOException in mainloop.");
				sendStatusCode(NetworkListener.STATUS_RECONNECT);
				reconnectCount++;
			}

			while ((currentStatus == STATUS_HANDSHAKE) || (currentStatus == STATUS_DATA)) {

				if (currentStatus == STATUS_HANDSHAKE)
					requestHandshake(socket);

				try {
					int mode = dataIn.readInt();
					log(this, "run(): message arrived. status = " + mode);
					if (mode == STATUS_HANDSHAKE)
						checkHandshake(dataIn);
					else {
						int size = dataIn.readInt();
						log(this, "run(): read " + size + " bytes from server");
						byte[] bytes = new byte[size];
						dataIn.readFully(bytes);
						// dataenempfang okay -> reconnectCount = 0
						reconnectCount = 0;
						sendToNetworkListener(decodeMessage(bytes));

					}
				} catch (Exception ex) {
					Debug.out(
						getClass().getName(),
						"server is gone. closing socket. trying to reconnect.",
						Debug.LEVEL_NOTICE);
					closeSocket(socket);
					currentStatus = STATUS_OFFLINE;
					reconnectCount++;
					sendStatusCode(NetworkListener.STATUS_RECONNECT);
				}
			} //end of while-loop

		} //end of mainloop

		// falls schleifenende wg. crash -> meldung an listener
		if (reconnectCount >= reconnectMax) {
			sendStatusCode(NetworkListener.STATUS_CRASHED);
			closeNetwork();
		}

	} //end of run

	/**
	 * Sendet eine Nachricht. Methode wird von MessageService-Objekt aufgerufen.
	 * 
	 * @param message NetworkMessage zum Senden
	 */
	public void send(NetworkMessage message) {
		if ((message != null) && (currentStatus == STATUS_DATA)) {
			byte[] bytes = encodeMessage(message);
			if (bytes != null && bytes.length > 0)
				sendBytes(bytes);
			else
				log(this, "send(): bytes is NULL after encodeMessage()");
		} else {
			log(this, "send(): called but client offline or status != data.");
			log(
				this,
				"send(): message = "
					+ (message == null ? "null" : "not null")
					+ "; currentStatus = "
					+ currentStatus);
		}
	}

	/**
	 * low-level Senderoutine fuer Datenpakete
	 * 
	 * @param bytes Array zum Versenden
	 */
	protected synchronized void sendBytes(byte[] bytes) {
		try {
			DataOutputStream dataOut = new DataOutputStream(socket.getOutputStream());
			dataOut.writeInt(currentStatus);
			dataOut.writeInt(bytes.length);
			dataOut.write(bytes);
			dataOut.flush();
			log(this, "sendBytes(): sending " + bytes.length + " bytes.");
		} catch (Exception ex) {
			log(this, ex + " in sendBytes(). currentStatus: data=>offline");
			currentStatus = STATUS_OFFLINE;
			sendStatusCode(NetworkListener.STATUS_CRASHED);
		}
	}

} //end of class TCPClient