/*
 * MonsterSetup.java
 */

package pacman3d.monster;

import javax.swing.*;
import java.awt.*;
import java.util.*;
import javax.vecmath.Point3i;

/**
 * <p>Durch das MonsterSetup ist es moeglich, die Eigenschaften jedes Monsters
 * waehrend des Spiels zu veraendern.</p>
 *
 * <p>Fuer jedes Monster wird ein Setup-Fenster aufgerufen, in dem mit
 * Schiebereglern (JSlider) Kommunikationsradius, Intelligenz und Suchradius
 * eingestellt werden koennen.</p>
 */
public class MonsterSetup extends JFrame {

  /** Erzeugen der JSlider fuer Kommunikationsradius, Sichtweite und Intelligenz */
  JSlider ai = new JSlider();
  JSlider com = new JSlider();
  JSlider search = new JSlider();

  /** Referenz auf die Awareness */
  private MonsterAwareness awareness;

  /** Graphics fuer die Darstellung */
  private Graphics graphics;

  /** Liste fuer die zu ladenden gifs */
  private LinkedList monsterImages = new LinkedList();

  /** die hiermit geladen werden */
  private Image image;

  /** Erzeugen der JLabels zur Beschriftung und fuer das Layout der Fenster */
  JLabel textField2 = new JLabel();
  JLabel textField1 = new JLabel();
  JLabel picture = new JLabel();
  JLabel itemSearch = new JLabel();
  JLabel itemCom = new JLabel();
  JLabel itemAi = new JLabel();

  /** Erzeugen des Layout-Managers, um die Objekte im Fenster anordnen zu koennen */
  GridBagLayout gridBagLayout1 = new GridBagLayout();
  GridBagLayout gridBagLayout2 = new GridBagLayout();

  /** Parameter der Monster-Eigenschaften, die in den Fenstern visualisiert
   *  werden sollen */
  private int searchMin = 0;
  private int searchMax = 0;
  private int searchInit = 0;
  private int commMin = 0;
  private int commMax = 0;
  private int commInit = 0;
  private int aiMin = 0;
  private int aiMax = 0;
  private int aiInit = 0;

  private int monsterModel = 0;
  private float monsterSpeed = 100.0f;
  private int monsterStatus = -1;

  /** Startposition des Monsters auf null setzen */
  private static Point3i startPosition = null;

  /** Flag wird zur Ermittlung des JagdStatus abgefragt und gesetzt */
  public boolean hunting = true;

  /**
   * erzeugt ein neues MonsterSetup
   *
   * @param aware  Referenz auf die MonsterAwareness
   * @see MonsterAwareness
   */
  public MonsterSetup(MonsterAwareness aware)
  {
    try
    {
      awareness = aware;

      // Initialisierung der Monster-Parameter
      // (diese Parameter werden auch zur Berechnung der Hintergrund-Farbwerte
      // benutzt, die sich je nach Einstellung der Schieberegler veraendern)
      searchMin = 1;
      searchMax = 10;
      searchInit = 6;

      commMin = 1;
      commMax = 5;
      commInit = 2;

      aiMin = 1;
      aiMax = 5;
      aiInit = 2;

      jbInit();
    }
    catch(Exception e) {
      e.printStackTrace();
    }
  }// end of constructor


  /**
   * erzeugt ein neues MonsterSetup, das die MonsterProperties auswertet
   *
   * @param monsterProperties was genau steht da drin; default werte der Monster-Eigenschaften ???
   */
  public MonsterSetup(MonsterProperties monsterProperties)
  {
    try
    {
      // Initialisierung der Monster-Parameter
      // (diese Parameter werden auch zur Berechnung der Hintergrund-Farbwerte
      // benutzt)
      searchMin = 1;
      searchMax = 10;

      commMin = 1;
      commMax = 8;

      aiMin = 1;
      aiMax = 5;

      commInit   = monsterProperties.getMonsterCommunicationRadius();
      searchInit = monsterProperties.getMonsterSearchDepth();
      aiInit     = monsterProperties.getMonsterStrategy();

      monsterModel = monsterProperties.getMonsterModel();
      monsterSpeed = monsterProperties.getMonsterSpeed();

      jbInit();
    }
    catch(Exception e) {
      e.printStackTrace();
    }
  }// end of constructor

  /**
   * initalisiert das Fenster und setzt die Graphik- und Text-Elemente
   */
  private void jbInit() throws Exception {
    // Setzen des Hintergrundes und der Gre des Monster-Bildes
    picture.setBackground(Color.darkGray);
    picture.setMaximumSize(new Dimension(70, 105));
    picture.setMinimumSize(new Dimension(70, 105));
    picture.setOpaque(true);
    picture.setHorizontalAlignment(SwingConstants.CENTER);

    // Setzen der Eigenschaften des Setup-Fensters
    this.getContentPane().setLayout(gridBagLayout2);
    this.setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
    this.setForeground(Color.black);
    this.setResizable(false);
    this.setTitle("MonsterSetup");
    GraphicsConfiguration conf = this.getGraphicsConfiguration();
    this.setSize(new Dimension(332, 203));
    String path = System.getProperty( "user.dir");

    // Setzen der Gre der Slider
    ai.setMaximumSize(new Dimension(80, 30));
    ai.setMinimumSize(new Dimension(80, 30));
    com.setMaximumSize(new Dimension(80, 30));
    com.setMinimumSize(new Dimension(80, 30));
    search.setMaximumSize(new Dimension(80, 30));
    search.setMinimumSize(new Dimension(80, 30));

    // Setzen des Designs und der Gre der Textfelder
    // TextField1, in dem die Kommunikation angezeigt wird
    textField1.setText("Kommunikation");
    textField1.setHorizontalTextPosition(SwingConstants.LEFT);
    textField1.setFont(new java.awt.Font("Dialog", 0, 10));
    textField1.setMaximumSize(new Dimension(100, 15));
    textField1.setMinimumSize(new Dimension(100, 15));
    textField1.setForeground(Color.black);

    // TextField2, in dem der Status angezeigt wird
    textField2.setText("Status");
    textField2.setHorizontalTextPosition(SwingConstants.LEFT);
    textField2.setFont(new java.awt.Font("Dialog", 0, 10));
    textField2.setMaximumSize(new Dimension(100, 15));
    textField2.setMinimumSize(new Dimension(100, 15));
    textField2.setForeground(Color.black);

    // die gifs werden aus der Liste monsterImages in image geladen
    // gif fr das Monster-Bild
    image = getToolkit().createImage(path + "/models/monster/Kodos.gif");
    monsterImages.addLast((Image) image);

    // gif fr den Suchradius-Slider
    image = getToolkit().createImage(path + "/models/monster/search.gif");
    monsterImages.addLast((Image) image);

    // gif fr den Intelligenz-Slider
    image = getToolkit().createImage(path + "/models/monster/ai.gif");
    monsterImages.addLast((Image) image);

    // gif fr den Kommunikationsradius-Slider
    image = getToolkit().createImage(path + "/models/monster/mega.gif");
    monsterImages.addLast((Image) image);

    // Setzen des Designs, der Position und der Gre der JLabels neben den Slidern
    // Suchradius-Label
    itemSearch.setBackground(Color.gray);
    itemSearch.setMinimumSize(new Dimension(36, 36));
    itemSearch.setMaximumSize(new Dimension(36, 36));
    itemSearch.setOpaque(true);
    itemSearch.setHorizontalAlignment(SwingConstants.CENTER);

    // Kommunikationsradius-Label
    itemCom.setBackground(Color.gray);
    itemCom.setMaximumSize(new Dimension(36, 36));
    itemCom.setMinimumSize(new Dimension(36, 36));
    itemCom.setOpaque(true);
    itemCom.setHorizontalAlignment(SwingConstants.CENTER);

    // Intelligenz-Label
    itemAi.setBackground(Color.gray);
    itemAi.setMaximumSize(new Dimension(36, 36));
    itemAi.setMinimumSize(new Dimension(36, 36));
    itemAi.setOpaque(true);
    itemAi.setHorizontalAlignment(SwingConstants.CENTER);

    // Monster-Parameter werden den Slidern bergeben
    // Parameter fr den Suchradius-Slider
    search.setMaximum(searchMax);
    search.setMinimum(searchMin);
    search.setPaintTicks(true);
    search.setValue(searchInit);
    search.setValueIsAdjusting(true);
    search.setSnapToTicks(true);

    // Parameter fr den Kommunikationsradius-Slider
    com.setMaximum(commMax);
    com.setMinimum(commMin);
    com.setPaintTicks(true);
    com.setValue(commInit);
    com.setValueIsAdjusting(true);
    com.setSnapToTicks(true);

    // Parameter fr den Intelligenz-Slider
    ai.setMaximum(aiMax);
    ai.setMinimum(aiMin);
    ai.setPaintTicks(true);
    ai.setValue(aiInit);
    ai.setValueIsAdjusting(true);
    ai.setSnapToTicks(true);

    // Anordnen der Elemente im Fenster mit Hilfe des Layout-Managers
    // Anordnen des Monster-Bildes und der Textfelder
    this.getContentPane().add(picture,  new GridBagConstraints(0, 0, 1, 3, 0.0, 0.0
            ,GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(22, 37, 18, 0), 0, 0));
    this.getContentPane().add(textField1,  new GridBagConstraints(0, 3, 2, 1, 0.0, 0.0
            ,GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(13, 70, 0, 0), 0, 0));
    this.getContentPane().add(textField2,  new GridBagConstraints(0, 4, 2, 1, 0.0, 0.0
            ,GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(0, 70, 15, 0), 0, 0));

    // Anordnen der Slider
    this.getContentPane().add(search,  new GridBagConstraints(2, 0, 1, 1, 0.0, 0.0
            ,GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(29, 10, 0, 70), 0, 0));
    this.getContentPane().add(ai,  new GridBagConstraints(2, 2, 1, 1, 0.0, 0.0
            ,GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(15, 8, 0, 70), 0, 0));
    this.getContentPane().add(com,  new GridBagConstraints(2, 1, 1, 1, 0.0, 0.0
            ,GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(11, 9, 0, 70), 0, 0));

    // Andordnen der Labels neben den Slidern
    this.getContentPane().add(itemAi,  new GridBagConstraints(1, 2, 1, 1, 0.0, 0.0
            ,GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(8, 18, 0, 0), 0, 0));
    this.getContentPane().add(itemSearch,  new GridBagConstraints(1, 0, 1, 1, 0.0, 0.0
            ,GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(22, 18, 0, 0), 0, 0));
    this.getContentPane().add(itemCom,  new GridBagConstraints(1, 1, 1, 1, 0.0, 0.0
            ,GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(6, 18, 0, 0), 0, 0));
  }

  /**
   * laedt ein gif aus der Liste monsterImages
   *
   * @param imageNumber Nummer des naechsten zu ladenden Bildes
   */
  public void drawMonster(int imageNumber)
  {
    System.out.println(monsterImages.toString());
    picture.setIcon(new ImageIcon((Image) monsterImages.get(imageNumber)));
    itemSearch.setIcon(new ImageIcon((Image) monsterImages.get(1)));
    itemAi.setIcon(new ImageIcon((Image) monsterImages.get(2)));
    itemCom.setIcon(new ImageIcon((Image) monsterImages.get(3)));
  }

  /**
   * faerbt die Rahmen der Labels je nach Einstellung der Slider ein
   */
  public void updatePicture()
  {
    // die Offsets nehmen Werte zwischen 0 und 127 an (abhaengig von aktuellem
    // Wert und Maximalwert)
    int searchOffset = (int) Math.floor(127 * search.getValue() / searchMax);
    int commOffset   = (int) Math.floor(127 *    com.getValue() /   commMax);
    int aiOffset     = (int) Math.floor(127 *     ai.getValue() /     aiMax);

    Color backroundColorSearch = new Color(128, 128 + searchOffset,  128);
    Color backroundColorCom    = new Color(128, 128 +   commOffset,  128);
    Color backroundColorAi     = new Color(128, 128 +     aiOffset,  128);

    itemSearch.setBackground(backroundColorSearch);
    itemCom.setBackground(backroundColorCom);
    itemAi.setBackground(backroundColorAi);
  }

  /**
   * fragt den Jagdstatus ab
   *
   * @return true wenn das Monster auf der Jagd ist
   */
  public boolean getMyState ()
  {
    return hunting;
  }

  /**
   * schreibt den aktuellen Status des Monsters in TextField2.
   * switch mit den cases:
   * 0: auf der Suche nach Pacman
   * 1: auf dem Weg zum Pacman
   * 2: zusaetzliches Broadcasten der Position des Pacman
   * 9: auf der Flucht
   * default: undefinierter Zustand
   */
  public void updateStatus()
  {
    switch (monsterStatus)
    {
      case 0:
      {
	textField2.setText("Suche Pacman");
        textField2.setForeground(Color.black);
        picture.setBackground(Color.darkGray);
      	break;
      }
      case 1:
      {
        textField2.setText("Verfolge Pacman !!!");
        Color darkRed = new Color(203, 50, 37);
        textField2.setForeground(darkRed);
        picture.setBackground(darkRed);
        break;
      }
      case 2:
      {
        textField2.setText("Broadcast Position + Verfolgung");
        textField2.setForeground(Color.black);
        picture.setBackground(Color.darkGray);
	break;
      }
      case 9:
      {
        textField2.setText("auf der Flucht");
        Color darkBlue = new Color(25, 36, 150);
        textField2.setForeground(darkBlue);
        picture.setBackground(darkBlue);
	break;
      }
      default:
      {
        textField2.setText("undefinierter Zustand");
        textField2.setForeground(Color.black);
        picture.setBackground(Color.darkGray);
	break;
      }
    } // end of switch

    return;
  }

  /**
   * liefert die aktuelle Suchtiefe des Monsters
   *
   * @return Suchtiefe als Integer-Wert
   */
  public int getValueSearchDepth()
  {
    return search.getValue();
  }

  /**
   * liefert den aktuellen Kommunikationsradius des Monsters
   *
   * @return Kommunikationsradius als Integer-Wert
   */
  public int getValueCom()
  {
    return com.getValue();
  }

  /**
   * liefert die aktuelle Intelligenz, d.h Strategie des Monsters
   *
   * @return Intelligenz als Integer-Wert
   */
  public int getValueAI()
  {
    return ai.getValue();
  }

  /**
   * liefert die Geschwindigkeit (Laufzeit) des
   * aktuellen Monsters; bisher noch nicht ueber
   * Slider einstellbar
   *
   * @return Geschwindigkeit des Mosnters als float-Wert
   */
  public float getValueSpeed()
  {
    return monsterSpeed;
  }

  /**
   * setzt die Startposition des Monsters
   */
  public void setStartPosition(Point3i startPosition)
  {
    this.startPosition = startPosition;
  }

  /**
   * setzt die MonsterStrategie
   *
   * @param strategy    die zu setzende Strategie
   */
  public void setStrategy(int strategy)
  {
    ai.setValue(strategy);
  }



  /**
   * aktualisiert die Com-Ausgabe, d.h. zeigt
   * gesendete und empfangene Nachrichten an/von anderen Monstern
   *
   * @param string String mit der Nachricht; wird geschrieben in TextField1
   */
  public void writeVoice(String string)
  {
    textField1.setText(string);
  }

  /**
   * setzt den Status des Monsters und gibt ihn zurck
   *
   * @param status Status des Monsters als Integerwert
   * @return status Status des Monsters als Integerwert
   */
  public void setMonsterStatus(int status)
  {
    this.monsterStatus = status;

    return;
  }
}