/*
 * MonsterResponse.java
 */

package pacman3d.monster;

import java.util.*;
import javax.vecmath.*;

import pacman3d.message.ID;

/**
 * <p>Die MonsterResponse stoesst die Berechnung von neuen Pfaden aufgrund von
 * unterschiedlichen, einstellbaren Strategien an.</p>
 *
 * <p>In der vorliegenden Version des Monster-Agenten-Modells basiert die
 * Strategie-Berechnung auf der Bestimmung einer neuen Zielpositionen und der
 * Berechnung eines kuerzesten Weges dorthin.</p>
 *
 * <p>Erweiterungen koennten zum Beispiel in einer staerker dynamischen und damit
 * flexibleren Strategie-Berechnung bestehen.</p>
 */
public class MonsterResponse
{
  /** Referenz auf das Monster-Objekt */
  private Monster monster;

  /** Referenz auf MonsterControl */
  private MonsterControl control;

  /** Referenz auf MonsterAwareness */
  private MonsterAwareness awareness;

  /** Referenz auf MonsterVoice */
  private MonsterVoice voice;

  /** eine jeden Fall nicht erreichbare Zelle, damit Breitensuche nicht
   *  abbricht */
  private static Point3i NOWAY = new Point3i(-1, -1, -1);

  /** wurde Pacman gefangen? ja/nein */
  private boolean pacmanReached = false;

  /** gibt die ID des zuletzt getoeteten Pacman an */
  private ID lastPacmanKilled = null;

  /** Zeitpunkt, zu dem der letzte Pacman getoetet wurde */
  private long killingTime = 0;

  /** Schonfrist in ms, die dem Pacman eingerumt wird, bevor ihm eine weitere
   *  die-Message zugestellt werden kann.
   */
  private long gracePeriod = 20000;

  /** die ID des naehesten Pacman */
  private ID nextPacman = null;

  /**
   * erzeugt eine neue MonsterResponse
   *
   * @param monster  eine Referenz auf das uebergeordnete Monster-Objekt
   */
  public MonsterResponse(Monster monster)
  {
    this.monster   = monster;

    // fuer Abwaertskompatibilitaet
    this.control   = monster.getControl();
    this.awareness = monster.getAwareness();
    this.voice     = monster.getVoice();
  }

  // Morpheus: "There's a difference between knowing the path and walking the path"...

  /**
   * diese Methode wird zyklisch in jedem GameLoop aufgerufen und stoesst
   * bei Bedarf (wenn das aktuelle Ziel erreicht ist) eine neue Pfadberechnung an.
   *
   * @param pacmanIDs  Liste mit den IDs aller Pacmen
   */
  public void whatAmIThinking(ID[] pacmanIDs)
  {
    // sind wir am Ziel? wenn ja: neuen Pfad berechnen
    if (control.didWeReachTheEnd())
    {
      setNewPath(pacmanIDs);
    }

    // So let's walk the path!
    control.walkThePath(true);
  }

  /**
   * wurde der Pacman gefangen, wird das Flag pacmanReached auf true gesetzt;
   * fraglich bleibt, ob der Pacman auch gettet wurde; schliesslich koennte er
   * ja einen Schild tragen, auf Monster-Jagd sein oder aehnliches...
   *
   * @param pacmanId  Id des Pacman, der als letztes "die" geschickt bekommen
   *                  hat.
   */
  public void setPacmanReached(ID pacmanId)
  {
    pacmanReached = true;

    lastPacmanKilled = pacmanId;
    killingTime = System.currentTimeMillis();
  }

  /**
   * berechnet einen neuen Pfad und erledigt alle anfallenden Arbeiten
   * in der MonsterControl; ruft getNewDestination() auf, um ein neues Ziel
   * fuer das Monster zu suchen.
   *
   * @param pacmanIDs  Liste mit den IDs aller Pacmen
   */
  private void setNewPath(ID[] pacmanIDs)
  {
    int searchDepth = awareness.getDepth();
    Point3i monsterPos = new Point3i(control.getMonsterPosition());

    // einen Suchbaum fuer den eingestellten Aktionsradius holen...
    MonsterSearchTree searchTree = control.doBFS(monsterPos, NOWAY, searchDepth);

    // ...neues Ziel in diesem Suchbaum bestimmen...
    Point3i monsterDestination = getNewDestination(searchTree, pacmanIDs);

    // ...den Pfad dorthin holen...
    LinkedList searchPath = searchTree.getParents(monsterDestination);

    // ...und in der MonsterControl ablegen.
    control.setSearchPath(searchPath);
    control.newPathComputed();

    return;
  }

  /**
   * bestimmt ein neues Ziel fuer das Monster aufgrund unterschiedlicher
   * Strategien; diese lassen sich ueber das MonsterSetup auswaehlen
   *
   * @param monsterTree  Breitensuchbaum, in dem ein neues Ziel bestimmt werden soll
   * @param pacmanIDs    Liste mit den IDs aller Pacmen
   *
   * @return             neues Ziel, das das Monster ansteuern wird
   */
  private Point3i getNewDestination(MonsterSearchTree monsterTree, ID[] pacmanIDs)
  {
    int strategySelection = awareness.getAIFromSetup();
    Point3i newDestination = new Point3i();

    // sollen wir fluechten??
    if (monster.getControl().getEscapeState() == true)
    {
      monster.getControl().setEscapeComputed(true);
      strategySelection = 5;
      // System.out.println("auf der Flucht...");
    }

    switch (strategySelection)
    {
      case 1: // zufaellige Bewegung
      {
        newDestination = doStrategy1(monsterTree);
        break;
      }
      case 2: // Pacman suchen, keine Kooperation
      {
        newDestination = doStrategy2(monsterTree, pacmanIDs);
        break;
      }
      case 3: // Pacman suchen, mit Kooperation
      {
        newDestination = doStrategy3(monsterTree, pacmanIDs);
        break;
      }
      case 4: // zufaellige Bewegung, mit Kooperation
      {
        newDestination = doStrategy4(monsterTree);
        break;
      }
      case 5: // Test der Fluchtbewegung
      {
        newDestination = doStrategy5(monsterTree, pacmanIDs);
        break;
      }
      default:
      {
        System.out.println("unbekannte Strategie");
        newDestination = doStrategy1(monsterTree);
        break;
      }
    } // end of switch

    return (newDestination);
  }

  /**
   * Strategie 1:
   * ===================================================================
   * - zufaellige Zelle ansteuern
   *
   * @param monsterTree   der Breitensuchbaum
   *
   * @return              neues Ziel des Monsters
   */
  private Point3i doStrategy1(MonsterSearchTree monsterTree)
  {
    Point3i newDestination = new Point3i();

    MonsterSearchTree tempTree = monsterTree.getRandomNode(monsterTree);
    newDestination = tempTree.getNodeInfo();

    awareness.setMonsterStatusInSetup(0);

    return newDestination;
  }

  /**
   * Strategie 2:
   * ===================================================================
   * - Pacman suchen
   * - falls gefunden, hinlaufen
   * - ansonsten: zufllige Zelle ansteuern
   *
   * @param monsterTree   der Breitensuchbaum
   *
   * @return              neues Ziel des Monsters
   */
  private Point3i doStrategy2(MonsterSearchTree monsterTree, ID[] pacmanIDs)
  {
    Point3i newDestination = new Point3i();

    MonsterSearchTree tempTree = monsterTree.lookup(getClosestPacmanPosition(monsterTree, pacmanIDs));

    if (tempTree != null) // Pacman ist im Suchbaum (hoho...)
    {
      if (lastPacmanKilled == nextPacman)
      {
        // diesen Pacman haben wir zuletzt getoetet
        if ((System.currentTimeMillis() - killingTime) < gracePeriod)
        {
          // System.out.println("Schonfrist ist noch nicht vorueber!");

          // zufaellige Zelle anlaufen
          newDestination = doStrategy1(monsterTree);
          awareness.setMonsterStatusInSetup(0);
        }
        else
        {
          // System.out.println("Schonfrist ist vorueber (har har)!");

          // auf ihn mit Gebruell
          newDestination = tempTree.getNodeInfo();
          awareness.setMonsterStatusInSetup(1);
        }
      }
      else // wir haben einen neuen Pacman gefunden
      {
        newDestination = tempTree.getNodeInfo();
        awareness.setMonsterStatusInSetup(1);
      }
    }
    else // Pacman nicht gefunden: zufaellige Zelle ansteuern
    {
      newDestination = doStrategy1(monsterTree);
      awareness.setMonsterStatusInSetup(0);
    }

    return newDestination;
  }

  /**
   * Strategie 3:
   * ===================================================================
   * wie Strategie 2 aber mit Kooperation
   * - ist PacMan in Suchbaum, wird seine Position gebroadcastet
   * - falls nicht: zufaellige Bewegung
   *
   * @param monsterTree   der Breitensuchbaum
   *
   * @return              neues Ziel des Monsters
   */
  private Point3i doStrategy3(MonsterSearchTree monsterTree, ID[] pacmanIDs)
  {
    Point3i newDestination = new Point3i();

    MonsterSearchTree tempTree = monsterTree.lookup(getClosestPacmanPosition(monsterTree, pacmanIDs));

    if (tempTree != null) // Pacman gefunden
    {
      newDestination = tempTree.getNodeInfo();

      // PacmanPosition posten
      MonsterMessage message = new MonsterMessage();
      message.setPacmanPosition(newDestination);
      voice.sendMessage(message);
      awareness.setMonsterStatusInSetup(2);
    }
    else
    {
      newDestination = doStrategy1(monsterTree);
      awareness.setMonsterStatusInSetup(0);
    }

    return newDestination;
  }

  /**
   * Strategie 4:
   * ===================================================================
   * wie Strategie 1 aber mit Kooperation
   * - falls in Message-Queue pacmanPosition steht:
   *   newDestination = pacmanPosition
   * - ansonsten weiter zufaellige Bewegung
   */
  private Point3i doStrategy4(MonsterSearchTree monsterTree)
  {
    Point3i newDestination = new Point3i();

    // steht etwas in der Message-Queue??
    MonsterMessage message = voice.lookupQueue();

    if (message != null)
    {
      newDestination = message.getPacmanPosition();

      // pacmanPosition im Suchbaum?
      if (monsterTree.lookup(newDestination) != null)
      {
        awareness.setMonsterStatusInSetup(1);
      }
      else
      {
        newDestination = doStrategy1(monsterTree);
      }
    }
    else
    {
      newDestination = doStrategy1(monsterTree);
    }

    return newDestination;
  }

  /**
   * Strategie 5:
   * ===================================================================
   * Flucht, nichts wie weg hier...
   *
   * @param monsterTree   der Breitensuchbaum
   *
   * @return              neues Ziel des Monsters
   */
  private Point3i doStrategy5(MonsterSearchTree monsterTree, ID[] pacmanIDs)
  {
    Point3i newDestination = null;

    MonsterSearchTree tempTree = monsterTree.lookup(getClosestPacmanPosition(monsterTree, pacmanIDs));

    if (tempTree != null) // Pacman ist im Suchbaum (oops...)
    {
      Point3i pacmanPosition = tempTree.getNodeInfo();

      // Wie weit ist er denn weg?
      int distance = monsterTree.lookup(pacmanPosition).getTreeDepth();
      int fluchtWeite = 1;

      int searchDepth = awareness.getDepth() + fluchtWeite;
      MonsterSearchTree pacmanTree = control.doBFS(pacmanPosition, NOWAY, searchDepth);

      // Ein kleiner Schritt fuer ein Monster, ...
      MonsterSearchTree escapeTree = control.doBFS(monsterTree.getNodeInfo(), NOWAY, fluchtWeite);

      // ...ein moeglichst grosser fuer einen Pacman!
      LinkedList possibleDestinations = escapeTree.getNodeChildren();

      for (Iterator i = possibleDestinations.iterator(); i.hasNext(); )
      {
        MonsterSearchTree currentTree = (MonsterSearchTree) i.next();
        Point3i tempDestination = (Point3i) currentTree.getNodeInfo();
        MonsterSearchTree pacmanSubTree = pacmanTree.lookup(tempDestination);

        if (pacmanSubTree == null) continue; // Ziel ist nicht in Pacman-Suchbaum

        int tempDistance = pacmanSubTree.getTreeDepth();

        if (tempDistance > distance)
        {
          newDestination = tempDestination;
          break;
        }
      }
      if (newDestination == null)
      {
        // wir stecken ziemlich in der Klemme: hier ist ggf. noch ein wenig mehr Grips noetig
        // System.out.println("kein neuer Pfad");

        newDestination = monsterTree.getNodeInfo();
      }
    }
    else // Pacman nicht gefunden: wir bleiben (erstmal), wo wir sind...
    {
      // System.out.println("kein Pacman gefunden");
      newDestination = monsterTree.getNodeInfo();

    }
    awareness.setMonsterStatusInSetup(9);

    return newDestination;
  }

  /**
   * liefert die Position des naechstgelegenen Pacman aus der Liste aller Pacman
   *
   * @param monsterTree    Breitensuchbaum
   * @param pacmanIDs      Liste der Pacmen
   * @param nextPacman     hier wird die ID des nchsten Pacman hineingeschrieben
   *
   * @return               Position des naehesten Pacman
   */
  private Point3i getClosestPacmanPosition(MonsterSearchTree monsterTree, ID[] pacmanIDs)
  {
    int i = 0;
    int tempTiefe = 20;
    int tiefe;

    Point3i pacmanPosition = new Point3i();
    Point3i returnPoint = null;
    MonsterSearchTree searchTree = null;

    getHim:
    {
      while (true)
      {
        try
        {
          pacmanPosition = control.getPacmanPosFromChannel(pacmanIDs[i]);
          nextPacman = pacmanIDs[i];
        }
        catch (ArrayIndexOutOfBoundsException e)
        {
          break getHim;
        }

        searchTree = monsterTree.lookup(pacmanPosition);

        if (searchTree != null)
        {
          tiefe = searchTree.getTreeDepth();
          if (tiefe <= tempTiefe)
          {
            returnPoint = searchTree.getNodeInfo();
            tempTiefe = tiefe;
          }
        }
        i++;
      }
    }

    return returnPoint;
  }
}