/*
 * MonsterChannel.java
 */

package pacman3d.monster;

import java.util.*;
import java.io.*;
import java.net.URL;

import javax.vecmath.*;
import javax.media.j3d.*;

import pacman3d.message.*;
import pacman3d.util.*;
import pacman3d.labyrinth.Labyrinth;
import pacman3d.labyrinth.Position;
import pacman3d.labyrinth.cells.*;//Cell;

import org.w3c.dom.*;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.DocumentBuilder;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

/**
 * <p>Der MonsterChannel ist die zentrale Schnittstelle der Monster zur Aussenwelt.</p>
 *
 * <p>Hier sind die Schnittstellen zum Labyrinth gekapselt, die von den anderen
 * Monster-Klassen benutzt werden.</p>
 *
 * <p>Der MonsterChannel ist der Nachrichten-Kanal fuer die interne
 * Monster-Kommunikation (nicht zu verwechseln mit dem MessageService,
 * z. B. fuer die Kommunikation zwischen Monster und Pacman).</p>
 *
 * <p>Im MonsterChannel werden die Monster erzeugt und die makeLoop-Methode zur Verfgung
 * gestellt; ueber den MonsterChannel kann der MonsterSzeneGraph abgefragt werden.</p>
 */
public class MonsterChannel implements
       pacman3d.message.MessageListener,
       pacman3d.util.Identifiable,
       pacman3d.util.XMLSerializeable
{
  /** Referenz auf das Labyrinth-Objekt */
  private static Labyrinth theLaby;

  /** die URL des zu ladenden Level-Files
   *  ...sollte von Intro/Game uebergeben bzw. abgefragt werden koennen...
   */
  private static URL levelURL = null;

  /** Liste mit allen angemeldeten Monstern */
  private LinkedList allMonsters = new LinkedList();

  /** Liste mit MonsterProperty-Objekten: enthaelt die Eigenschaften
   *  der einzelnen Monster; wird aus Level-File geladen
   */
  private LinkedList monsterPropertiesList = new LinkedList();

  /** Liste mit allen Startpositionen */
  private LinkedList initPositions = new LinkedList();

  /** beinhaltet den Monster-Teil des Szenegraphen */
  private BranchGroup monsterGraph = new BranchGroup();

  /** die ID des MonsterChannel */
  private ID nextID = new ID();

  /** Referenz auf den Message-Service zum Empfangen und Senden von Nachrichten */
  private static MessageService messageService;

  /** Queue fuer eingehende Nachrichten */
  private Vector monsterMessageQueue = new Vector();

  /** Queue fuer das Versenden der Position und Blickrichtung */
  private LinkedList positionMessage = new LinkedList();

  /** zhlt vergangene Ticks bis zum Updaten der Monster-Positionen auf den Clients */
  static private int messageUpdateTicks = 0;

  /** setzt den Schwellwert fr die zum Update der Position ntigen Ticks */
  static private int messageUpdateBarrier = 10;

  /** der Sound fuer das Monster */
  static private PointSound monsterSound;

  /** der Sound, der bei PacmanReached abgespielt wird */
  static private PointSound munchSound;


  /**
   * erzeugt einen neuen MonsterChannel; es werden soviele Monster erzeugt,
   * wie Startpositionen uebergeben wurden
   *
   * @param theLabyrinth  Referenz auf Labyrinth
   * @param initPos       die Startpositionen des Monsters
   */
  public MonsterChannel(Labyrinth theLabyrinth, LinkedList initPos)
  {
    // Message-Service holen
    messageService = MessageService.getInstance();
    messageService.addMessageListener(this);

    this.theLaby = theLabyrinth;
    this.initPositions = initPos;

    Monster tempMonster;
    for (int i = 0; i < this.initPositions.size(); i++)
    {
      // neues Monster mit eigener ID auf seiner Startposition erzeugen
      Point3i tempPosition = (Point3i) initPositions.get(i);

      if (theLaby.isServer() || !theLaby.isMultiplayerGame()) tempMonster = new Monster(new ID(), tempPosition, this);
      else tempMonster = new Monster(new ID(), tempPosition, this, true);
      // Monster in MonsterListe einfuegen
      allMonsters.addLast(tempMonster);

      // Eigenschaften der MonsterBrachGroup setzen
      monsterGraph.setCapability(BranchGroup.ALLOW_CHILDREN_READ);
      monsterGraph.setCapability(BranchGroup.ALLOW_CHILDREN_WRITE);
      monsterGraph.setCapability(BranchGroup.ALLOW_DETACH);
      monsterGraph.setCapability(BranchGroup.ALLOW_CHILDREN_READ);
      monsterGraph.addChild(tempMonster.getView());
    }



    // ??? wofuer ???
    initPositions.clear();
  } // end of constructor

  /**
   * erzeugt einen neuen MonsterChannel
   *
   * !!! EINLESEN DER MONSTER-PARAMETER AUS LEVEL-FILE !!!
   *
   * @param theLabyrinth  Referenz auf Labyrinth
   */
  public MonsterChannel(Labyrinth theLabyrinth, URL levelURL)
  {
    // Einlesen der Monster-Infos aus Level-Datei
    try
    {
      loadMonsters(levelURL);
    }
    catch (Exception ex)
    {
      Debug.out (this.getClass().getName(), ex, Debug.LEVEL_CRITICAL);
    }

    // Message-Service holen
    messageService = MessageService.getInstance();
    messageService.addMessageListener(this);

    this.theLaby = theLabyrinth;
    // wird aus XML-File gelesen...
    // this.initPositions = initPos;

    Monster tempMonster;
    for (int i = 0; i < this.initPositions.size(); i++)
    {
      // neues Monster mit eigener ID auf seiner Startposition erzeugen
      Point3i tempPosition = (Point3i) initPositions.get(i);
      MonsterProperties currentProperty = (MonsterProperties) monsterPropertiesList.get(i);

      if (theLaby.isServer() || !theLaby.isMultiplayerGame()) tempMonster = new Monster(new ID(), currentProperty, this);
      else tempMonster = new Monster(new ID(), tempPosition, this, true);
      // Monster in MonsterListe einfuegen
      allMonsters.addLast(tempMonster);

      // Eigenschaften der MonsterBranchGroup setzen
      monsterGraph.setCapability(BranchGroup.ALLOW_CHILDREN_READ);
      monsterGraph.setCapability(BranchGroup.ALLOW_CHILDREN_WRITE);
      monsterGraph.setCapability(BranchGroup.ALLOW_DETACH);
      monsterGraph.setCapability(BranchGroup.ALLOW_CHILDREN_READ);
      monsterGraph.addChild(tempMonster.getView());
    }
    //Sounds in SceneGraph einfgen
    monsterSound = SoundBg("file:./media/sound/ghost2.wav", monsterGraph);
    munchSound = SoundBg("file:./media/sound/crunches.wav", monsterGraph);

  } // end of constructor

  /**
   * laedt die Monster-Elemente aus dem Level-File
   *
   * !!! BISHER NUR EINLESEN DER STARTPOSITIONEN !!!
   *
   * @param levelFileURL      URL des einzulesenden Level-Files
   *
   */
  public void loadMonsters (URL levelFileURL) throws
                                pacman3d.util.InvalidLevelFileException,
                                java.io.IOException,
                                pacman3d.util.InvalidLevelFileException,
                                Exception
  {
    // Stream zum Einlesen der Level-Datei
    InputStream oLevelStream = null;

    try
    {
      oLevelStream = levelFileURL.openStream();
    }
    catch (Exception ex)
    {
      Debug.out (this.getClass().getName(), ex, Debug.LEVEL_CRITICAL);
      throw ex;
    }

    // ==== Code entsprechend Labyrinth-Gruppe ====

    Document doc;

    try
    {
      // Erzeugen des DOM-Parsers und Einlesen des XML-Stroms
      DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
      DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
      doc = docBuilder.parse (oLevelStream);

      // normalize text representation
      doc.getDocumentElement ().normalize ();
      Element elDoc = doc.getDocumentElement();

      // <levelstructure>-Element suchen
      Element elLevelStructure = XmlUtils.getChildElement (elDoc, LevelXmlSyntax.TAG_LEVELSTRUCTURE);

      // <Monsters>-Element suchen und loadFromDOM-Methode uebergeben
      Element elMonsters = XmlUtils.getChildElement (elLevelStructure, LevelXmlSyntax.TAG_MONSTERS);
      loadFromDOM(elMonsters);
    }
    catch (SAXException e)
    {
      Exception x = e.getException ();
      ((x == null) ? e : x).printStackTrace ();

      throw new pacman3d.util.InvalidLevelFileException();
    }
    catch (Throwable t)
    {
      t.printStackTrace ();

      throw new pacman3d.util.InvalidLevelFileException();
    }
  }

  /**
   * Hauptschleife fuer alle Monster; wird im GameLoop aufgerufen. makeLoop ruft
   * fuer jedes Monster die update-Methode auf, in der dann die eigentlichen Monster-
   * Berechnungen durchgefuehrt werden.
   *
   * @param     deltaTicks die Ticks, die den Monstern fuer ihre
   *            Bewegung zugestanden wird
   *
   */
  public void makeLoop(int deltaTicks)
  {
    //messageUpdateTicks = deltaTicks;

    if (theLaby.isServer() || !theLaby.isMultiplayerGame())
    {
      // Schleife ueber alle Monster
      for (int i = 0; i < allMonsters.size(); i++)
      {
        // ??? muss das wirklich hier stehen, wieso kann das nicht aus der Schleife
        //     raus ???

          ID pacmanIDs[] = getAllPacmanIdsFromLaby();

          // Update-Methode des Monsters aufrufen
          Monster tempM = (Monster) allMonsters.get(i);
          tempM.update(deltaTicks, pacmanIDs);

      }//endof for (int i = 0; i < allMonsters.size(); i++)

      if(theLaby.isServer() && theLaby.isMultiplayerGame())
      {
        messageUpdateTicks += deltaTicks;
        if (messageUpdateTicks >= messageUpdateBarrier)
          {
            messageUpdateTicks = 0;
        try
          {
            Message updatePositionForClients = new Message(nextID);
            updatePositionForClients.addReceiverGroup(1);
            updatePositionForClients.setContent(positionMessage);
            messageService.sendMessage(updatePositionForClients);
            positionMessage.clear();
          }
          catch (pacman3d.message.MissingMessageSenderException ex)
          {
            System.out.println("Fehler beim Senden an andere Channel...........");
          }
          }
      }
    }//endof if (theLaby.isServer())

  // hier holen wir uns die aktuellen Positionen der Monster aus der Message des Master-Channel
  if(!theLaby.isServer() && theLaby.isMultiplayerGame())
    {
      if (!monsterMessageQueue.isEmpty())
      {
        //System.out.println("hallo, ich bin beim Empfang");
        Message testMessage = (Message) (monsterMessageQueue.lastElement());
        if (testMessage.getContent() instanceof LinkedList) {
                positionMessage = (LinkedList) testMessage.getContent();
                Monster tempMonster;// = new Monster()
                Vector4f tempVector = new Vector4f();
                for (int i = 0; i < allMonsters.size(); i++)
                {
                  tempVector = (Vector4f) positionMessage.get(i);
                  tempMonster = (Monster) allMonsters.get(i);
                  tempMonster.getControl().setClientPosition ((Vector4f) tempVector);

                }// end of for (int i = 0; i < allMonsters.size(); i++)
        } else {
                Debug.out( getClass().getName(),
                        "cannot handle message: " + testMessage.toString(),
                        Debug.LEVEL_WARNING );
        }
        monsterMessageQueue.clear();

      }// end of if (!monsterMessageQueue.isEmpty())
    }//end of else
  } // end of makeLoop

  /**
   * liefert die Position des Monsters mit ID tempID in der Liste allMonsters
   */
  public Integer getMonsterNumber (ID tempID)
  {
    int i = allMonsters.indexOf(getMonsterFromID(tempID));
    Integer temp = new Integer(i);
    return temp;
  }
  /**
   * liefert den Teil des Szenegraphen, in dem sich Monster befinden; wird vom
   * Game aufgerufen
   *
   * @return BranchGroup mit allen Monstern
   */
  public BranchGroup getJ3DNode()
  {
    // ??? duerfen wir an dieser Stelle noch nicht compilen ???
    // monsterGraph.compile();
    // System.out.println("SceneGraph bergebenm" + monsterGraph.toString());


    return monsterGraph;
  }

  // ??? sollen Nachrichten vom Pacman (und Positionsangaben an Slave-Monster)
  //     den MonsterChannel umgehen? - Dann ist die folgende Methode ggf. in die
  //     MonsterControl zu verlegen... ???

  /**
   * empfaengt eine Nachricht von der Poststelle des Labyrinths und
   * schreibt diese in die Message-Queue. Kann fuer die Kommunikation mit den
   * Pacmen und die Uebermittlung von Positionsinformationen ueber das Netz
   * benutzt werden.
   *
   * @param monsterMessage das MessageObject, das empfangen wurde
   */
  public void getMessage(Message monsterMessage)
  {
    //Debug.out(this.getClass().getName(), "Nachricht erhalten", Debug.LEVEL_NOTICE);

    if (monsterMessage.getContent() instanceof Integer)
    {
      if (theLaby.isServer() || !theLaby.isMultiplayerGame())
      {
        Integer tempInteger = (Integer) monsterMessage.getContent();
        int i = tempInteger.intValue();
        Monster tempMonster = (Monster) allMonsters.get(i);
        tempMonster.getControl().moveToInitPosition();
      }
    }

    if (monsterMessage.getContent() instanceof String)
    {
      if (!theLaby.isServer() && theLaby.isMultiplayerGame())
      {
        String string = (String) monsterMessage.getContent();
        if (string.equals("playMunchSound"))
        {
          //System.out.println("Munch Sound sollte jetzt zu hren sein");
          PlaySoundPacmanReached();
        }
        //else System.out.println("String erkannt, aber nicht -playMunchSound-");
      }
      //System.out.println("St, aber nicht -playMunchSound-");

    }
    else  monsterMessageQueue.add(monsterMessage);
  }

  /**
   * Methode, um den aktuellen Monster-Zustand zu speichern
   *
   * @param    oDoc
   *
   * @return   Element
   */
  public org.w3c.dom.Element saveToDOM (org.w3c.dom.Document oDoc)
  {
    return null;
  }

  /**
   * laedt die im Level-File abgelegten Monster-Informationen; fuer jedes
   * Monster werden die folgenden Attribute gelesen:
   *
   * - Startpositionen
   * - (weitere Attribute: TODO...)
   *
   * @param oElement
   */
  public void loadFromDOM (org.w3c.dom.Element elMonsters) throws
                                               java.lang.ClassNotFoundException,
                                               XMLMissingAttributeException
  {
    NodeList oMonsters = elMonsters.getElementsByTagName(LevelXmlSyntax.TAG_MONSTER);

    // Vector fuer Startpositionen (dirty: nicht mehr wirklich ein member...)
    Vector m_vecMonsterStartPositions = new Vector();

    if (oMonsters.getLength() > 0)
    {
      for (int iCurrentMonsterNo = 0; iCurrentMonsterNo < oMonsters.getLength(); iCurrentMonsterNo++)
      {
        // naechstes <Monster>-Element behandeln
        Element elCurrentMonster = (Element)(oMonsters.item(iCurrentMonsterNo));

        // Monster-Eigenschaften werden in MonsterProperty-Objekt abgelegt
        MonsterProperties monsterEigenschaften = new MonsterProperties(theLaby);

        if (elCurrentMonster != null)
        {
          // Startposition lesen
          Point3i startPos = getStartPosition(elCurrentMonster);
          this.initPositions.add (startPos);
          monsterEigenschaften.setMonsterPosition(startPos);
          monsterEigenschaften.setMonsterStrategy(getStrategy(elCurrentMonster));
          monsterEigenschaften.setMonsterWindowVisibility(getWindowVisibility(elCurrentMonster));
          monsterEigenschaften.setMonsterModel(getModel(elCurrentMonster));

          // wenn alle Attribute gelesen wurden: monsterEigenschaften in Liste schreiben
          monsterPropertiesList.add(monsterEigenschaften);

        }
      } // end of for
    }
  }

  /**
   * liest die Startposition fuer ein Monster
   *
   * @param monsterElement  Monster-Element, aus dem die Startposition gelesen
   *                        werden soll.
   *
   * @return                Startposition fuer das uebergebene Monster-Element
   */

  private Point3i getStartPosition(org.w3c.dom.Element monsterElement) throws
                                               java.lang.ClassNotFoundException,
                                               XMLMissingAttributeException
  {
    // ==== Code entsprechend Labyrinth-Gruppe ====

    Point3i oStartPosition = null;

    Element elStartPosition = XmlUtils.getChildElement (monsterElement, LevelXmlSyntax.TAG_STARTPOSITION);

    int iX = 0, iY = 0, iZ = 0;

    if (elStartPosition != null)
    {
      try
      {
        String sX = elStartPosition.getAttribute(LevelXmlSyntax.ATTR_X);
        if (sX != null) { iX = new Integer(sX).intValue(); }
        String sY = elStartPosition.getAttribute(LevelXmlSyntax.ATTR_Y);

        if (sY != null) { iY = new Integer(sY).intValue(); }
        String sZ = elStartPosition.getAttribute(LevelXmlSyntax.ATTR_Z);

        if (sZ != null) { iZ = new Integer(sZ).intValue(); }
        oStartPosition = new Point3i(iX, iY, iZ);
      }
      catch (java.lang.NumberFormatException ex)
      {
        Debug.out (this.getClass().getName(), "keine Zahlen in den x/y/z-Attributen vorgefunden.", Debug.LEVEL_WARNING);
      }
    }

    return (oStartPosition);
  }

  /**
   *
   */
  private int getStrategy(org.w3c.dom.Element monsterElement) throws
                                               java.lang.ClassNotFoundException,
                                               XMLMissingAttributeException
  {
    String strategieText = XmlUtils.getTextFromChildElement(monsterElement, LevelXmlSyntax.TAG_MONSTER_STRATEGIE);

    int monsterStrategie = 0;

    if (strategieText.equals("zufall"))
    {
      monsterStrategie = 1;
    }
    if (strategieText.equals("jagd"))
    {
      monsterStrategie = 2;
    }
    if (strategieText.equals("senden"))
    {
      monsterStrategie = 3;
    }
    if (strategieText.equals("empfangen"))
    {
      monsterStrategie = 4;
    }
    if (strategieText.equals("flucht"))
    {
      monsterStrategie = 5;
    }

    // immer noch keine Strategie gefunden? -> default setzen
    if (monsterStrategie == 0)
    {
      Debug.out(this.getClass().getName(), "keine gltige Monster-Strategie gefunden, setze Default", Debug.LEVEL_NOTICE);
      monsterStrategie = 1;
    }
    return (monsterStrategie);
  }

  /**
   * liest das im Level-File definierte Monster-Modell ein, das geladen werden soll
   *
   * @param monsterElement     Referenz auf das <monster>-Element im DOM
   */
  private int getModel(org.w3c.dom.Element monsterElement) throws
                                               java.lang.ClassNotFoundException,
                                               XMLMissingAttributeException
  {
    String modelText = XmlUtils.getTextFromChildElement(monsterElement, LevelXmlSyntax.TAG_MONSTER_MODELL);

    int monsterModell = 0;

    if (modelText.equals("kodos")) monsterModell = 1;

    if (modelText.equals("ghost")) monsterModell = 2;

    if (modelText.equals("pacborg")) monsterModell = 3;

    // immer noch kein Modell gefunden? -> default setzen
    if (monsterModell == 0)
    {
      Debug.out(this.getClass().getName(), "kein gltiges Monster-Modell gefunden, setze Default", Debug.LEVEL_NOTICE);
      monsterModell = 2;
    }
    return (monsterModell);
  }

  /**
   * fragt ab, ob das MonsterSetup-Fenster angezeigt werden soll
   *
   * @param monsterElement     Referenz auf das <monster>-Element im DOM
   */
  private boolean getWindowVisibility(org.w3c.dom.Element monsterElement) throws
                                               java.lang.ClassNotFoundException,
                                               XMLMissingAttributeException
  {
    String jFrameText = XmlUtils.getTextFromChildElement(monsterElement, LevelXmlSyntax.TAG_MONSTER_SETUP_FENSTER);

    boolean anzeigen = false;

    if (jFrameText.equals("anzeigen"))
    {
      anzeigen = true;
    }

    return (anzeigen);
  }

  /**
   * liefert die ID des MonsterChannels
   *
   * @return MonsterChannelID
   */
  public ID getID()
  {
    return this.nextID;
  }

  /**
   * liefert die Anzahl der aktiven Pacmen
   *
   * @return   Pacman-Anzahl
   *
   * @see      MonsterControl#getPacmanCountFromChannel
   */
  public int getPacmanCountFromLaby()
  {
    return theLaby.getPacmanCount();
  }

  /**
   * liefert eine Liste mit den IDs aller aktiven Pacmen
   *
   * @return  Pacman-ID-Liste
   */
  public ID[] getAllPacmanIdsFromLaby()
  {
    return theLaby.getAllPacmanIds();
  }

  /**
   * liefert eine Referenz auf das Labyrinth
   *
   * @return  Labyrinth-Referenz
   */
  public Labyrinth getLabyrinth()
  {
    return this.theLaby;
  }

  /**
   * liefert die Position des Pacman mit der angegebenen ID
   *
   * @param pacmanID   ID des Pacman, dessen Position abgefragt werden soll
   *
   * @return           Pacman-Position
   *
   * @see              MonsterControl#getPacmanPosFromChannel
   */
  public Point3i getPacmanPosFromLaby(ID pacmanID)
  {
    return theLaby.getPacmanPos(pacmanID);
  }

  /**
   * setzt das Monster mit der angegebenen ID auf die gewuenschte Position
   *
   * @param id         die ID des zu setzenden Monsters
   * @param position   die gewunschte Position
   */
  public void setMonsterPosInLaby(ID id, Point3i position)
  {
    try
    {
      theLaby.setMonsterPos(id, position);
    }
    // wir wissen, wo's langgeht: sollte nicht auftreten...
    catch (Exception ex)
    {
      System.out.println (ex);
      ex.printStackTrace();
    }
  }

  /**
   * sendet dem Objekt mit der ID <code>receiverID</code> die Textnachricht
   * <code>messageText</code>
   *
   * @param receiverID       ID des Empfaengers
   * @param messageText      die zu sendende Textnachricht
   */
  public void sendMessage(ID receiverID, String messageText)
  {
    Message message = new Message(nextID, messageText, receiverID);

    try
    {
      messageService.sendMessage(message);
    }
    catch(Exception e)
    {
    }
    //die folgende wird noch nicht benutzt, da die Response noch nicht
    //ber die passende Methode zur Auswertung einer solchen Nachricht verfgt
    //monster.getResponse().setPacmanReached(true);
  }


  /**
   * meldet das Monster mit der angegebenen ID im Labyrinth an
   *
   * @param id       id des uebergebenen Monsters
   * @param monster  MonsterObjekt, das dem Labyrinth uebergeben wird
   */
  public void setMonsterInLaby(ID id, Monster monster)
  {
    this.theLaby.setMonster(id, monster);
  }

  /**
   * liefert eine Cell-Referenz
   *
   * @param cellPosition  die Position der gewuenschten Zelle
   *
   * @return              Cell-Referenz
   */
  public Cell getCellFromLaby(Point3i cellPosition)
  {
    return this.theLaby.getCell(cellPosition);
  }

  /**
   * gesendete Nachrichten eines Monsters an alle anderen verteilen
   *
   * @param senderID  die ID des sendenden Monsters
   * @param message   die zu sendende Nachricht
   *
   * @see             MonsterVoice#sendMessage
   */
  public void doBroadcast(ID senderID, MonsterMessage message)
  {
    // Referenz auf Monster mit senderID holen
    Monster talkMonster = this.getMonsterFromID(senderID);

    // alle Monster durchlaufen
    for (Iterator i = allMonsters.iterator(); i.hasNext(); )
    {
      Monster recvMonster = (Monster) i.next();
      MonsterVoice listener = recvMonster.getVoice();

      // ID des aktuellen Monsters holen
      ID recvID = recvMonster.getMonsterID();

      // aktuelles Monster ist nicht Sender der Nachricht, aber in CommRange
      if (!senderID.equals(recvID) && this.inCommRange(talkMonster, recvMonster))
      {
        listener.receiveMessage(message);
      }
    }
  }

  /**
   * liefert eine Referenz auf das Monster-Objekt mit der
   * angegebenen ID
   *
   * @param id   die ID des gesuchten Monsters
   *
   * @return     Referenz auf Monster-Objekt
   *
   * @see        #doBroadcast
   */
  public Monster getMonsterFromID(ID id)
  {
    Monster monster = null;

    for (Iterator i = allMonsters.iterator(); i.hasNext(); )
    {
      monster = (Monster) i.next();
      ID monsterID = monster.getMonsterID();
      if (monsterID.equals(id)) break;
    }

    return(monster);
  }

  /**
   * liefert die ID des aktiven Pacman, im Multiplayer-Modus also die
   * ID des lokalen Pacman
   *
   * @return   Pacman-ID
   */
  public ID getPacmanIDFromLaby()
  {
    return theLaby.getActivePacmanID();
  }

  /**
   * liefert true, falls die uebergebenen Monster in Kommuniktionsreich-
   * weite sind, ansonsten false.
   *
   * @return   in Kommunikationsweite ja/nein
   */
  private boolean inCommRange(Monster monsterA, Monster monsterB)
  {
    boolean reachable = false;

    Point3i positionA = new Point3i(monsterA.getControl().getMonsterPosition());
    Point3i positionB = new Point3i(monsterB.getControl().getMonsterPosition());

    int distanceA = monsterA.getAwareness().getCommDistanceFromSetup();
    int distanceB = monsterB.getAwareness().getCommDistanceFromSetup();

    int reichweite = distanceA + distanceB;

    // was bedeutet es, in Kommunikationsreichweite zu sein?
    // eine (beliebige) Komponente der Vektordifferenz der Positionen
    // ist kleiner als die Summe der Kommunikationsradien (think about it...)

    int abstand = Math.abs(positionA.x - positionB.x);
    if ((reichweite - abstand) > 0) reachable = true;
    else reachable = false;

    return(reachable);
  }

  public void writeToPositionMessage(Vector4f vector)
  {
    positionMessage.addLast(vector);
  }

  public void sendMessagePlaySound()
  {
    try
      {
        Message playMunchSound = new Message(nextID);
        playMunchSound.addReceiverGroup(1);
        playMunchSound.setContent("playMunchSound");
        messageService.sendMessage(playMunchSound);
      }
    catch (pacman3d.message.MissingMessageSenderException ex)
      {
        System.out.println("Fehler beim Senden an andere Channel...........");
      }
  }

  /**
   * stellt fest, ob sich der Pacman in der Naehe des Monsters befindet;
   * baut einen Suchbaum einer bestimten Tiefe von der activePacman-Position aus und
   * testet, ob die aktuelle Monster-Position enthalten ist.
   * (noetig, um festzustellen, ob der Sound abgespielt werden soll, oder nicht)
   *
   * @param suchtiefe Suchtiefe fuer die Breitensuche
   */
  public void PlaySoundIfPacmanInReach()
  {
    if (!monsterSound.isPlaying())  monsterSound.setEnable(true);
  }

    /**
   * spielt einen Sound, falls der Pacman gefangen wurde
   */
  public void PlaySoundPacmanReached()
  {
    if (!munchSound.isPlaying())
    {
      munchSound.setEnable(true);
      if (theLaby.isServer())    sendMessagePlaySound();
    }
  }

  /**
   * stellt einen PointSound zur Verfgung, dem eine URL beregeben werden kann,
   * die den abzuspielenden Sound enthlt. Dieser PointSound kann dann an
   * entsprechender Stelle in den Szenegraphen eingefgt werden.
   *
   * @param urlString das zu ladende Soundfile
   * @param soundTransform eine eigene TransformGroup fr den Sound
   * @return einen PointSound zum Einfgen in den Szenegraphen
   */
  private PointSound SoundBg(String urlString, BranchGroup soundTransform)
  {

    BranchGroup objroot = new BranchGroup();
    TransformGroup trans = new TransformGroup();
    trans.setCapability(trans.ALLOW_TRANSFORM_WRITE);

    MediaContainer stream = new MediaContainer();
    stream.setCapability(MediaContainer.ALLOW_URL_WRITE);
    stream.setCapability(MediaContainer.ALLOW_URL_READ);

      try
      {
        stream.setURLObject(new URL(urlString));
      }
      catch (Exception e)
      {
        System.out.println(e.getMessage());
      }

    PointSound sound = new PointSound();
    BoundingSphere bounds = new BoundingSphere(new Point3d(0.0,0.0,0.0), 100.0);
    sound.setSchedulingBounds(bounds);
    sound.setCapability(PointSound.ALLOW_ENABLE_WRITE);
    sound.setCapability(PointSound.ALLOW_INITIAL_GAIN_WRITE);
    sound.setCapability(PointSound.ALLOW_SOUND_DATA_WRITE);
    sound.setCapability(PointSound.ALLOW_SCHEDULING_BOUNDS_WRITE);
    sound.setCapability(PointSound.ALLOW_CONT_PLAY_WRITE);
    sound.setCapability(PointSound.ALLOW_RELEASE_WRITE);
    sound.setCapability(PointSound.ALLOW_DURATION_READ);
    sound.setCapability(PointSound.ALLOW_IS_PLAYING_READ);
    sound.setCapability(PointSound.ALLOW_LOOP_WRITE);
    sound.setCapability(PointSound.ALLOW_POSITION_WRITE);

    sound.setLoop(1);
    sound.setContinuousEnable(true);
    sound.setReleaseEnable(false);
    sound.setSoundData(stream);
    sound.setInitialGain(0.3f);

    sound.setEnable(false);
    // monsterSound = sound;

    trans.addChild(sound);
    //objroot.addChild(trans);

  //  objroot.compile();

    soundTransform.addChild(trans);
    return sound;
  }


  /**
   * erzeugt einen neuen MonsterChannel
   *
   * @param  monsterList Liste mit MonsterProperties-Objekten, in denen die
   *         Eigenschaften des Monsters abgelegt sind
   */

  /* ??? MonsterProperties werden nicht verwendet; stattdessen: Laden aller Monster-
         Infos aus Level-(XML)-File ???

  public MonsterChannel(LinkedList monsterList)
  {
    // Message-Service holen
    messageService = MessageService.getInstance();
    messageService.addMessageListener(this);

    MonsterProperties monsterProperties = null;

    for (int i = 0; i < monsterList.size(); i++)
    {
      monsterProperties = (MonsterProperties) monsterList.get(i);
      this.theLaby = monsterProperties.getLabyrinth();

      // neues Monster erzeugen und in MonsterListe schreiben
      Monster newMonster = new Monster(new ID(), monsterProperties, this);
      allMonsters.addLast(newMonster);

      // Eigenschaften der MonsterBranchGroup setzen
      monsterGraph.setCapability(BranchGroup.ALLOW_CHILDREN_READ);
      monsterGraph.setCapability(BranchGroup.ALLOW_CHILDREN_WRITE);
      monsterGraph.setCapability(BranchGroup.ALLOW_DETACH);
      monsterGraph.setCapability(BranchGroup.ALLOW_CHILDREN_READ);

      monsterGraph.addChild(newMonster.getView());
    }
  } // end of constructor

  */


} // end of MonsterChannel

