package pacman3d.message;

import java.util.Hashtable;
import java.util.Vector;
import pacman3d.util.Identifiable;
import java.util.Iterator;
import pacman3d.util.Debug;

/**
 * <b>Title:</b>      	Pacman 3D - Nicht wirklich ;)</br>
 * <b>Description:</b><br>
 *
 * <p>Nachrichtendienst fuer den Austausch von Nachrichten zwischen
 * &quot;Monstern&quot;, &quot;Pacman(s)&quot; und anderen Subjekten des
 * Spiels, die per Netzwerk ausgetauscht werden muessen.</p>
 *
 * <p>Kunden benutzen die Methode <code>send()</code> zum Versenden einer Nachricht,
 * und implementieren das Interface <code>MessageListener</code> zum
 * ereignisgesteuerten Empfangen von Nachrichten. Nachrichten sind von der
 * Klasse <code>Message</code>.</p>
 *
 * <p>Es kann zu jeder Zeit nur eine einzige Instanz von
 * <code>MessageService</code> geben. Es existiert die statische Factory-
 * Methode <code>getInstance</code>, um eine Referenz auf die Instanz zu
 * bekommen.</p>
 *
 * <p>Eine Verwendung der Klasse koennte also wie folgt aussehen:</p>
 *
 * <p><code>
 * <pre>
 * oMessage = new pacman3d.message.Message();
 * ... // Nachricht initialisieren, Nutzlast und Empfnger definieren
 * oMessageService = pacman3d.message.MessageService.getInstance();
 * oMessageService.send( oMessage );
 * </pre>
 * </code></p>
 *
 * <b>Copyright:</b>	Copyright (c) 2001<br>
 *
 * @author              Labyrinth-Gruppe<br>
 * @version             26.11.2001<br>
 */
public class MessageService implements pacman3d.net.NetworkListener {

	/**
	 * Die einzige Instanz des MessageService, die es geben darf.
	 * @see getInstance()
	 */
	private static MessageService m_oInstance = null;

	/**
	 * Die Instanz von <code>NetworkService</code>, die von dem
	 * <code>MessageService</code> verwendet werden soll, um Nachrichten
	 * auch ueber das Netzwerk zu senden.
	 */
	private pacman3d.net.NetworkService m_oNetworkService;

	/**
	 * Die Liste aller registrierten Objekte, d.h. der Objekte, die vom
	 * MessageService Nachrichten zugestellt bekommen koennen.
	 */
	private Hashtable m_oMessageListeners;

	/**
	 * Identifizierung der Empfaengerart "PACMAN" innerhalb der
	 * vorgruppierten Empfaengerliste.
	 */
	protected final static int RECEIVER_PACMAN = 0;

	/**
	 * Identifizierung der Empfaengerart "MONSTER" innerhalb der
	 * vorgruppierten Empfaengerliste.
	 */
	protected final static int RECEIVER_MONSTER = 1;

	/**
	 * Identifizierung der Empfaengerart "GAME" innerhalb der
	 * vorgruppierten Empfaengerliste.
	 */
	protected final static int RECEIVER_GAME = 2;

	/**
	 * Identifizierung der Empfaengerart "ALLE" innerhalb der
	 * vorgruppierten Empfaengerliste.
	 */
	protected final static int RECEIVER_ALL = 3;

	/**
	 * Identifizierung der Propagierung "NUR NETZWERK" fuer den
	 * Versand von Nachrichten. Fuer kombinierte Propagierung sind
	 * die entsprechenden Konstanten additiv zu kombinieren.
	 */
	public final static int PROPAGATE_NETWORK = 1;

	/**
	 * Identifizierung der Propagierung "NUR LOKAL" fuer den
	 * Versand von Nachrichten. Fuer kombinierte Propagierung sind
	 * die entsprechenden Konstanten additiv zu kombinieren.
	 */
	public final static int PROPAGATE_LOCAL = 2;

	/**
	 * Der folgende Container enthlt Vorsortierungen aller registrierten
	 * Empfaenger fuer Nachrichten, um bei Nachrichteneingang in
	 * Konstantzeit die Empfaengermenge bestimmen zu koennen. ;-)
	 */
	private Vector[] m_aoMessageListenerGroups;

	/**
	 */
	private Hashtable m_aoMessageTypeListeners;

	/**
	 * Die initiale Kapazitaet der Container fuer die verzoegerten
	 * Nachrichten.
	 */
	private final static int m_iInitialCapacityDelayedMessages = 256;

	/**
	 * Die aktuelle Anzahl verzoegerter Nachrichten.
	 */
	private int m_iDelayedMessagesCount = 0;

	/**
	 * Die Liste aller Verzoegerungszeiten.
	 */
	private long[] m_aiDelayTimes;

	/**
	 * Die Liste aller verzoegerten Propagation-Flags.
	 */
	private int[] m_aiDelayPropagations;

	/**
	 * Die Liste aller verzoegerten Nachrichten.
	 */
	private Message[] m_aoDelayMessages;

	/**
	 * Der Zeitpunkt, zu dem das Letzte mal die Liste der verzoegerten
	 * Nachrichten ueberprueft wurde.
	 */
	private long m_iDelayedMessagesTimeStamp = 0;

	/**
	 * Die "eigene ID" des MessageService.
	 */
	private ID m_oID;

	/**
	 * Konstruktor, ist nur ein einziges Mal an zentraler Stelle
	 * aufzurufen!
	 */
	private MessageService() {
		// fuer "sich selber" eine ID erzeugen
		m_oID = new ID();

		this.m_oMessageListeners = new Hashtable();
		this.m_aoMessageTypeListeners = new Hashtable();

		this.m_aiDelayPropagations = new int[ m_iInitialCapacityDelayedMessages ];
		this.m_aiDelayTimes = new long[ m_iInitialCapacityDelayedMessages ];
		java.util.Arrays.fill( m_aiDelayTimes, Long.MAX_VALUE );
		this.m_aoDelayMessages = new Message[ m_iInitialCapacityDelayedMessages ];

		this.m_aoMessageListenerGroups = new Vector[4];
		this.m_aoMessageListenerGroups[ RECEIVER_PACMAN ] = new Vector();
		this.m_aoMessageListenerGroups[ RECEIVER_MONSTER ] = new Vector();
		this.m_aoMessageListenerGroups[ RECEIVER_GAME ] = new Vector();
		this.m_aoMessageListenerGroups[ RECEIVER_ALL ] = new Vector();
	}

	public static MessageService getInstance() {
		if (m_oInstance == null) {
			m_oInstance = new MessageService();
		}

		return m_oInstance;
	}

	/**
	 * Setzt die Instanz von <code>NetworkService</code>, die von dem
	 * <code>MessageService</code> verwendet werden soll, um Nachrichten
	 * auch ueber das Netzwerk zu senden.
	 */
	public void setNetworkService( pacman3d.net.NetworkService oNetworkService ) {
		this.m_oNetworkService = oNetworkService;
		if (oNetworkService != null) {
			oNetworkService.addNetworkListener( this );
		}
	}

	/**
	 * Berechnet Vorsortierungen aller registrierten
	 * Empfaenger fuer Nachrichten, um bei Nachrichteneingang in
	 * Konstantzeit die Empfaengermenge bestimmen zu koennen. ;-)
	 */
	private void updateMessageListenerGroups( MessageListener oMessageListener ) {
		ID oID = oMessageListener.getID();

		if (oMessageListener instanceof pacman3d.pacman.Pacman) {
			/* In die Liste aller Instanzen von Pacman bzw. deren
			Nachfahren eintragen. */
			m_aoMessageListenerGroups[ RECEIVER_PACMAN ].add( oMessageListener );
			m_aoMessageListenerGroups[ RECEIVER_ALL ].add( oMessageListener );
		} else if (oMessageListener instanceof pacman3d.monster.MonsterChannel) {
			/* dito Monster */
			m_aoMessageListenerGroups[ RECEIVER_MONSTER ].add( oMessageListener );
			m_aoMessageListenerGroups[ RECEIVER_ALL ].add( oMessageListener );
		} else if (oMessageListener instanceof pacman3d.Game) {
			/* dito Game */
			m_aoMessageListenerGroups[ RECEIVER_GAME ].add( oMessageListener );
			m_aoMessageListenerGroups[ RECEIVER_ALL ].add( oMessageListener );
		} else {
			m_aoMessageListenerGroups[ RECEIVER_ALL ].add( oMessageListener );
		}
	}

	/**
	 * Registriert ein Objekt fuer den Empfang von Nachrichten. Objekte
	 * muessen sich registrieren, bevor sie Nachrichten empfangen koennen!
	 *
	 * @param oMessageListener      Ein Objekt einer beliebigen Klasse, die
	 *                              das Interface
	 *                              <code>MessageListener</code>
	 *                              implementiert. Dieses Objekt wird in
	 *                              Zukunft Nachrichten von dem
	 *                              MessageService erhalten.
	 */
	public void addMessageListener(
		pacman3d.message.MessageListener oMessageListener ) {

		m_oMessageListeners.put(
			((Identifiable) oMessageListener).getID(),
			oMessageListener );
		updateMessageListenerGroups( oMessageListener );
	}

	/**
	 * De-Registriert ein Objekt fuer den Empfang von Nachrichten. Dadurch
         * erhaelt das Objekt keine weiteren Nachrichten mehr.
	 *
	 * @param oMessageListener      Ein Objekt einer beliebigen Klasse, die
	 *                              das Interface
	 *                              <code>MessageListener</code>
	 *                              implementiert. Dieses Objekt wird in
	 *                              Zukunft keine Nachrichten mehr von dem
	 *                              MessageService erhalten.
	 */
	public void removeMessageListener(
		pacman3d.message.MessageListener oMessageListener ) {

                // erstmal aus der Liste der "gewoehnlichen" Empfaenger entfernen
                m_oMessageListeners.remove(
                        ((Identifiable) oMessageListener).getID() );

                // nun aus der Liste der Pacmans, Monster, und was auch immer
                // entfernen
                m_aoMessageListenerGroups[ RECEIVER_PACMAN ].remove(
                        oMessageListener );
                m_aoMessageListenerGroups[ RECEIVER_ALL ].remove(
                        oMessageListener );
                m_aoMessageListenerGroups[ RECEIVER_GAME ].remove(
                        oMessageListener );
                m_aoMessageListenerGroups[ RECEIVER_MONSTER ].remove(
                        oMessageListener );

                // nun ggf. aus der Liste der Subscriber fuer bestimmte
                // Nachrichtentypen entfernen
                java.util.Enumeration oEnumeration = m_aoMessageTypeListeners.elements();
                while (oEnumeration.hasMoreElements()) {
                        Vector oVector = (Vector) oEnumeration.nextElement();
                        oVector.remove( oMessageListener );
                }
	}

	/**
	 * Registriert ein Objekt fuer den Empfang von Nachrichten. Objekte
	 * muessen sich registrieren, bevor sie Nachrichten empfangen koennen!
	 * Es kann zusaetzlich ein Typ von Nachrichten benannt werden, die
	 * empfangen werden sollen.
	 *
	 * @param oMessageListener      Ein Objekt einer beliebigen Klasse, die
	 *                              das Interface
	 *                              <code>MessageListener</code>
	 *                              implementiert. Dieses Objekt wird in
	 *                              Zukunft Nachrichten von dem
	 *                              MessageService erhalten.
	 * @param sMessageType          Der Name eines Typs von Nachrichten, die
	 *                              empfangen werden sollen.
	 */
	public void addMessageListener(
		pacman3d.message.MessageListener oMessageListener,
		String sMessageType ) {

		addMessageListener( oMessageListener );

		Vector oMessageListeners =
			(Vector) m_aoMessageTypeListeners.get( sMessageType );
		if (oMessageListeners == null) {
			oMessageListeners = new Vector();
			oMessageListeners.add( oMessageListener );
			m_aoMessageTypeListeners.put( sMessageType, oMessageListeners );
		} else {
			oMessageListeners.add( oMessageListener );
		}

	}

	/**
	 * Gibt eine Nachricht zum Versenden auf. Die Nachricht wird an alle
	 * entfernten Rechner und an den lokalen Rechner gesendet.
	 * @param oMessage              Die zu versendende Nachricht.
	 * @see pacman3d.message.Message
	 */
	public void sendMessage( Message oMessage )
		throws MissingMessageSenderException {

		sendMessage( oMessage, PROPAGATE_LOCAL | PROPAGATE_NETWORK, 0 );
	}

	/**
	 * Gibt eine Nachricht zum verzoegerten Versenden auf. Die Nachricht
	 * wird an alle entfernten Rechner und an den lokalen Rechner gesendet.
	 * @param oMessage              Die zu versendende Nachricht.
	 * @param iDelayMS              Die Anzahl Millisekunden, die die
	 *                              Nachricht verzoegert werden soll. Ein
	 *                              Wert von <code>0</code> entspricht dem
	 *                              unverzoegerten Versenden.
	 * @see pacman3d.message.Message
	 */
	public void sendDelayedMessage( Message oMessage, long iDelayMS )
		throws MissingMessageSenderException {

		sendMessage( oMessage, PROPAGATE_LOCAL | PROPAGATE_NETWORK, iDelayMS );
	}

	/**
	 * Gibt eine Nachricht zum Versenden auf. Die Nachricht wird an alle
	 * entfernten Rechner gesendet, und optional auch an den lokalen
	 * Rechner.
	 * @param oMessage              Die zu versendende Nachricht.
	 * @param bLocalLoopback        <code>true</code>, wenn die Nachricht
	 *                              auch auf dem lokalen Rechner versendet
	 *                              werden soll. Anderenfalls wird sie nur
	 *                              auf den entfernten Rechnern versendet.
	 * @see pacman3d.message.Message
	 */
	public void sendMessage( Message oMessage, boolean bLocalLoopback )
		throws MissingMessageSenderException {

		if (bLocalLoopback) {
			sendMessage( oMessage,
				PROPAGATE_LOCAL | PROPAGATE_NETWORK, 0 );
		} else {
			sendMessage( oMessage, PROPAGATE_NETWORK, 0 );
		}
	}

	/**
	 * Gibt eine Nachricht zum Versenden auf. Die Nachricht wird an alle
	 * entfernten Rechner und/oder an den lokalen Rechner versendet.
	 * @param oMessage              Die zu versendende Nachricht.
	 * @param iPropagation          Eine (oder mehrere) der <code>PROPAGATE_...</code>-
	 *                              Konstanten. Dieser Parameter definiert,
	 *                              ob die Nachricht nur lokal, nur ueber
	 *                              das Netzwerk, oder sowohl als auch
	 *                              versendet werden soll. Fuer kombinierte
	 *                              Propagierung sind die entsprechenden
	 *                              Konstanten additiv zu kombinieren.
	 * @see pacman3d.message.Message
	 */
	public void sendMessage( Message oMessage, int iPropagation )
		throws MissingMessageSenderException {

		sendMessage( oMessage, iPropagation, 0 );
	}

	/**
	 * Gibt eine Nachricht zum verzoegerten Versenden auf. Die Nachricht
	 * wird an alle entfernten Rechner und/oder an den lokalen Rechner
	 * versendet.
	 * @param oMessage              Die zu versendende Nachricht.
	 * @param iPropagation          Eine (oder mehrere) der <code>PROPAGATE_...</code>-
	 *                              Konstanten. Dieser Parameter definiert,
	 *                              ob die Nachricht nur lokal, nur ueber
	 *                              das Netzwerk, oder sowohl als auch
	 *                              versendet werden soll. Fuer kombinierte
	 *                              Propagierung sind die entsprechenden
	 *                              Konstanten additiv zu kombinieren.
	 * @param iDelayMS              Die Anzahl Millisekunden, die die
	 *                              Nachricht verzoegert werden soll. Ein
	 *                              Wert von <code>0</code> entspricht dem
	 *                              unverzoegerten Versenden.
	 * @see pacman3d.message.Message
	 */
	public void sendMessage( Message oMessage, int iPropagation, long iDelayMS )
		throws MissingMessageSenderException {

		// eventuell verzoegerte Nachrichten bearbeiten!
		processDelayedMessages();

		/* Falls kein Empfnger definiert ist -> Fehler! */
		/*
		if (!oMessage.hasReceiver()) {
			throw new MissingMessageSenderException();
		}
		*/

		// Wenn Verzoegerung==0 => sofort senden!
		if (iDelayMS == 0) {
			// Nachricht sofort senden!
			processOutgoingMessage( oMessage, iPropagation );
		} else {
			// Nachricht in Warteschlange stecken!
			scheduleDelayedMessage( oMessage, iPropagation,
				iDelayMS );
		}
	}

	/**
	 * Nimmt eine Nachricht von der Netzwerkschnittstelle entgegen, und
	 * verteilt sie an die entsprechenden Empfaenger.
	 */
	public void dispatchMessage( Message oMessage ) {
		if (oMessage instanceof Message) {
			/* Empfangsdatum der Nachricht setzen */
			((Message) oMessage).setReceiveTime();

			/* Eine Behlter initialisieren, indem wir die Instanzen
			aller betroffenen Objekte sammeln */
			java.util.HashSet oReceiverObjects = new java.util.HashSet();

			/* Durch die Liste der Empfnger iterieren */
			Iterator oIterator = ((Message) oMessage).getReceivers().iterator();

			/* Gibt's noch einen weiteren Empfnger? */
			while (oIterator.hasNext()) {
				/* Hole den nchsten Empfnger! */
				Object oReceiver = oIterator.next();

				/* Handelt es sich um eine Empfngergruppe? */
				if (oReceiver instanceof Integer) {
					/* bernehme die Empfngergruppen-Objekte! */
					oReceiverObjects.addAll(
						m_aoMessageListenerGroups[
						((Integer) oReceiver).intValue()
						] );
					// Debug.out (this.getClass().getName(), "dispatchMessage: " + oReceiver);
				} else if (oReceiver instanceof ID) {
					Object oMessageListener =
						m_oMessageListeners.get( oReceiver );
					if (oMessageListener == null) {
						pacman3d.util.Debug.out( getClass().getName(),
							"unknown message receiver: " +
							oReceiver.toString() +
                                                        " (" +
                                                        oMessage.toString() +
                                                        ")", pacman3d.util.Debug.LEVEL_NOTICE );
					} else {
						oReceiverObjects.add( oMessageListener );
					}
				}

			}

			// je nach Typ der Nachricht noch weitere
			// Empfaenger hinzufuegen
			Object oThis = m_aoMessageTypeListeners.get( oMessage.getMessageType() );
			if (oThis != null) {
				Vector oMessageListeners = (Vector) oThis;
				if (oMessageListeners != null) {
					oReceiverObjects.addAll( oMessageListeners );
				}
			}

			oIterator = oReceiverObjects.iterator();
			while (oIterator.hasNext()) {
				MessageListener oMessageListener = (MessageListener) oIterator.next();
				oMessageListener.getMessage( (Message) oMessage );
			}
		} else {
			System.out.println( "Error: Unknown message class " +
				"\"" + oMessage.getClass().getName() +
				"\" in MessageService.dispatchMessage()!" );
		}
	}

	/**
	 * Methode zum Empfangen von Statusmeldungen &uuml;ber das Netzwerk.
	 */
	public void networkStatus( int iStatus ) {
		// Nachricht versenden
		NetworkStateMessage oNetworkStateMessage =
			new NetworkStateMessage( iStatus );
		Message oMessage = new Message( m_oID, oNetworkStateMessage,
			"network.state" );

		try {
			sendMessage( oMessage );
		} catch( MissingMessageSenderException oException ) {
			Debug.out( getClass().getName(), oException,
				Debug.LEVEL_WARNING );
		}
	}

	/**
	 * Methode zum Empfangen von Nachrichten. Wird beim Erhalt einer Nachricht
	 * aufgerufen.
	 */
	public void networkMessageReceived( pacman3d.net.NetworkMessage oNetworkMessage ) {
		if (oNetworkMessage instanceof Message) {
                        if (((Message) oNetworkMessage).getContent() instanceof pacman3d.message.PacmanSpawnedContent) {
                                Debug.out( getClass().getName(),
                                        "PacmanSpawnedContent received: " +
                                        oNetworkMessage.toString(),
                                        Debug.LEVEL_NOTICE );
                        }
                        if (((Message) oNetworkMessage).getContent() instanceof pacman3d.message.PlayerNewContent) {
                                Debug.out( getClass().getName(),
                                        "PlayerNewContent received: " +
                                        oNetworkMessage.toString(),
                                        Debug.LEVEL_NOTICE );
                        }

			this.dispatchMessage( (Message) oNetworkMessage );
		} else {
			pacman3d.util.Debug.out( getClass().getName(),
				"cannot handle network message of class: " +
				oNetworkMessage.getClass().getName(),
				pacman3d.util.Debug.LEVEL_WARNING );
		}
	}

	/**
	 * Verarbeitet eine ausgehende Nachricht, d.h. stellt diese an die
	 * entsprechenden Empfaengergruppen zu, und setzt den Sende-Timestamp.
	 * @param oMessage      Die zu versendende Nachricht
	 * @param iPropagation          Eine (oder mehrere) der <code>PROPAGATE_...</code>-
	 *                              Konstanten. Dieser Parameter definiert,
	 *                              ob die Nachricht nur lokal, nur ueber
	 *                              das Netzwerk, oder sowohl als auch
	 *                              versendet werden soll. Fuer kombinierte
	 *                              Propagierung sind die entsprechenden
	 *                              Konstanten additiv zu kombinieren.
	 */
	private void processOutgoingMessage( Message oMessage, int iPropagation ) {
		/* Sendedatum der Nachricht setzen */
		oMessage.setSendTime();

		/* Nachricht ggf. lokal senden! */
		if ((iPropagation & PROPAGATE_LOCAL) > 0) {
			dispatchMessage( oMessage );
		}

		/* Netzwerk-Layer aufrufen und Nachricht bergeben! */
		if ((iPropagation & PROPAGATE_NETWORK) > 0) {
			// Debug.out (this.getClass().getName(), "processOutgoingMessage: " + oMessage.getReceivers().get(0) );
			if (m_oNetworkService != null) {
				m_oNetworkService.send( oMessage );
			}
		}
	}

	/**
	 * Steckt eine zu verzoegernde Nachricht in die Warteschlange.
	 * @param oMessage              Die zu versendende Nachricht
	 * @param iPropagation          Eine (oder mehrere) der <code>PROPAGATE_...</code>-
	 *                              Konstanten. Dieser Parameter definiert,
	 *                              ob die Nachricht nur lokal, nur ueber
	 *                              das Netzwerk, oder sowohl als auch
	 *                              versendet werden soll. Fuer kombinierte
	 *                              Propagierung sind die entsprechenden
	 *                              Konstanten additiv zu kombinieren.
	 * @param iDelayMS              Die Anzahl Millisekunden, die die
	 *                              Nachricht verzoegert werden soll. Ein
	 *                              Wert von <code>0</code> entspricht dem
	 *                              unverzoegerten Versenden.
	 */
	private void scheduleDelayedMessage( Message oMessage, int iPropagation,
		long iDelayMS ) {

		// die zukuenftige Sendezeit berechnen
		long iEstimatedSendTime = new java.util.Date(
			new java.util.Date().getTime() + iDelayMS ).getTime();

		// sicherstellen, dass mindestens noch zwei Plaetze frei ist:
		if (m_iDelayedMessagesCount + 2 > m_aiDelayTimes.length) {
			// kein Platz mehr => umkopieren!
			int iCapacity = m_aiDelayTimes.length;
			int iNewCapacity = iCapacity * 2;
			int[] aiDelayPropagations = new int[ iNewCapacity ];
			long[] aiDelayTimes = new long[ iNewCapacity ];
			java.util.Arrays.fill( aiDelayTimes, Long.MAX_VALUE );
			Message[] aoDelayMessages = new Message[ iNewCapacity ];

			System.arraycopy( m_aiDelayPropagations, 0,
				aiDelayPropagations, 0, iCapacity );
			System.arraycopy( m_aiDelayTimes, 0,
				aiDelayTimes, 0, iCapacity );
			System.arraycopy( m_aoDelayMessages, 0,
				m_aoDelayMessages, 0, iCapacity );

			m_aiDelayTimes = aiDelayTimes;
			m_aiDelayPropagations = aiDelayPropagations;
			m_aoDelayMessages = aoDelayMessages;
		}

		// nun darf ruhigen Gewissens davon ausgegangen werden, dass
		// mindestens noch ein Eintrag im Array frei ist. Nun eine
		// binaere Suche starten, und die richtige Position zum
		// Einfuegen ermitteln:

		int iPos = java.util.Arrays.binarySearch( m_aiDelayTimes,
			iEstimatedSendTime );
		if (iPos < 0) {
			iPos = -iPos - 1;
		}
		System.arraycopy( m_aiDelayTimes, iPos, m_aiDelayTimes, iPos+1,
			m_iDelayedMessagesCount-iPos );
		System.arraycopy( m_aiDelayPropagations, iPos, m_aiDelayPropagations, iPos+1,
			m_iDelayedMessagesCount-iPos );
		System.arraycopy( m_aoDelayMessages, iPos, m_aoDelayMessages, iPos+1,
			m_iDelayedMessagesCount-iPos );

		m_iDelayedMessagesCount++;

		m_aiDelayTimes[ iPos ] = iEstimatedSendTime;
		m_aiDelayPropagations[ iPos ] = iPropagation;
		m_aoDelayMessages[ iPos ] = oMessage;
	}

	/**
	 * Betrachtet alle Nachrichten in der Verzoegerungs-Warteschlange, und
	 * sendet all jene, deren Sendedatum erreicht oder ueberschritten ist.
	 * Zuvor wird allerdings ueberprueft, ob seit dem letzten Aufruf der
	 * Methode eine bestimmte Zeitdauer vergangen ist, um zu verhindern,
	 * dass diese Handlung "zu oft" ausgefuehrt wird - sie kann durchaus
	 * etwas Zeit kosten.
	 */
	private void processDelayedMessages() {

		long iTimeStamp = (new java.util.Date()).getTime();

		if (m_iDelayedMessagesTimeStamp + 1000 > iTimeStamp ) {
			return;
		}

		// Index der ersten Nachricht finden, die vom aktuellen Zeitpunkt aus
		// betrachtet erst in der "Zukunft" versendet werden soll.
		int iPos=0;
		while (m_aiDelayTimes[ iPos ] <= iTimeStamp) iPos++;

		// Falls wir keine Nachricht, sondern die "Abbruchbedingung"
		// gefunden haben, nichts unternehmen. Anderenfalls:
		if (iPos < m_iDelayedMessagesCount) {
			// Alle Nachrichten versenden, deren Sendedatum in der
			// Vergangenheit liegt:
			for (int i=0; i<iPos; i++) {
				processOutgoingMessage( m_aoDelayMessages[ i ],
					m_aiDelayPropagations[ i ] );
			}
			// Alle diese Nachrichten loeschen!
			System.arraycopy( m_aoDelayMessages, iPos,
				m_aoDelayMessages, 0,
				m_iDelayedMessagesCount-iPos );
			System.arraycopy( m_aiDelayTimes, iPos,
				m_aiDelayTimes, 0,
				m_iDelayedMessagesCount-iPos );
			System.arraycopy( m_aiDelayPropagations, iPos,
				m_aiDelayPropagations, 0,
				m_iDelayedMessagesCount-iPos );

			m_iDelayedMessagesCount = m_iDelayedMessagesCount - iPos;

			java.util.Arrays.fill( m_aiDelayTimes,
				m_iDelayedMessagesCount, m_aiDelayTimes.length-1,
				Long.MAX_VALUE );
		}
	}
}