package pacman3d.message;

import java.util.Date;
import java.util.Vector;
import pacman3d.net.NetworkMessage;
import java.io.Serializable;
import java.lang.Cloneable;

/**
 * <b>Title:</b>      	Pacman 3D - Nicht wirklich ;)</br>
 * <b>Description:</b><br>
 *
 * Basisklasse fuer alle Nachrichten, die zwischen &quot;Monstern&quot;,
 * &quot;Pacman(s)&quot; und anderen Subjekten des Spiels per
 * Netzwerk ausgetauscht werden muessen.<br>
 *
 * <b>Copyright:</b>	Copyright (c) 2001<br>
 *
 * @author              Labyrinth-Gruppe<br>
 * @version             26.11.2001<br>
 */
public class Message implements NetworkMessage, Serializable, Cloneable {

        /**
         * Die Liste der Empfaenger der Nachricht.
         */
        private Vector m_oReceivers;

        /**
         * Die Liste der "physischen" Empfaenger der Nachricht, in dieser
         * Aufzhlung sind also nur konkrete IDs enthalten, und keine
         * Empfngergruppen oder hnliches.
         */
        // private Vector m_oPhysicalReceivers;

        /**
         * Der Absender der Nachricht.
         */
        private ID m_oSender;

        /**
         * Die Nutzlast der Nachricht.
         */
        private Object m_oContent;

        /**
         * Der Zeitpunkt, zu dem die Nachricht durch die Klasse
         * <code>MessageService</code> versendet worden ist.
         */
        private Date m_oSendTime;

        /**
         * Der Typ der Nachricht.
         */
        private String m_sMessageType = "";

        /**
         * Der Zeitpunkt, zu dem die Nachricht durch die Klasse
         * <code>MessageService</code> empfangen worden ist.
         */
        private Date m_oReceiveTime;

        /**
         * Die Prioritt der Nachricht.
         */
        private int m_iPriority;

        /**
         * Platzhalter fuer Nachrichten-Empfaenger: Alle Subjekte.
         */
        public final static int RECEIVER_ALL = MessageService.RECEIVER_ALL;

        /**
         * Platzhalter fuer Nachrichten-Empfaenger: Alle Games.
         */
        public final static int RECEIVER_ALL_GAMES = MessageService.RECEIVER_GAME;

        /**
         * Platzhalter fuer Nachrichten-Empfaenger: Alle Monster.
         */
        public final static int RECEIVER_ALL_MONSTERS = MessageService.RECEIVER_MONSTER;

        /**
         * Platzhalter fuer Nachrichten-Empfaenger: Alle Pacmans.
         */
        public final static int RECEIVER_ALL_PACMANS = MessageService.RECEIVER_PACMAN;

        /**
         * Nachrichten-Prioritaet: Niedrigste Prioritt.
         */
        public final static int PRIORITY_LOWEST = -2;

        /**
         * Nachrichten-Prioritaet: Niedrige Prioritt.
         */
        public final static int PRIORITY_LOW = -1;

        /**
         * Nachrichten-Prioritaet: Vorgabewert.
         */
        public final static int PRIORITY_DEFAULT = 0;

        /**
         * Nachrichten-Prioritaet: Hohe Prioritt.
         */
        public final static int PRIORITY_HIGH = 1;

        /**
         * Nachrichten-Prioritaet: Hchste Prioritt.
         */
        public final static int PRIORITY_HIGHEST = 2;

        /**
         * Konstruktor, erzeugt eine Nachricht.
         *
         * @param oSender       Die eindeutige Bezeichnung des Senders der
         *                      Nachricht, d.h. eine Instanz von
         *                      <code>pacman3d.message.ID</code>.
         * @see pacman3d.message.ID
         */
        public Message( ID oSender ) {
                m_oSender = oSender;
                m_oReceivers = new Vector();
                // m_oPhysicalReceivers = new Vector();
        }

        /**
         * Konstruktor, erzeugt eine Nachricht, und fuegt der Nachricht eine
         * Empfaengergruppe hinzu, wie z.B. &quot;alle Monster&quot; oder
         * &quot;alle Pacmans&quot;. Siehe die <code>RECEIVER_...</code>-Konstanten
         * dieser Klasse!
         * @param oSender       Die eindeutige Bezeichnung des Senders der
         *                      Nachricht, d.h. eine Instanz von
         *                      <code>pacman3d.message.ID</code>.
         * @param oContent      Die Nutzlast dieser Nachricht.
         *
         * @see pacman3d.message.ID
         * @see #addReceiver(ID)
         * @see #setReceiver(ID)
         */
        public Message( ID oSender, Object oContent, int iReceiverGroup ) {
                this( oSender );
                this.setContent( oContent );
                this.addReceiverGroup( iReceiverGroup );
        }

        /**
         * Konstruktor, erzeugt eine Nachricht, und fuegt der Nachricht einen
         * Inhaltstyp hinzu.
         * @param oSender       Die eindeutige Bezeichnung des Senders der
         *                      Nachricht, d.h. eine Instanz von
         *                      <code>pacman3d.message.ID</code>.
         * @param sMessageType  Der Typ dieser Nachricht.
         */
        public Message( ID oSender, Object oContent, String sMessageType ) {
                this( oSender );
                this.setContent( oContent );
                this.setMessageType( sMessageType );
        }

        /**
         * Konstruktor, erzeugt eine Nachricht.
         *
         * @param oSender       Die eindeutige Bezeichnung des Senders der
         *                      Nachricht, d.h. eine Instanz von
         *                      <code>pacman3d.message.ID</code>.
         * @param oContent      Die Nutzlast dieser Nachricht.
         * @param oReceiver     Der Empfnger dieser Nachricht.
         * @see pacman3d.message.ID
         */
        public Message( ID oSender, Object oContent, ID oReceiver ) {
                this( oSender );
                this.setContent( oContent );
                this.setReceiver( oReceiver );
        }

        /**
         * Setzt die Sendezeit der Nachricht auf die aktuelle Zeit. Sollte
         * ausschliesslich von der Klasse <code>MessageService</code>
         * aufgerufen werden.
         * @see MessageService
         */
        protected void setSendTime() {
                this.m_oSendTime = new Date();
        }

        /**
         * Setzt die Sendezeit der Nachricht auf die gegebene Zeit. Sollte
         * ausschliesslich von der Klasse <code>MessageService</code>
         * aufgerufen werden.
         * @see MessageService
         */
        protected void setSendTime( Date oSendTime ) {
                this.m_oSendTime = oSendTime;
        }

        /**
         * Gibt die Klasse der Nachricht zurueck.
         * @see MessageService
         */
        public String getMessageType() {
                return this.m_sMessageType;
        }

        /**
         * Setzt die Klasse der Nachricht auf die gegebene Klasse.
         * @see MessageService
         */
        public void setMessageType( String sMessageType ) {
                this.m_sMessageType = sMessageType;
        }

        /**
         * Setzt die Empfangszeit der Nachricht auf die aktuelle Zeit. Sollte
         * ausschliesslich von der Klasse <code>MessageService</code>
         * aufgerufen werden.
         * @see MessageService
         */
        protected void setReceiveTime() {
                this.m_oReceiveTime = new Date();
        }

        /**
         * Setzt die Empfangszeit der Nachricht auf die gegebene Zeit. Sollte
         * ausschliesslich von der Klasse <code>MessageService</code>
         * aufgerufen werden.
         * @see MessageService
         */
        protected void setReceiveTime( Date oReceiveTime ) {
                this.m_oReceiveTime = oReceiveTime;
        }

        /**
         * Liefert <code>true</code> zurck, wenn zumindest ein (gltiger)
         * Empfnger angegeben ist.
         */
        public boolean hasReceiver() {
                return (! m_oReceivers.isEmpty());
        }

        /**
         * Liefert <code>true</code> zurck, wenn zumindest ein (gltiger)
         * Absender angegeben ist.
         */
        public boolean hasSender() {
                return (m_oSender != null);
        }

        /**
         * Gibt die Prioritt der Nachricht zurck.
         * Siehe Definition der Konstanten <code>Message.PRIORITY_DEFAULT</code>,
         * <code>Message.PRIORITY_...</code>.
         */
        public int getPriority() {
                return this.m_iPriority;
        }

        /**
         * Setzt die Prioritt der Nachricht auf den gegebenen Wert.
         * Siehe Definition der Konstanten <code>Message.PRIORITY_DEFAULT</code>,
         * <code>Message.PRIORITY_...</code>.
         */
        public void setPriority( int iPriority ) {
                this.m_iPriority = iPriority;
        }

        /**
         * Gibt den Absender der Nachricht zurueck.
         */
        public ID getSender() {
                return this.m_oSender;
        }

        /**
         * Setzt den Empfnger der Nachricht und lscht ggf. bereits
         * definierte Empfnger.
         *
         * @see #addReceiver(ID)
         * @see #addReceiverGroup(int)
         */
        public void setReceiver( ID oReceiver ) {
                m_oReceivers.clear();
                m_oReceivers.add( oReceiver );
        }

        /**
         * Fuegt der Nachricht ein &quot;Monster&quot; oder einen
         * &quot;Pacman&quot, also ein konkretes Subjekt als einen Empfaenger
         * hinzu.
         *
         * @see #setReceiver(ID)
         * @see #addReceiverGroup(int)
         */
        public void addReceiver( ID oReceiver ) {
                // Empfnger zur Empfngerliste hinzufgen
                m_oReceivers.add( oReceiver );
        }

        /**
         * Fuegt der Nachricht eine Empfaengergruppe hinzu, wie z.B.
         * &quot;alle Monster&quot; oder &quot;alle Pacmans&quot;. Siehe
         * die <code>RECEIVER_...</code>-Konstanten dieser Klasse!
         *
         * @see #addReceiver(ID)
         * @see #setReceiver(ID)
         */
        public void addReceiverGroup( int iReceiverGroup ) {
                m_oReceivers.add( new Integer( iReceiverGroup ) );
        }

        /**
         * Gibt die Liste der Empfaenger zurueck.
         */
        public Vector getReceivers() {
                return m_oReceivers;
        }

        /**
         * Gibt die Nutzlast der Nachricht zurueck.
         */
        public Object getContent() {
                return m_oContent;
        }

        /**
         * Setzt die Nutzlast der Nachricht.
         */
        public void setContent( Object oContent ) {
                m_oContent = oContent;
        }

	/** toString: eine einfache TextConvertierung der Message ... */
	public String toString() {
		return "sender: " + this.m_oSender + " receiver: " +
			this.m_oReceivers + " content: " + this.m_oContent;
	}

}