package pacman3d.labyrinth.leveleditor;

import pacman3d.labyrinth.cells.*;
import javax.swing.*;
import java.io.*;
import javax.media.j3d.*;
import com.sun.j3d.utils.image.TextureLoader;
import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import javax.swing.event.*;
import java.net.*;

/**
 * <b>Titel:</b>Pacman 3D - Leveleditor<p>
 *
 * <b>Beschreibung:</b><br>
 * Diese Klasse ist eine Subklasse von {@link javax.swing.JPanel JPanel} und
 * kann als solche GUI-Komponente verwendet werden.<br>
 * Diese Subklasse ist um diverse GUI-Komponenten und Funktionalitt erweitert,
 * so da durch sie die dem Leveleditor zur Verfgung stehenden Texturen und
 * Farben auf dem Bildschirm dargestellt und vom Benutzer ausgewhlt werden
 * knnen.<p>
 *
 * <b>Copyright:</b>	Copyright (c) 2001/02<br>
 *
 * @author              Labyrinth-Gruppe<br>
 * @version             1.0 03/15/2002<br>
 */
public class TextureSelectionJPanel extends JPanel implements java.awt.event.ActionListener {
  /**
   * Instanz der Klasse {@link java.lang.String String}, die, falls die gewhlte
   * Textur aus eine JPG-Datei erzeugt wurde, den Wert "image/jpeg" und
   * falls die Textur aus einer GIF-Datei erzeugt wurde, den Wert "image/gif"
   * trgt.
   */
  protected String mime_type = null;

  /**
   * Dateigre der der selektierten Textur zugrundeliegenden Bilddatei in
   * Bytes.
   */
  protected long size = 0;

  /**
   * Instanz der Klasse {@link java.net.URL URL}, die den Ort der der selektierten
   * Textur zugrundeliegenen Bilddatei angibt.
   */
  protected java.net.URL image_url = null;

  /**
   * Instanz der Klasse {@link javax.media.j3d.ImageComponent2D}, welche die
   * der selektierten Textur zugrundeliegende Bilddatei enthlt.
   */
  protected ImageComponent2D texImage = null;

  /**
   * @see pacman3d.labyrinth.cells.CellWall#black
   */
  static protected javax.vecmath.Color3f black    = pacman3d.labyrinth.cells.CellWall.black;

  /**
   * @see pacman3d.labyrinth.cells.CellWall#white
   */
  static protected javax.vecmath.Color3f white    = pacman3d.labyrinth.cells.CellWall.white;

  /**
   * Instanz der Klasse {@link javax.swing.JButton}, um die Texturierung der
   * selektierten Zellen im Vector {@link
   * pacman3d.labyrinth.leveleditor.LevelEditingJPanel#selectedCellsVector}
   * vorzunehmen.
   */
  protected JButton textureSelectedCellsButton;

  /**
   * Instanz der Klasse {@link javax.swing.JButton}, um die Kolorierung der
   * selektierten Zellen im Vector {@link
   * pacman3d.labyrinth.leveleditor.LevelEditingJPanel#selectedCellsVector}
   * vorzunehmen.
   */
  protected JButton colorSelectedCellsButton;

  /**
   * Hilfsvariable, die die Gesamtanzahl der eingelesenen Bilddateien angibt.
   */
  protected int numOfFiles;

  /**
   * Instanz der Klasse {@link javax.media.j3d.Appearance Appearance}, welche
   * zur Erzeugung einer Textur verwendet wird.
   */
  protected Appearance appearance_tmp = new Appearance();

  /**
   * Instanz der Klasse {@link pacman3d.labyrinth.leveleditor.PreviewJLabel
   * PreviewJLabel}, welche dazu verwendet wird, um vom Benutzer selektierte
   * Farben in dieser Kompomente darzustellen.
   */
  protected final PreviewJLabel label = new PreviewJLabel();

  /**
   * Akutell vom Benutzer selektierte Textur als Instanz von
   * {@link javax.media.j3d.Texture2D Texture2D}.
   */
  protected Texture2D selectedTexture;

  /**
   * Aktuell vom Benutzer selektierte Farbe als Instanz von
   * {@link javax.vecmath.Color3f Color3f}.
   */
  protected javax.vecmath.Color3f selectedColor = new javax.vecmath.Color3f(1.0f, 1.0f, 1.0f);

  /**
   * Instanz der Klasse {@link pacman3d.labyrinth.leveleditor.PreviewJLabel
   * PreviewJLabel}, welche dazu verwendet wird, um vom Benutzer selektierte
   * Texturen in dieser Kompomente darzustellen.
   */
  protected PreviewJLabel showSelectedTextureLabel;;

  /**
   * Array vom Typ {@link javax.swing.ImageIcon ImageIcon}, in welchem sich alle
   * eingelesenen und fr Texturen zur Verfgung stehende Bilddateien befinden,
   * allerdings als Instanzen der Klasse {@link javax.swing.ImageIcon ImageIcon}, um
   * diese einfacher in JComboBoxes darzustellen.
   */
  protected ImageIcon[][] imageIcons;

  /**
   * Referenz auf die {@link pacman3d.labyrinth.leveleditor.Leveleditor Hauptklasse
   * des Leveleditors}, welche diese Klasse instantiiert hat und auch in einer
   * entsprechenden Komponente darstellt.
   */
  protected Leveleditor leveleditor;

  /**
   * URL-Referenz auf das Basisverzeichnis, in dem sich die fr Texturen zur
   * Verfgung stehenden Bilddatein befinden.
   */
  protected URL urli = null;
  /**
   * Konstruktor:<p>
   *
   * Zur Verwendung dieser Komponente zur Textur- und Farbauswahl ist diese Klasse
   * nur mit einer Instanz auf die {@link pacman3d.labyrinth.leveleditor.Leveleditor
   * Hauptklasse des Leveleditors} zu instantiieren.
   *
   * @param leveleditor Referenz auf die Hauptklasse des Leveleditors.
   */
  public TextureSelectionJPanel(Leveleditor leveleditor) {
    this.leveleditor = leveleditor;
    // Hilfsmethode zum Bilderladen aufrufen &
    try {
      urli = new URL("file:" + leveleditor.imageSourceFile.toString());
    } catch (Exception e) {}
    // Bilder zur Texturauswahl einlesen
    loadImages(leveleditor.imageSourceFile);

    // dann GUI aufbauen.
    initGUI();
  }

  /**
   * Diese Hilfsmethode bernimmt das Einladen aller in den Unterverzeichnissen
   * des Verzeichnis <b>file</b> befindlichen Bilddatein in das Array {@link
   * #imageIcons}, um sie spter in dieser Komponente zwecks Auswahl in JComboBoxes
   * darzustellen.
   *
   * @param file File-Instanz, welche auf das Verzeichnis zeigt, in welchem sich
   *             zur Verfgung stehenden Bilddateien befinden.
   */
  private final void loadImages(File file) {
    pacman3d.util.Debug.out( getClass().getName(), "loading files and images from " +
        file.getAbsolutePath(), pacman3d.util.Debug.LEVEL_NOTICE );

    // Alle Unterverzeichnisse des durch "file" referierten Verzeichnisses
    // bestimmen... Dazu den "FileFitler" verwenden, der nur "Verzeichnisse"
    // in seine "Filelist" aufnimmt und zurckliefert.
    String[] subdirectories = file.list(new MyFileFilter(MyFileFilter.DIR));
    // Alle JPG/GIF-Dateien des durch "file" referierten Verzeichnisses
    // bestimmen... Dazu den "FileFitler" verwenden, der nur JPG/GIF-Dateien
    // in seine "Filelist" aufnimmt und zurckliefert.
    String[] files        = file.list(new MyFileFilter(MyFileFilter.GIF_AND_JPEG));

    // Test, ob sich in den entsprechenden Unterverzeichnissen berhaupt Bilder
    // befinden.
    String[] tmp_subdirectories = new String[subdirectories.length];
    int pos = 0;
    for (int i = 0; i < subdirectories.length; i++) {
      File currentDir = new File(file, subdirectories[i]);
      if (currentDir.list(new MyFileFilter(MyFileFilter.GIF_AND_JPEG)).length >= 1)
        tmp_subdirectories[pos++] = subdirectories[i];
    }
    subdirectories = new String[pos];
    System.arraycopy(tmp_subdirectories, 0, subdirectories, 0, pos);

    // ImageIcon Array anlegen -> 2dimensionales:
    // imageIcons[i][] mit i == Anzahl der Unterverzeichnisse von "file"
    imageIcons = new ImageIcon[subdirectories.length][];

    // Anzahl der Grafikdatein in den Unterverzeichnissen von "file" bestimmen
    // und entsprechend ImageIcon initialisieren.
    for (int i = 0; i < subdirectories.length; i++) {
      File currentDir = new File(file, subdirectories[i]);
      imageIcons[i] = new ImageIcon[currentDir.list(new MyFileFilter(MyFileFilter.GIF_AND_JPEG)).length];
    }

    // Grafikdatein der einzelnen Unterverzeichnisse bestimmen und einlesen:
    Toolkit toolkit = Toolkit.getDefaultToolkit();
    for (int i = 0; i < subdirectories.length; i++) {
      File currentDir = new File(file, subdirectories[i]);
      String[] filenames = currentDir.list(new MyFileFilter(MyFileFilter.GIF_AND_JPEG));
      imageIcons[i] = new ImageIcon[filenames.length];

      for (int j = 0; j < filenames.length; j++) {
        // Mit Toolkit Bild laden und skalieren, damit alle Bilder einheitlich angezeigt
        // werden knnen.
        Image image = (toolkit.getImage( (new File(
                                          new File(file,subdirectories[i]),
                                          filenames[j])
                                          ).getAbsolutePath() )).getScaledInstance(60, 60, Image.SCALE_FAST);

        // ImageIcon erzeugen
        imageIcons[i][j] = new ImageIcon(image);
        // Und entsprechendem ImageIcon den Pfad mitteilen, von
        // wo es geladen wurde, d.h. wo sich das durch durch das
        // entsprechende ImageIcon dargestellt Bild befindet.

        try {
          imageIcons[i][j].setDescription(this.urli.toString() + "/" +
                                          subdirectories[i] + "/" + filenames[j]
                                          );
        } catch (Exception hoho) {}
      }
    }
  }

  /**
   * Die durch das von dieser Klasse implementierte Interface {@link java.awt.event.ActionListener}
   * geforderte Methode, um auf {@link java.awt.event.ActionEvent ActionEvents} reagieren
   * zu knnen.<br>
   * Innerhalb dieser Methode erfolgt z.B. bei Auswahl einer bestimmt Textur, da
   * diese als Instanz von {@link javax.media.j3d.Texture2D Texture2D} abgelegt
   * wird und somti fr die weitere Verwendung zur Verfgung steht.
   *
   * @param actionEvent {@link java.awt.event.ActionEvent}
   *
   * @see java.awt.event.ActionListener
   */
  public void actionPerformed(java.awt.event.ActionEvent actionEvent) {
    // ActionEvent von einer JComboBox generiert -> Textur selektiert:
    if (actionEvent.getSource() instanceof JComboBox) {
      JComboBox comboBox = (JComboBox)actionEvent.getSource();
      // ImageIcon des selektierten Elements der selektierten JComboBox holen
      // und "Description" abfragen, welche beim Laden der ImageIcons mit
      // dem Pfad zum entsprechenden Bild belegt wurde.
      try {
        this.image_url = new URL( ((ImageIcon)comboBox.getSelectedItem()).getDescription());
      }
      catch (Exception exception) {
        pacman3d.util.Debug.out(this.getClass().getName(), exception);
      }
      // Textur2D aus dem entsprechend ausgewhltem Bild genierieren.
      loadTexture(this.image_url);
      // Selektierte Textur anzeigen.
      showSelectedTextureLabel.setImage(texImage.getImage());

      // Appearance-Instanz mit der geladenen Textur erzeugen.
      this.appearance_tmp = new Appearance();
      TextureAttributes texAttr = new TextureAttributes();
      texAttr.setTextureMode(TextureAttributes.MODULATE);
      this.appearance_tmp.setTextureAttributes(texAttr);
      this.appearance_tmp.setTexture(this.getTexture());
      this.appearance_tmp.setMaterial(new Material(white,black,white,white,10.0f));
    }
  }

  /**
   * Innerhalb dieser Methode erfolgt der Aufbau dieser Komponente, d.h. die
   * Initialsierung der GUI und die notwendigen Initialisierungen der bentigten
   * Klassen. Diese Methode wird vom Konstruktor dieser Klasse automatisch aufgerufen
   * und sollte somit fr den Endanwender von keinem Interesse sein.
   *
   * @see #TextureSelectionJPanel
   */
  protected void initGUI() {
    // Das Hauptlayout ist: BorderLayout
    this.setLayout(new BorderLayout());

    // Hauptlayout setzen
    GridBagLayout gblMain = new GridBagLayout();
    GridBagConstraints gbc;

    JPanel previewPanel2 = new JPanel();
    GridBagLayout gbl2 = new GridBagLayout();
    previewPanel2.setLayout(gbl2);

    JPanel texturePanel = new JPanel(new BorderLayout());
    JPanel buttonsPanel = new JPanel(new BorderLayout());
    JPanel colorPanel = new JPanel(new GridLayout(2,1));

    // TabbedPane in's Hauptpanel einfgen: Trgt Textur/Farbauswahlkomponenten
    JTabbedPane tabbedPane = new JTabbedPane(JTabbedPane.TOP);
    this.add(tabbedPane, BorderLayout.CENTER);

    // TexturPanel einfgen
    tabbedPane.addTab("Texturen", null, texturePanel, "Texturauswahl");
    // ColorChooser-Panel einfgen
    final JColorChooser colorChooser = new JColorChooser(new Color(selectedColor.x, selectedColor.y, selectedColor.z));
    tabbedPane.addTab("Farbe", null, colorChooser, "Farbauswahl");

    // PreviewPanel fr gewhlte Farbe:
    JPanel colorPreviewPanel = new JPanel(new FlowLayout());
    colorPreviewPanel.setBorder(BorderFactory.createTitledBorder("Farbe"));
    colorPreviewPanel.add(label);

    // Das PreviewPanel des ColorChooser lschen
    colorChooser.setPreviewPanel(new JPanel());

    // label als neues PreviewPanel des ColorChoosers verwenden
    colorChooser.getSelectionModel().addChangeListener( new ChangeListener() {
      public void stateChanged(ChangeEvent e) {
        selectedColor = new javax.vecmath.Color3f((float)colorChooser.getColor().getRed()/255,
                                                  (float)colorChooser.getColor().getGreen()/255,
                                                  (float)colorChooser.getColor().getBlue()/255);

        appearance_tmp = new Appearance();
        appearance_tmp.setMaterial(new Material(selectedColor,black,white,selectedColor,10.0f));
        label.setColor(colorChooser.getColor());
        leveleditor.itemJPanel.signalLampColorLabel.setColor(selectedColor);
      }
    });

    this.add(previewPanel2, BorderLayout.EAST);

    gbc = makeGBC(0, 0, 1, 1);
    gbc.fill = GridBagConstraints.BOTH;
    gbc.anchor = GridBagConstraints.CENTER;
    gbc.weightx = 100;
    gbc.weighty = 100;
    gbl2.setConstraints(colorPreviewPanel, gbc);

    previewPanel2.add(colorPreviewPanel);

    // Panel fr die Checkboxes fr die Seitenauswahl der Wnde
    gbc = makeGBC(0, 1, 2, 1);
    gbc.fill = GridBagConstraints.HORIZONTAL;
    gbc.weightx = 100;
    gbc.anchor = GridBagConstraints.CENTER;
    this.add(buttonsPanel, BorderLayout.SOUTH);

    // aktuell selektierte Textur laden & darstellen
    try {
      loadTexture(new URL(imageIcons[0][0].getDescription()));
    } catch (Exception urlexcep) {}

    try {
      this.image_url = new URL(imageIcons[0][0].getDescription());
    } catch (Exception hoho) {}

    showSelectedTextureLabel = new PreviewJLabel("leer", JLabel.CENTER);
    showSelectedTextureLabel.setImage(((ImageComponent2D)this.getTexture().getImage(0)).getImage());

    TextureAttributes texAttr = new TextureAttributes();
    texAttr.setTextureMode(TextureAttributes.MODULATE);
    this.appearance_tmp.setTextureAttributes(texAttr);
    this.appearance_tmp.setTexture(this.getTexture());
    // Farbe und Material werden gesetzt
    this.appearance_tmp.setMaterial(new Material(white,black,white,white,10.0f));

    JPanel centeredPanel = new JPanel();
    centeredPanel.setBorder(BorderFactory.createTitledBorder("Textur"));

    gbc = makeGBC(0, 1, 1, 1);
    gbc.fill = GridBagConstraints.BOTH;
    gbc.anchor = GridBagConstraints.CENTER;
    gbc.weightx = 100;
    gbc.weighty = 100;
    centeredPanel.add(showSelectedTextureLabel);
    gbl2.setConstraints(centeredPanel, gbc);

    previewPanel2.add(centeredPanel);

    // Panel mit den ComboBoxes fr die Texturauswahl
    JPanel comboBoxPanel = new JPanel(new GridLayout(2,5));
    JScrollPane scrollPane = new JScrollPane(comboBoxPanel);
    scrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
    scrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);

    // Eingeladene Bilder den JComboBoxes hinzufgen
    for (int i = 0; i < imageIcons.length; i++) {
      JComboBox box = new JComboBox(imageIcons[i]);
      box.addActionListener(this);
      comboBoxPanel.add(box);
    }

    texturePanel.add(scrollPane, BorderLayout.CENTER);

    // TextureMode Buttons
    JPanel dummyPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
    buttonsPanel.add(dummyPanel, BorderLayout.SOUTH);
    textureSelectedCellsButton = new JButton("Textur zuweisen");
    textureSelectedCellsButton.setActionCommand("texture cell");
    dummyPanel.add(textureSelectedCellsButton);

    colorSelectedCellsButton = new JButton("Farbe zuweisen");
    colorSelectedCellsButton.setActionCommand("color cell");
    dummyPanel.add(colorSelectedCellsButton);
  }

  /**
   * Hilfsmethode, die das erzeugen einer von Benutzer selektierten Textur
   * vornimmt, wozu der Ort der zugrundeliegenden Bilddatei ber eine Instanz
   * der Klasse {@link java.net.URL URL} anzugeben ist.
   *
   * @param url Ort der der Textur zugrundeliegenden Bilddatei.
   */
  private final void loadTexture(URL url) {
    File file = new File(url.getPath());
    this.size = file.length();

    String string =  new String (file.toString());
    if (string.endsWith("jpg") || string.endsWith("jpeg"))
      this.mime_type = "image/jpeg";
    else
      this.mime_type = "image/gif";

    TextureLoader loader = new TextureLoader(url, TextureLoader.BY_REFERENCE, new Frame());

    texImage = loader.getImage();

    if(texImage == null) {
      pacman3d.util.Debug.out(this.getClass().getName(), "cannont load image");
    }
    else {
      // Textur-Instanz anlegen
      Texture2D texture = new Texture2D(Texture.BASE_LEVEL, Texture.RGB, texImage.getWidth(), texImage.getHeight());
      // Textur setzen
      texture.setImage(0, texImage);
      selectedTexture = texture;
    }
  }

  /**
   * Liefert die vom Benutzer selektierte Farbe als Instanz der Klasse
   * {@link javax.vecmath.Color3f Color3f} zurck.
   *
   * @return Die vom Benutzer selektierte Farbe als Instanz der Klasse
   *         {@link javax.vecmath.Color3f Color3f}.
   */
  public javax.vecmath.Color3f getColor3f() {
    return this.selectedColor;
  }

  /**
   * Liefert die vom Benutzer selektierte Farbe als Instanz der Klasse
   * {@link java.awt.Color Color} zurck.
   *
   * @return Die vom Benutzer selektierte Farbe als Instanz der Klasse
   *         {@link java.awt.Color Color}.
   */
  public Color getColor() {
    return new Color(this.selectedColor.x, this.selectedColor.y, this.selectedColor.z);
  }

  /**
   * Mittels dieser Methode wird der ActionListener {@link java.awt.event.ActionListener
   * actionListener} an den JButtons {@link #textureSelectedCellsButton} und
   * {@link #colorSelectedCellsButton} als ActionListener registriert.
   *
   * @param actionListener ActionListener, der an den ActionEvents der JButtons
   *                      {@link #textureSelectedCellsButton} und
   *                      {@link #colorSelectedCellsButton} interessiert ist.
   */
  protected void addActionListener(java.awt.event.ActionListener actionListener) {
    textureSelectedCellsButton.addActionListener(actionListener);
    colorSelectedCellsButton.addActionListener(actionListener);
  }

  /**
   * Mittels dieser Methode wird der ActionListener {@link java.awt.event.ActionListener
   * actionListener} von den JButtons {@link #textureSelectedCellsButton} und
   * {@link #colorSelectedCellsButton} entfernt.
   *
   * @param actionListener ActionListener, der von JButtons
   *                      {@link #textureSelectedCellsButton} und
   *                      {@link #colorSelectedCellsButton} entfernt werden soll.
   */
  protected void removeActionListener(java.awt.event.ActionListener actionListener) {
    textureSelectedCellsButton.removeActionListener(actionListener);
    colorSelectedCellsButton.removeActionListener(actionListener);
  }

  /**
   * Diese Methode liefert die vom Benutzer selektierte Textur als Instanz
   * der Klasse {@link javax.media.j3d.Texture2D Texture2D} zurck.
   *
   * @return  Die vom Benutzer selektierte Textur als Instanz
   *          der Klasse {@link javax.media.j3d.Texture2D Texture2D}.
   */
  public Texture2D getTexture() {
    return selectedTexture;
  }

  /**
   * Hilfsmethode fr den GridBagLayout-Manager.
   *
   * @param x x-Koordinate der einzufgenden Komponente.
   * @param y y-Koordinate der einzufgenden Komponente.
   * @param width Horizontal Ausbreitung der einzufgenden Komponente.
   * @param height Vertikale Ausbreitung der einzufgenden Komponente.
   *
   * @return java.awt.GridBagConstraints Instanz.
   */
  private GridBagConstraints makeGBC(int x, int y, int width, int height) {
    GridBagConstraints gbc = new GridBagConstraints();
    gbc.gridx = x;
    gbc.gridy = y;
    gbc.gridwidth = width;
    gbc.gridheight = height;
    gbc.insets = new Insets(1, 1, 1, 1);
    return gbc;
  }
/*
  protected void setAppearance(Appearance appearance) {
    if (appearance != null) {
      if (appearance.getTexture() != null) {
        this.selectedTexture = (Texture2D)appearance.getTexture();
        showSelectedTextureLabel.setImage(((ImageComponent2D)selectedTexture.getImage(0)).getImage());
      }
      else if (appearance.getMaterial() != null) {
        appearance.getMaterial().getAmbientColor(this.selectedColor);
        this.label.setColor(new Color(selectedColor.x, selectedColor.y, selectedColor.z));
      }
    }
  }
  public Appearance getAppearance() {
    return this.appearance_tmp;
  }
*/
}











