package pacman3d.labyrinth.leveleditor;

import javax.swing.*;
import java.awt.*;
import java.awt.image.*;
import javax.vecmath.*;

/**
 * <b>Titel:</b>Pacman 3D - Leveleditor<p>
 *
 * <b>Beschreibung:</b><br>
 * Diese Klasse ist eine Subklasse von {@link javax.swing.JPanel JPanel} und
 * kann als solche GUI-Komponente verwendet werden.<br>
 * Diese Subklasse bietet die Mglichkeit, innerhalb der durch sie dargestellten
 * GUI-Komponente auf dem Bildschirm in einem rechteckigen Bereich Farben und
 * Bilder darzustellen. Sie wird vom Leveleditor insofern verwendet, so da vom
 * Benutzer ausgewhlte Farben und Texturen angezeigt werden knnen.<p>
 *
 * <b>Copyright:</b>	Copyright (c) 2001/02<br>
 *
 * @author              Labyrinth-Gruppe<br>
 * @version             1.0 03/15/2002<br>
 */
public class PreviewJLabel extends JLabel {
  /**
   * Initiale Gre dieser Komponente, die 60 x 60 Pixel betrgt.
   */
  protected Dimension size = new Dimension(60, 60);

  /**
   * Referenz auf diejenige Instanz von {@link java.awt.Image Image}, die
   * durch diese Komponente dargestellt werden soll.
   */
  protected Image image = null;

  /**
   * Referenz auf diejenige Instanz von {@link java.awt.Color Color}, die
   * durch diese Komponente dargestellt werden soll.
   */
  protected Color color = Color.white;

  /**
   * Referenz auf diejenige Instanz von {@link javax.vecmath.Color3f Color3f},
   * die durch diese Komponente dargestellt werden soll.
   */
  protected Color3f color3f = new Color3f(1.0f, 1.0f, 1.0f);

  /**
   * Konstruktor: <p>
   *
   * Instantiierung einer neuen Instanz dieser Klasse, wobei initial der Text
   * <b>text</b> mit der Ausrichtung <b>horizontalAlignment</b> angezeigt wird,
   * falls kein Bild oder keine Farbe dargestellt werden soll.
   *
   * @param text Anzuzeigender Text, falls kein Bild oder keine Farbe dargestellt
   *             werden soll.
   * @param horizontalAlignment Ausrichtung des evtl. anzuzeigenden Textes <b>text</b>.
   *
   * @see #PreviewJLabel()
   * @see #PreviewJLabel(Dimension size)
   */
  public PreviewJLabel(String text, int horizontalAlignment) {
    super(text, horizontalAlignment);
    this.setSize(size);
    super.setSize(size);
    color = null;
  }

  /**
   * Konstruktor: <p>
   *
   * Instantiierung einer neuen Instanz dieser Klasse, wobei die Komponente die
   * initiale Gre <b>size</b> trgt und nichts angezeigt wird, d.h. keine
   * Farbe, Textur oder Text.
   *
   * @param size Initiale Gre dieser Komponente.
   *
   * @see #PreviewJLabel(String text, int horizontalAlignment)
   * @see #PreviewJLabel()
   */
  public PreviewJLabel(Dimension size) {
    this.size = size;
    this.setSize(size);
  }

  /**
   * Konstruktor: <p>
   *
   * Instantiierung einer neuen Instanz dieser Klasse mit der initialen Gre
   * {@link #size}, wobei nichts angezeigt wird, d.h. keine Farbe, keine Textur
   * und kein Text.
   *
   * @see #PreviewJLabel(String text, int horizontalAlignment)
   * @see #PreviewJLabel(Dimension size)
   */
  public PreviewJLabel() {
    super();
    this.setSize(size);
  }

  /**
   * Liefert die prferierte Gre dieser Komponente als Instanz von
   * {@link java.awt.Dimension} zurck.
   *
   * @return Prferierte Gre dieser Komponente.
   */
  public Dimension getPreferredSize() {
    return size;
  }

  /**
   * Liefert die minimale Gre dieser Komponente als Instanz von
   * {@link java.awt.Dimension} zurck.
   *
   * @return Minimale Gre dieser Komponente.
   */
  public Dimension getMinimalSize() {
    return size;
  }

  /**
   * Liefert die maximale Gre dieser Komponente als Instanz von
   * {@link java.awt.Dimension} zurck.
   *
   * @return Maximale Gre dieser Komponente.
   */
  public Dimension getMaximumSize() {
    return size;
  }

  /**
   * Liefert die Gre dieser Komponente als Instanz von
   * {@link java.awt.Dimension} zurck.
   *
   * @return Gre dieser Komponente.
   */
  public Dimension getSize() {
    return size;
  }

  /**
   * Von {@link javax.swing.JLabel JLabel} geerbte und berschriebene Methode, die
   * immer dann automatisch aufgerufen wird, wenn sich diese Komponente darstellen
   * soll.
   *
   * @param g Graphics-Instanz mit Referenz auf den Zeichenbereich.
   */
  public void paint(Graphics g) {
    super.paint(g);
    if (image != null) {
      Image image2 = image.getScaledInstance(size.width, size.height, Image.SCALE_FAST);
      g.drawImage(image2, 0, 0, Color.lightGray, new Frame());
    }
    else if (color != null) {
      g.setColor(color);
      g.fillRect(0, 0, size.width, size.height);
    }
  }

  /**
   * bergabe derjenigen {@link java.awt.Image Image-Instanz}, welche durch diese
   * Komponente dargestellt werden soll.
   *
   * @param image Darzustellendes Bild.
   */
  public void setImage(Image image) {
    if (image != null) {
      this.image = image;
      color = null;
    }
    repaint();
  }

  /**
   * bergabe derjenigen {@link java.awt.Color Color-Instanz}, welche durch diese
   * Komponente dargestellt werden soll.
   *
   * @param color Darzustellende Farbe.
   */
  public void setColor(Color color) {
    if (color != null) {
      this.color3f = new Color3f(color);
      this.color = color;
      this.image = null;
    }
    repaint();
  }

  /**
   * bergabe derjenigen {@link javax.vecmath.Color3f Color3f-Instanz}, welche
   * durch diese Komponente dargestellt werden soll.
   *
   * @param color3f Darzustellende Farbe.
   */
  public void setColor(Color3f color3f) {
    if (color3f != null) {
      this.color3f = color3f;
      this.color = new Color(color3f.x, color3f.y, color3f.z);
      this.image = null;
    }
    repaint();
  }

  /**
   * bergabe derjenigen {@link java.lang.String String-Instanz}, welche als
   * Text in dieser Komponente dargestellt werden soll, falls keine Bild oder
   * keine Farbe dargestellt wird.
   *
   * @param image Darzustellendes Bild.
   */
  public void setText(String text) {
    super.setText(text);
    image = null;
    color = null;
    repaint();
  }

  /**
   * Liefert die durch diese Komponente dargestellte Farbe als Instanz von
   * {@link javax.vecmath.Color3f Color3f} zurck.
   *
   * @return Dargestellte Farbe als Instanz von {@link javax.vecmath.Color3f}.
   */
  public Color3f getColor3f() {
    return color3f;
  }

  /**
   * Liefert die durch diese Komponente dargestellte Farbe als Instanz von
   * {@link java.awt.Color Color} zurck.
   *
   * @return Dargestellte Farbe als Instanz von {@link java.awt.Color}.
   */
  public Color getColor() {
    return color;
  }
}