package pacman3d.labyrinth.leveleditor;

import java.awt.geom.*;
import pacman3d.labyrinth.*;
import pacman3d.message.*;
import pacman3d.monster.*;
import pacman3d.pacman.*;
import pacman3d.util.*;
import pacman3d.labyrinth.items.*;
import pacman3d.labyrinth.cells.*;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.event.*;
import javax.vecmath.*;
import javax.media.j3d.*;
import com.sun.j3d.utils.image.TextureLoader;
import java.awt.image.*;
import java.util.*;

/**
 * <b>Titel:</b>Pacman 3D - Leveleditor<p>
 *
 * <b>Beschreibung:</b><br>
 * Diese Klasse ist eine Subklasse von {@link javax.swing.JPanel JPanel} und
 * kann als solche GUI-Komponente verwendet werden.<br>
 * Diese Subklasse stellt diejenige Komponente dar, welche zur 2D-Leveleditierung
 * mittels der Maus verwendet wird. Dabei wird ein zu editierendes {@link
 * pacman3d.labyrinth.Labyrinth Labyrinth} ebenenweise auf dem Bildschirm
 * dargestellt. Genau genommen, stellt diese Klasse das "Herzstck" des
 * Leveleditors dar, da durch diese Klasse diejenige Funktionalitt implementiert
 * ist, die z.B. bentigt wird, um in dem zu editierenden Labyrinth verschiedene
 * {@link pacman3d.labyrinth.cells.Cell Zellentypen} und {@link
 * pacman3d.labyrinth.items.Item Items} zu setzen, {@link
 * pacman3d.labyrinth.cells.Cell Zellen} eine Farbe oder Textur zuzuweisen und
 * auch, um z.B. {@link pacman3d.monster.Monster Monster} in einem Labyrinth
 * zu setzen.<p>
 *
 * <b>Copyright:</b>	Copyright (c) 2001/02<br>
 *
 * @author              Labyrinth-Gruppe<br>
 * @version             1.0 03/15/2002<br>
 */
public class LevelEditingJPanel extends JPanel implements MouseMotionListener, MouseListener, java.awt.event.ActionListener {
  /**
   * Referenz auf die vom Benutzer durch diese Klasse editierte Instanz der
   * Klasse {@link pacman3d.labyrinth.Labyrinth Labyrinth}.
   *
   * @see pacman3d.labyrinth.Labyrinth
   */
  protected Labyrinth lab;

  /**
   * Boolscher Flag, ob gesetzte Pacmanstartpositionen in der 2D-Darstellung
   * des vom Benutzer bearbeiteten Labyrinhts ein- oder ausgeblendet werden sollen.
   * Trgt die Variable den Wert <b>true</b>, so werden gesetzte Pacmanstartpositionen
   * angezeigt, andernfalls nicht.
   */
  private boolean drawPacmanStartPositions = true;

  /**
   * Boolscher Flag, der den Wert <b>true</b> trgt, falls der Benutzer gerade
   * eine Pacmanstartposition setzt und Pacman-Blickrichtung nach Norden zeigt.
   *
   * @see #west
   * @see #south
   * @see #east
   */
  private boolean north = false;

  /**
   * Boolscher Flag, der den Wert <b>true</b> trgt, falls der Benutzer gerade
   * eine Pacmanstartposition setzt und Pacman-Blickrichtung nach Westen zeigt.
   *
   * @see #north
   * @see #south
   * @see #east
   */
  private boolean west  = false;

  /**
   * Boolscher Flag, der den Wert <b>true</b> trgt, falls der Benutzer gerade
   * eine Pacmanstartposition setzt und Pacman-Blickrichtung nach Sden zeigt.
   *
   * @see #west
   * @see #north
   * @see #east
   */
  private boolean south = false;

  /**
   * Boolscher Flag, der den Wert <b>true</b> trgt, falls der Benutzer gerade
   * eine Pacmanstartposition setzt und Pacman-Blickrichtung nach Osten zeigt.
   *
   * @see #west
   * @see #south
   * @see #north
   */
  private boolean east  = false;

  //protected boolean setMonsterStartposition = false;

  /**
   * Boolscher Flag, der den Wert <b>true</b> trgt, falls gerade vom Benutzer
   * eine Pacmanstartposition gesetzt wird.
   */
  private boolean setPacmanStartposition = false;

  /**
   * Instanz der Klasse {@link java.awt.Point Point}, die den Endpunkt des Pfeiles
   * der vom Benutzer gesetzten Blickrichtung einer evtl. Pacmanstartposition trgt.
   *
   * @see java.awt.Point
   */
  private Point arrowEndPoint;


  //protected boolean guiUpdated = true;

  /**
   * Referenz auf diejenige Instanz der Klasse
   * {@link pacman3d.labyrinth.leveleditor.Leveleditor Leveleditor}, die diese
   * Klasse zwecks 2D-Editierung eines neuen Labyrinthes instantiiert hat. Diese
   * Referenz wird dazu verwendet, um sich auf diverse andere Klassen des Leveleditors
   * Zugriff zu verschaffen.
   *
   * @see pacman3d.labyrinth.leveleditor.Leveleditor
   */
  protected Leveleditor leveleditor;

  /**
   * Farbe des evtl. whrend der 2D-Leveleditierung angezeigten Gitternetzes.
   *
   * @see java.awt.Color
   */
  protected static Color colorGrid   = Color.blue;

  /**
   * Farbe der evtl. whrend der 2D-Leveleditierung angezeigten und gesetzten
   * Items des editierten Labyrinthes.
   *
   * @see java.awt.Color
   */
  protected static Color colorItem   = new Color(0.0f, 0.5f, 0.0f);

  /**
   * Farbe der evtl. whrend der 2D-Leveleditierung angezeigten und gesetzten
   * Pacmanstartpositionen des editierten Labyrinthes.
   *
   * @see java.awt.Color
   */
  protected static Color colorPacmanStartPosition = Color.yellow;

  /**
   * Farbe der evtl. whrend der 2D-Leveleditierung angezeigten und gesetzten
   * Monster des editierten Labyrinthes.
   *
   * @see java.awt.Color
   */
  protected static Color colorMonster = new Color(0.0f, 1.0f, 1.0f);

  /**
   * Farbe der evtl. whrend der 2D-Leveleditierung angezeigten und gesetzten
   * Monsterstartpositionen des editierten Labyrinthes.
   *
   * @see java.awt.Color
   */
  protected static Color colorMonsterStartPosition = new Color(1.0f, 0.0f, 0.0f);

  /**
   * Farbe des Mausfangs, welcher den Bereich der Selektion mittels Mausdrag
   * anzeigt.
   *
   * @see java.awt.Color
   */
  protected static Color colorMouse  = new Color(0.0f, 0.0f, 1.0f);

  /**
   * Farbe des Zellentyps {@link pacman3d.labyrinth.cells.CellWall CellWall}.
   *
   * @see java.awt.Color
   * @see pacman3d.labyrinth.cells.CellWall
   */
  protected static Color colorWall   = Color.darkGray;

  /**
   * Farbe des evtl. whrend der 2D-Leveleditierung angezeigten Gitternetzes.
   *
   * @see java.awt.Color
   */
  protected static Color colorLowerWalls = colorWall.brighter();

  /**
   * Farbe des evtl. whrend der 2D-Leveleditierung angezeigten Gitternetzes.
   *
   * @see java.awt.Color
   */
  protected static Color colorUpperWalls = colorWall.brighter();

  /**
   * Farbe des Zellentyps {@link pacman3d.labyrinth.cells.CellWall CellWall}.
   *
   * @see java.awt.Color
   * @see pacman3d.labyrinth.cells.CellFloor
   */
  protected static Color colorFloor  = Color.lightGray;

  /**
   * Farbe des evtl. whrend der 2D-Leveleditierung angezeigten Gitternetzes.
   *
   * @see java.awt.Color
   */
  protected static Color colorLowerFloors = colorFloor.darker();

  /**
   * Farbe des evtl. whrend der 2D-Leveleditierung angezeigten Gitternetzes.
   *
   * @see java.awt.Color
   */
  protected static Color colorUpperFloors = colorFloor.darker();

  /**
   * Farbe der Zellentypen {@link pacman3d.labyrinth.cells.CellArchwayHorizontal CellArchwayHorizontal},
   * {@link pacman3d.labyrinth.cells.CellArchwayVertical CellArchwayVertical} und
   * {@link pacman3d.labyrinth.cells.CellArchwayConnection CellArchwayConnection}.
   *
   * @see java.awt.Color
   * @see pacman3d.labyrinth.cells.CellArchwayHorizontal
   * @see pacman3d.labyrinth.cells.CellArchwayVertical
   * @see pacman3d.labyrinth.cells.CellArchwayConnection
   */
  protected static Color colorArchway = Color.lightGray.darker();

  /**
   * Farbe des Zellentyps {@link pacman3d.labyrinth.cells.CellSimpleCone CellSimpleCone}.
   *
   * @see java.awt.Color
   * @see pacman3d.labyrinth.cells.CellSimpleCone
   */
  protected static Color colorSimplePillar = Color.darkGray;

  /**
   * Farbe der {@link pacman3d.labyrinth.cells.CellSimpleWindowHorizontal
   * CellSimpleWindowHorizontal} und {@link
   * pacman3d.labyrinth.cells.CellSimpleWindowVertical CellSimpleWindowVertical}.
   *
   * @see java.awt.Color
   * @see pacman3d.labyrinth.cells.CellSimpleWindowHorizontal
   * @see pacman3d.labyrinth.cells.CellSimpleWindowVertical
   */
  protected static Color colorSimpleWindow = Color.darkGray;

  /**
   * Farbe des Zellentyps {@link pacman3d.labyrinth.cells.CellSimpleCone CellSimpleCone}.
   *
   * @see java.awt.Color
   * @see pacman3d.labyrinth.cells.CellSimpleCone
   */
  protected static Color colorCone         = Color.darkGray;

  /**
   * Farbe der Zellentypen {@link  pacman3d.labyrinth.cells.CellLadderNorth CellLadderNorth},
   * {@link  pacman3d.labyrinth.cells.CellLadderEast CellLadderEast},
   * {@link  pacman3d.labyrinth.cells.CellLadderSouth CellLadderSouth},
   * {@link  pacman3d.labyrinth.cells.CellLadderWest CellLadderWest}.
   *
   * @see java.awt.Color
   * @see pacman3d.labyrinth.cells.CellLadderNorth
   * @see pacman3d.labyrinth.cells.CellLadderEast
   * @see pacman3d.labyrinth.cells.CellLadderSouth
   * @see pacman3d.labyrinth.cells.CellLadderWest
   */
  protected static Color colorLadder       = Color.gray;

  /**
   * Farbe des Zellentyps {@link pacman3d.labyrinth.cells.CellGlassWall CellGlassWall}.
   *
   * @see java.awt.Color
   * @see pacman3d.labyrinth.cells.CellGlassWall
   */
  protected static Color colorGlassWall    = Color.darkGray;

  /**
   * Farbe der Zellentypen {@link  pacman3d.labyrinth.cells.CellSimpleDoorNorth CellSimpleDoorNorth},
   * {@link  pacman3d.labyrinth.cells.CellSimpleDoorEast CellSimpleDoorEast},
   * {@link  pacman3d.labyrinth.cells.CellSimpleDoorSouth CellSimpleDoorSouth},
   * {@link  pacman3d.labyrinth.cells.CellSimpleDoorWest CellSimpleDoorWest}.
   *
   * @see java.awt.Color
   * @see pacman3d.labyrinth.cells.CellSimpleDoorNorth
   * @see pacman3d.labyrinth.cells.CellSimpleDoorEast
   * @see pacman3d.labyrinth.cells.CellSimpleDoorSouth
   * @see pacman3d.labyrinth.cells.CellSimpleDoorWest
   */
  protected static Color colorSimpleDoor   = Color.gray;

  /**
   * Farbe des Zellentyps {@link pacman3d.labyrinth.cells.CellHalfWall CellHalfWall}.
   *
   * @see java.awt.Color
   * @see pacman3d.labyrinth.cells.CellHalfWall
   */
  protected static Color colorQuaterHighWall = Color.blue;


  /**
   * Boolscher Flag, der den Wert <b>true</b> trgt, falls gesetzte Items
   * angezeigt werden sollen.
   */
  protected boolean drawItems = true;

  /**
   * Boolscher Flag, der den Wert <b>true</b> trgt, falls gesetzte Monster
   * angezeigt werden sollen.
   */
  protected boolean drawMonsters = true;

  /**
   * Boolscher Flag, der den Wert <b>true</b> trgt, falls begehbare Zellen der
   * Levelebene unterhalb der aktuell editierten Levelebene in diese eingeblendet
   * werden sollen.
   */
  protected boolean drawLowerLevelFloors = false;

  /**
   * Boolscher Flag, der den Wert <b>true</b> trgt, falls nicht begehbare Zellen
   * der Levelebene unterhalb unterhalb der aktuell editierten Levelebene in diese
   * eingeblendet werden sollen.
   */
  protected boolean drawLowerLevelWalls = false;

  /**
   * Boolscher Flag, der den Wert <b>true</b> trgt, falls begehbare Zellen der
   * Levelebene oberhalb der aktuell editierten Levelebene in diese eingeblendet
   * werden sollen.
   */
  protected boolean drawUpperLevelFloors = false;

  /**
   * Boolscher Flag, der den Wert <b>true</b> trgt, falls nicht begehbare Zellen
   * der Levelebene oberhalb der aktuell editierten Levelebene in diese eingeblendet
   * werden sollen.
   */
  protected boolean drawUpperLevelWalls = false;

  /**
   * Boolscher Flag, der den Wert <b>true</b> trgt, falls ein Gitternetz
   * eingeblendet werden soll.
   */
  protected boolean gridOn = true;

  /**
   * Integerwert, mit dem alle auf dem 2D-Leveleditierungsbereich dargestellten
   * Elemente multipliziert werden, um ein einfaches Zoomverhalten zu realisieren.
   */
  private int zoomFac = 30;

  /**
   * Aktuell editierte Levelebene.
   */
  private int editLevelNumber = 1;

  /**
   * Instanz der Klasse {@link pacman3d.labyrinth.leveleditor.CellVector CellVector},
   * in welchem alle vom Benutzer selektierten Zellen abgelegt werden, um sie evtl.
   * spter weiter zu bearbeiten.
   *
   * @see pacman3d.labyrinth.leveleditor.CellVector
   */
  private CellVector selectedCellsVector = new CellVector();

  /**
   * Referenz auf die vom Leveleditor zur Textur- & Farbauswahl verwendete Instanz
   * der Klasse {@link pacman3d.labyrinth.leveleditor.TextureSelectionJPanel
   * TextureSelectionJPanel}. Diese wird innerhalb einer Instanz dieser Klasse
   * dazu verwendet, um sich vom Benutzer ausgewhlte Texturen bzw. Farben zu holen
   * und den vom ihm ausgewhlten Zellentypen zuzuweisen.
   *
   * @see pacman3d.labyrinth.leveleditor.TextureSelectionJPanel
   */
  private TextureSelectionJPanel texSelecterPanel;

  // Referenz auf "AppearanceJPanel", welches die Mglichkeit zur
  // Editierung der Appearances der CellWalls bietet.
  //protected AppearanceJPanel appJPanel;

  // Referenz auf "ItemJPanel", welches die Mglichkeit zur Selektion
  // eines Items bietet, welches dann in CellFloor-Cells gesetzt werden kann
  private ItemJPanel itemJPanel;

  /**
   * Instanz der Klasse {@link javax.swing.JPopupMenu JPopupMenu}, welche dazu
   * verwendet wird, um auf dem 2D-Leveleditierungsbereich ein Popup-Men anzuzeigen,
   * falls es der Kontext verlangt.
   *
   * @see javax.swing.JPopupMenu
   */
  private JPopupMenu popup;

  /**
   * Instanz der Klasse {@link java.awt.Point Point}, welche den Startpunkt bei
   * einer Selektion mittels Maus-Drag auf dem 2D-Leveleditierungsbereich trgt.
   *
   * @see java.awt.Point
   */
  private Point pointStart;

  /**
   * Instanz der Klasse {@link java.awt.Point Point}, welche den Endpunkt bei
   * einer Selektion mittels Maus-Drag auf dem 2D-Leveleditierungsbereich trgt.
   *
   * @see java.awt.Point
   */
  private Point pointEnd;

  /**
   * Boolscher Flag, der den Wert <b>true</b> trgt, falls der Benutzer auf dem
   * 2D-Darstellungsbereich mittels Maus-Drag eine Selektion diverser Levelelemente
   * ausfhrt.
   */
  private boolean isMouseDragged = false;

  /**
   * Vector mit allen im Labyrinth gesetzten Monsterstartpositionen.
   */
  private Vector vecMonsterStartpositions = null;

  /**
   * Vector mit allen im Labyrinth gesetzten Pacmanstartpositionen.
   */
  private Vector vecPacmanStartpositions = null;

  /**
   * x-Koordinate eines Mausklicks auf dem 2D-Leveleditierungsbereich.
   */
  private int x;

  /**
   * y-Koordinate eines Mausklicks auf dem 2D-Leveleditierungsbereich.
   */
  private int z;

  /**
   * Konstruktor:<br>
   *
   * Um eine Instanz dieser Klasse zur 2D-Leveleditierung einer Instanz der Klasse
   * {@link pacman3d.labyrinth.Labyrinth Labyrinth} zu verwenden, ist diese Klasse
   * zu instantiieren mit einer Referenz auf eine Instanz der Klasse {@link
   * pacman3d.labyrinth.leveleditor.Leveleditor Leveleditor}, welche diverse
   * Elemente bietet, um z.B. einen bestimmten {@link pacman3d.labyrinth.cells.Cell Zellentyp}
   * oder {@link pacman3d.labyrinth.items.Item Item} auszuwhlen.<br>
   * Diese Klasse ist eine Subklasse von {@link javax.swing.JPanel JPanel} und kann
   * als solche wie jede andere Swing-Komponente verwendet werden.
   *
   * @param lab         pacman3d.labyrinth.leveleditor.Leveleditor
   */
  public LevelEditingJPanel(Leveleditor leveleditor) {
    super();

    // bergebene Referenzen ablegen...
    this.leveleditor = leveleditor;
    this.itemJPanel = leveleditor.itemJPanel;
    this.lab = leveleditor.lab;

    // Weitere GUI-Vorbereitungen
    guiInit();
  }

  /**
   * Von {@link javax.swing.JPanel JPanel} geerbet und berschriebene Methode.
   * Sie ist zustndig dafr, da eine 2D-Darstellung des edtierten Labyrinthes
   * gezeichnet wird. Diese Methode wird immer dann eigenstndig aufgerufen, wenn
   * sich diese Komponente darstellen soll, weshalb diese Methode nicht von Interesse
   * fr den Endanwender sein sollte.
   *
   * @param g Instanz der Klasse {@link java.awt.Graphics Graphics}
   * @see   javax.swing.JPanel#paint
   */
  public final void paint(Graphics g) {
    // Alle Zellen der editierten Ebene durchlaufen und je nach Zelltyp, entsprechend
    // farblich zeichnen:
    for (int x = 0; x < lab.getConstraints().x; x++)
      for (int z = 0; z < lab.getConstraints().z; z++) {
        Cell cell = lab.getCell(x, this.editLevelNumber, z);

        if (cell instanceof CellFloor) {
          g.setColor(colorFloor);
          g.fillRect(x*zoomFac, z*zoomFac, zoomFac, zoomFac);
        }
        else if (cell instanceof CellGlassWall) {
          g.setColor(this.colorGlassWall);
          g.fillRect(x*zoomFac, z*zoomFac, zoomFac, zoomFac);
        }
        else if (cell instanceof CellWall) {
          g.setColor(colorWall);
          g.fillRect(x*zoomFac, z*zoomFac, zoomFac, zoomFac);
        }
        else if (cell instanceof CellArchwayHorizontal ||
                 cell instanceof CellArchwayVertical ||
                 cell instanceof CellArchwayConnection) {
          g.setColor(colorArchway);
          g.fillRect(x*zoomFac, z*zoomFac, zoomFac, zoomFac);
        }
        else if (cell instanceof CellLadderEast ||
                 cell instanceof CellLadderNorth ||
                 cell instanceof CellLadderWest ||
                 cell instanceof CellLadderSouth) {
          g.setColor(this.colorLadder);
          g.fillRect(x*zoomFac, z*zoomFac, zoomFac, zoomFac);
        }
        else if (cell instanceof CellSimpleCone) {
          g.setColor(this.colorCone);
          g.fillRect(x*zoomFac, z*zoomFac, zoomFac, zoomFac);
        }
        else if (cell instanceof CellSimpleDoorEast ||
                 cell instanceof CellSimpleDoorNorth ||
                 cell instanceof CellSimpleDoorWest ||
                 cell instanceof CellSimpleDoorSouth) {
          g.setColor(this.colorSimpleDoor);
          g.fillRect(x*zoomFac, z*zoomFac, zoomFac, zoomFac);
        }
        else if (cell instanceof CellSimplePillar) {
          g.setColor(this.colorSimplePillar);
          g.fillRect(x*zoomFac, z*zoomFac, zoomFac, zoomFac);
        }
        else if (cell instanceof CellSimpleWindowHorizontal ||
                 cell instanceof CellSimpleWindowVertical) {
          g.setColor(this.colorSimpleWindow);
          g.fillRect(x*zoomFac, z*zoomFac, zoomFac, zoomFac);
        }
        else if (cell instanceof CellHalfWall) {
          g.setColor(this.colorQuaterHighWall);
          g.fillRect(x*zoomFac, z*zoomFac, zoomFac, zoomFac);
        }
        else if (lab.getCell(x, editLevelNumber, z).isWalkable()) {
          g.setColor(colorFloor);
          g.fillRect(x*zoomFac, z*zoomFac, zoomFac, zoomFac);
        }
        else {
          g.setColor(colorWall);
          g.fillRect(x*zoomFac, z*zoomFac, zoomFac, zoomFac);
        }
      }

    // Falls vom Benutzer Zellen selektiert wurden, so sind diese im Vector
    // "selectedCellsVector" als Referenzen abgelegt. Diesen arbeiten wir jetzt
    // ab und zeichnen die selektierten Zellen in einer gehighlighteten Farbe, d.h.
    // in ihrer ursprnglichen Darstellungsfarbe, nur etwas heller.
    if (selectedCellsVector.isEmpty() == false)
      for (int i = 0; i < selectedCellsVector.size(); i++) {
        Cell cell = (Cell)selectedCellsVector.get(i);
        Point point = new Point(cell.getPosition().x, cell.getPosition().z);

        if (cell instanceof CellFloor) {
          g.setColor(colorFloor.brighter());
          g.fillRect(point.x*zoomFac, point.y*zoomFac, zoomFac, zoomFac);
        }
        else if (cell instanceof CellGlassWall) {
          g.setColor(this.colorGlassWall.brighter());
          g.fillRect(point.x*zoomFac, point.y*zoomFac, zoomFac, zoomFac);
        }
        else if (cell instanceof CellWall) {
          g.setColor(colorWall.brighter());
          g.fillRect(point.x*zoomFac, point.y*zoomFac, zoomFac, zoomFac);
        }
        else if (cell instanceof CellArchwayHorizontal ||
                 cell instanceof CellArchwayVertical ||
                 cell instanceof CellArchwayConnection) {
          g.setColor(colorArchway.brighter());
          g.fillRect(point.x*zoomFac, point.y*zoomFac, zoomFac, zoomFac);
        }
        else if (cell instanceof CellLadderEast ||
                 cell instanceof CellLadderNorth ||
                 cell instanceof CellLadderWest ||
                 cell instanceof CellLadderSouth) {
          g.setColor(this.colorLadder.brighter());
          g.fillRect(point.x*zoomFac, point.y*zoomFac, zoomFac, zoomFac);
        }
        else if (cell instanceof CellSimpleCone) {
          g.setColor(this.colorCone.brighter());
          g.fillRect(point.x*zoomFac, point.y*zoomFac, zoomFac, zoomFac);
        }
        else if (cell instanceof CellSimpleDoorEast ||
                 cell instanceof CellSimpleDoorNorth ||
                 cell instanceof CellSimpleDoorWest ||
                 cell instanceof CellSimpleDoorSouth) {
          g.setColor(this.colorSimpleDoor.brighter());
          g.fillRect(point.x*zoomFac, point.y*zoomFac, zoomFac, zoomFac);
        }
        else if (cell instanceof CellSimplePillar) {
          g.setColor(this.colorSimplePillar.brighter());
          g.fillRect(point.x*zoomFac, point.y*zoomFac, zoomFac, zoomFac);
        }
        else if (cell instanceof CellSimpleWindowHorizontal ||
                 cell instanceof CellSimpleWindowVertical) {
          g.setColor(this.colorSimpleWindow.brighter());
          g.fillRect(point.x*zoomFac, point.y*zoomFac, zoomFac, zoomFac);
        }
        else if (cell instanceof CellHalfWall) {
          g.setColor(this.colorQuaterHighWall.brighter());
          g.fillRect(point.x*zoomFac, point.y*zoomFac, zoomFac, zoomFac);
        }
        else if (cell.isWalkable()) {
          g.setColor(colorFloor.brighter());
          g.fillRect(point.x*zoomFac, point.y*zoomFac, zoomFac, zoomFac);
        }
        else {
          g.setColor(colorWall.brighter());
          g.fillRect(point.x*zoomFac, point.y*zoomFac, zoomFac, zoomFac);
        }
      }

    // Will der Benutzer ein Gitternetz einblendet haben, so zeichnen wir dieses
    // ber alles vorhergehende drber...
    if (gridOn) {
      g.setColor(colorGrid);
      for (int y = 0; y <= lab.getConstraints().z; y++) {
        g.drawLine(0, y * this.zoomFac, lab.getConstraints().x * this.zoomFac, y * this.zoomFac);
      }
      for (int x = 0; x <= lab.getConstraints().x; x++) {
        g.drawLine(x * this.zoomFac, 0, x * this.zoomFac, lab.getConstraints().z * this.zoomFac);
      }
    }

    //
    // Test, ob Benutzer weitere Dinge bei der Editierung angezeigt haben mchte...
    // siehe dazu entsprechende Methoden...
    //
    // Untere begehbare Levelebene in aktuell editierte einblenden
    if (drawLowerLevelFloors == true)
      drawLowerLevelFloors(g);

    // Untere nicht begehbare Levelebene in aktuell editierte einblenden
    if (drawLowerLevelWalls == true)
      drawLowerLevelWalls(g);

    // Obere begehbare Levelebene in aktuell editierte Levelebene einblenden
    if (drawUpperLevelFloors == true)
      drawUpperLevelFloors(g);

    // Obere nicht begehbare Levelebene in aktuell editierte Levelebene einblenden
    if (drawUpperLevelWalls == true)
      drawUpperLevelWalls(g);

    // Gesetzte Monster und Monsterstartpositionen anzeigen
    if (drawMonsters == true) {
      drawMonsters(g);
      drawMonsterPositions(g);
    }

    // Gesetzte Items anzeigen
    if (drawItems == true)
      drawItems(g);

    // Gesetzte Pacmanstartpositionen anzeigen
    if (drawPacmanStartPositions)
      drawPacmanPositions(g);

    // Evtl. Methode aufrufen, die das interaktive Setzen einer Pacmanstartposition
    // ermglicht.
    if (this.setPacmanStartposition && this.arrowEndPoint != null)
      setPacmanStartposition(g);


    // Falls der Benutzer gerade mittels MouseDrag dabei ist, Zellen zu selektieren,
    // zeichne wir jetzt ber obige gezeichnete nichtselektierte und selektierte
    // Zellen, den Bereich, den der Benutzer selektiert hat. Dieser wird durch
    // ein nichtausgeflltest Rechteck dargestellt.
    if (isMouseDragged == true && pointStart != null && pointEnd != null) {
      // Anfangskoordinaten und Breit und Hhe des zu zeichnenden Rechtecks bestimmen...
      int x = (pointStart.x > pointEnd.x)? pointEnd.x : pointStart.x;
      int y = (pointStart.y > pointEnd.y)? pointEnd.y : pointStart.y;
      int width   = Math.abs(pointStart.x-pointEnd.x);
      int height  = Math.abs(pointStart.y-pointEnd.y);

      g.setColor(colorMouse);
      g.drawRect(x, y, width, height);
    }
  }

  /**
   * Diese Methode ermglicht dem Benutzer das interaktive Setzen einer neuen
   * Pacmanstartposition auf dem 2D-Leveleditierungsbereich. Diese Methode wird
   * automatisch vom der Methode {@link #paint paint(Grahics g)} aufgerufen.
   *
   * @param g Graphics-Instanz mit Referenz auf den Zeichenbereich.
   * @see   #paint
   */
  private final void setPacmanStartposition(Graphics g) {
    // Nur zeichnen, wenn auch der Benutzer interaktiv eine Pacmanstartposition
    // setzen will.
    if (this.setPacmanStartposition && this.arrowEndPoint != null) {
      Point3i point3i = (Point3i)((Cell)this.selectedCellsVector.get(0)).getPosition();
      Point startPoint = new Point(point3i.x*zoomFac + (int)(zoomFac/2), point3i.z*zoomFac + (int)(zoomFac/2));
      g.setColor(this.colorPacmanStartPosition);

      int arrowLineThickness = (int)zoomFac/12;
      int arrowThickness = (int)zoomFac/4;
      int yOffset        = 0;

      // Pfeilkoordinaten abspeichern:
      int xPointsNorth[] = {startPoint.x-arrowLineThickness-1,
                            startPoint.x-arrowLineThickness-1,
                            startPoint.x-arrowLineThickness-1-arrowThickness,
                            startPoint.x,
                            startPoint.x+arrowLineThickness+1+arrowThickness,
                            startPoint.x+arrowLineThickness+1,
                            startPoint.x+arrowLineThickness+1};
      int yPointsNorth[] = {startPoint.y-yOffset,
                            startPoint.y-zoomFac+arrowThickness,
                            startPoint.y-zoomFac+arrowThickness,
                            startPoint.y-zoomFac,
                            startPoint.y-zoomFac+arrowThickness,
                            startPoint.y-zoomFac+arrowThickness,
                            startPoint.y-yOffset};

      int xPointsSouth[] = {startPoint.x-arrowLineThickness-1,
                            startPoint.x-arrowLineThickness-1,
                            startPoint.x-arrowLineThickness-1-arrowThickness,
                            startPoint.x,
                            startPoint.x+arrowLineThickness+1+arrowThickness,
                            startPoint.x+arrowLineThickness+1,
                            startPoint.x+arrowLineThickness+1};
      int yPointsSouth[] = {startPoint.y+yOffset,
                            startPoint.y+zoomFac-arrowThickness,
                            startPoint.y+zoomFac-arrowThickness,
                            startPoint.y+zoomFac,
                            startPoint.y+zoomFac-arrowThickness,
                            startPoint.y+zoomFac-arrowThickness,
                            startPoint.y+yOffset};

      int yPointsEast[] = { startPoint.y-arrowLineThickness-1,
                            startPoint.y-arrowLineThickness-1,
                            startPoint.y-arrowLineThickness-1-arrowThickness,
                            startPoint.y,
                            startPoint.y+arrowLineThickness+1+arrowThickness,
                            startPoint.y+arrowLineThickness+1,
                            startPoint.y+arrowLineThickness+1};
      int xPointsEast[] = { startPoint.x+yOffset,
                            startPoint.x+zoomFac-arrowThickness,
                            startPoint.x+zoomFac-arrowThickness,
                            startPoint.x+zoomFac,
                            startPoint.x+zoomFac-arrowThickness,
                            startPoint.x+zoomFac-arrowThickness,
                            startPoint.x+yOffset};

      int yPointsWest[] = {startPoint.y-arrowLineThickness-1,
                            startPoint.y-arrowLineThickness-1,
                            startPoint.y-arrowLineThickness-1-arrowThickness,
                            startPoint.y,
                            startPoint.y+arrowLineThickness+1+arrowThickness,
                            startPoint.y+arrowLineThickness+1,
                            startPoint.y+arrowLineThickness+1};
      int xPointsWest[] = {startPoint.x-yOffset,
                            startPoint.x-zoomFac+arrowThickness,
                            startPoint.x-zoomFac+arrowThickness,
                            startPoint.x-zoomFac,
                            startPoint.x-zoomFac+arrowThickness,
                            startPoint.x-zoomFac+arrowThickness,
                            startPoint.x-yOffset};

      Polygon northArrow = new Polygon(xPointsNorth, yPointsNorth, xPointsNorth.length);
      Polygon eastArrow  = new Polygon(xPointsEast, yPointsEast, xPointsEast.length);
      Polygon southArrow = new Polygon(xPointsNorth, yPointsSouth, xPointsSouth.length);
      Polygon westArrow  = new Polygon(xPointsWest, yPointsWest, xPointsWest.length);

      // Pfeil in die Richtung der Maustaste zeichnen:
      if (north) {
        g.fillPolygon(northArrow);
      }
      else if (east) {
        g.fillPolygon(eastArrow);
      }
      else if (south) {
        g.fillPolygon(southArrow);
      }
      else if (west) {
        g.fillPolygon(westArrow);
      }
    }
  }

  /**
   * Diese Methode zeichnet alle in der aktuell editierten Levelebene befindlichen
   * Monsterstartpositionen. Diese Methode wird automatisch bei Bedarf durch die
   * Methode {@link #paint paint(Graphics g)} aufgerufen.
   *
   * @param g Graphics-Instanz auf Zeichenbereich.
   * @see #paint
   */
  private final void drawMonsterPositions(Graphics g) {
    // Vector mit den Monsterstartpositionen vom editierten Labyrinth holen.
    if (vecMonsterStartpositions == null) {
      vecMonsterStartpositions = lab.getAllMonsterStartPositions();
    }
    // Falls Monsterstartpositionen gesetzt wurden, so zeichnen wir diese evtl.
    if (vecMonsterStartpositions != null) {
      Position position = null;
      Point startPoint = null;

      for (int i = 0; i < vecMonsterStartpositions.size(); i++) {
        position = (pacman3d.labyrinth.Position)vecMonsterStartpositions.get(i);
        g.setColor(this.colorMonsterStartPosition);

        // Falls Monsterstartpositionen selektiert wurden, so highlighten wir
        // diese
        for (int k = 0; k < this.selectedCellsVector.size(); k++) {
          Point3i point = (Point3i)((Cell)this.selectedCellsVector.get(k)).getPosition();
          if (point.equals(position.getPosition())) {
            g.setColor(this.colorMonsterStartPosition.brighter());
            break;
          }
        }

        // Zeichnen der Monsterstartpositionen
        startPoint = new Point(position.getPosition().x*zoomFac+(int)zoomFac/2, position.getPosition().z*zoomFac+(int)zoomFac/2);
        if (position.getPosition().y == this.editLevelNumber) {
          for (int j = 0; j < (int)zoomFac/4; j++) {
            g.drawLine(position.getPosition().x*zoomFac+1+j, position.getPosition().z*zoomFac+zoomFac-1, position.getPosition().x*zoomFac+zoomFac-1, position.getPosition().z*zoomFac+1+j);
          }
        }
      }
    }
  }

  /**
   * Diese Methode zeichnet alle in der aktuell editierten Levelebene befindlichen
   * Pacmanstartpositionen. Diese Methode wird automatisch bei Bedarf durch die
   * Methode {@link #paint paint(Graphics g)} aufgerufen.
   *
   * @param g Graphics-Instanz auf Zeichenbereich.
   * @see #paint
   */
  private final void drawPacmanPositions(Graphics g) {
    // Vom aktuell editierten Labyrinth all Pacmanstartpositionen holen
    if (vecPacmanStartpositions == null) {
      vecPacmanStartpositions = lab.getAllPacmanStartPositions();
    }
    // Sind Pacmanstartpositionen gesetzt, so zeichnen wir evtl.:
    if (vecPacmanStartpositions != null) {
      Position position = null;
      Vector3d lineOfSight = null;
      Point startPoint = null;

      int arrowLineThickness = (int)zoomFac/12;
      int arrowThickness = (int)zoomFac/4;
      int yOffset        = 0;

      for (int i = 0; i < vecPacmanStartpositions.size(); i++) {
        position = (pacman3d.labyrinth.Position)vecPacmanStartpositions.get(i);
        g.setColor(this.colorPacmanStartPosition);

        // Ist eine Pacmanstartposition selektiert, so highlighten wir diese:
        for (int k = 0; k < this.selectedCellsVector.size(); k++) {
          Point3i point = (Point3i)((Cell)this.selectedCellsVector.get(k)).getPosition();
          if (point.equals(position.getPosition())) {
            g.setColor(this.colorPacmanStartPosition.brighter());
            break;
          }
        }

        // Pfeile fr "LineOfSight" einer Pacmanstartposition berechnen:
        lineOfSight = position.getLineOfSight();
        startPoint = new Point(position.getPosition().x*zoomFac+(int)zoomFac/2, position.getPosition().z*zoomFac+(int)zoomFac/2);

        int xPointsNorth[] = {startPoint.x-arrowLineThickness-1,
                              startPoint.x-arrowLineThickness-1,
                              startPoint.x-arrowLineThickness-1-arrowThickness,
                              startPoint.x,
                              startPoint.x+arrowLineThickness+1+arrowThickness,
                              startPoint.x+arrowLineThickness+1,
                              startPoint.x+arrowLineThickness+1};
        int yPointsNorth[] = {startPoint.y-yOffset,
                              startPoint.y-zoomFac+arrowThickness,
                              startPoint.y-zoomFac+arrowThickness,
                              startPoint.y-zoomFac,
                              startPoint.y-zoomFac+arrowThickness,
                              startPoint.y-zoomFac+arrowThickness,
                              startPoint.y-yOffset};

        int xPointsSouth[] = {startPoint.x-arrowLineThickness-1,
                              startPoint.x-arrowLineThickness-1,
                              startPoint.x-arrowLineThickness-1-arrowThickness,
                              startPoint.x,
                              startPoint.x+arrowLineThickness+1+arrowThickness,
                              startPoint.x+arrowLineThickness+1,
                              startPoint.x+arrowLineThickness+1};
        int yPointsSouth[] = {startPoint.y+yOffset,
                              startPoint.y+zoomFac-arrowThickness,
                              startPoint.y+zoomFac-arrowThickness,
                              startPoint.y+zoomFac,
                              startPoint.y+zoomFac-arrowThickness,
                              startPoint.y+zoomFac-arrowThickness,
                              startPoint.y+yOffset};

        int yPointsEast[] = { startPoint.y-arrowLineThickness-1,
                              startPoint.y-arrowLineThickness-1,
                              startPoint.y-arrowLineThickness-1-arrowThickness,
                              startPoint.y,
                              startPoint.y+arrowLineThickness+1+arrowThickness,
                              startPoint.y+arrowLineThickness+1,
                              startPoint.y+arrowLineThickness+1};
        int xPointsEast[] = { startPoint.x+yOffset,
                              startPoint.x+zoomFac-arrowThickness,
                              startPoint.x+zoomFac-arrowThickness,
                              startPoint.x+zoomFac,
                              startPoint.x+zoomFac-arrowThickness,
                              startPoint.x+zoomFac-arrowThickness,
                              startPoint.x+yOffset};

        int yPointsWest[] = {startPoint.y-arrowLineThickness-1,
                              startPoint.y-arrowLineThickness-1,
                              startPoint.y-arrowLineThickness-1-arrowThickness,
                              startPoint.y,
                              startPoint.y+arrowLineThickness+1+arrowThickness,
                              startPoint.y+arrowLineThickness+1,
                              startPoint.y+arrowLineThickness+1};
        int xPointsWest[] = {startPoint.x-yOffset,
                              startPoint.x-zoomFac+arrowThickness,
                              startPoint.x-zoomFac+arrowThickness,
                              startPoint.x-zoomFac,
                              startPoint.x-zoomFac+arrowThickness,
                              startPoint.x-zoomFac+arrowThickness,
                              startPoint.x-yOffset};

        Polygon northArrow = new Polygon(xPointsNorth, yPointsNorth, xPointsNorth.length);
        Polygon eastArrow  = new Polygon(xPointsEast, yPointsEast, xPointsEast.length);
        Polygon southArrow = new Polygon(xPointsNorth, yPointsSouth, xPointsSouth.length);
        Polygon westArrow  = new Polygon(xPointsWest, yPointsWest, xPointsWest.length);

        if (position.getPosition().y == this.editLevelNumber) {
          if (((int)lineOfSight.x) == 0 && ((int)lineOfSight.z) == -1) {
            g.fillPolygon(northArrow);
          }
          else if (((int)lineOfSight.x) == 1 && ((int)lineOfSight.z) == 0) {
            g.fillPolygon(eastArrow);
          }
          else if (((int)lineOfSight.x) == 0 && ((int)lineOfSight.z) == 1) {
            g.fillPolygon(southArrow);
          }
          else if (((int)lineOfSight.x) == -1 && ((int)lineOfSight.z) == 0) {
            g.fillPolygon(westArrow);
          }
        }
      }
    }
  }

  /**
   * Hilfsmethode, die die vom Benutzer selektierten Zellen auf deren Typ und
   * Inhalt berprft, d.h. ob sie z.B. noch Items, Pacmanstartpositionen oder
   * Monster enthalten. Diese berprfung wird vorgenommen, um kontextsensitive
   * Popupmens auf dem 2D-Leveleditierungsbereich anzuzeigen.
   *
   * @return Array vom Typ <b>boolean</b>, mit den Belegungen: <br>
   *         0te Arrayposition: <b>true</b>, falls mindestens eine selektierte Zelle
   *              keine Instanz von {@link pacman3d.labyrinth.cells.CellFloor
   *              CellFloor} ist, anderenfalls <b>false</b>.
   *         1te Arrayposition: <b>true</b>, falls mindestens eine begehbare Zelle
   *              selektiert wurde, andernfalls <b>false</b>.
   *         2te Arrayposition: <b>true</b>, falls nur eine einzige Zelle selektiert
   *              wurde, anderenfalls <b>false</b>.
   *         3te Arrayposition: <b>true</b>, falls mindestens ein Item selektiert wurde,
   *              andernfalls <b>false</b>.
   *         4te Arrayposition: <b>true</b>, falls mindestens ein Monster
   *              selektiert wurde, andernfalls <b>false</b>.
   *         5te Arrayposition: <b>true</b>, fallse mindestens eine Monsterstartposition
   *              selektiert wurde, andernfalls <b>false</b>.
   *         6te Arrayposition: <b>true</b>, falls mindestens eine Pacmanstartposition
   *              selektiert wurde, andernfalls <b>false</b>.
   *         7te Arrayposition: <b>true</b>, falls nur eine einzige Zelle selektiert wurde,
   *              andernfalls <b>false</b>.
   */
  private boolean[] checkCellTypesAndTheirContents() {
    boolean[] flags = {false, false, false, false, false, false, false, false};

    for (int i = 0; i < this.selectedCellsVector.size(); i++) {
      Cell cell = (Cell)this.selectedCellsVector.get(i);
      // Mindestens eine Zelle ist nicht vom Typ "CellFloor"
      if (!(cell instanceof CellFloor) && flags[0] == false)
        flags[0] = true;
      // Mindestens eine Zelle ist "walkable"
      if (cell.isWalkable())
        flags[1] = true;
      // Nur eine Zelle ist selektiert (aus alten Zeiten leider noch)
      if (this.selectedCellsVector.size() == 1)
        flags[2] = true;

      if (cell.isWalkable()) {
        ID[] ids = lab.getAllMonsterIds();
        // Mindestens ein Monster befindet sich unter der Selektion
        for (int j = 0; j < ids.length; j++) {
          Point3i point3i = lab.getMonsterPos(ids[j]);
          if (point3i.equals((Point3i)cell.getPosition())) {
            flags[3] = true;
            break;
          }
        }
        // Mindestens eine Monsterstartposition befindet sich unter der Selektion
        if (lab.getAllMonsterStartPositions() != null)
          for (int j = 0; j < lab.getAllMonsterStartPositions().size(); j++) {
            Point3i point3i = ((Position)lab.getAllMonsterStartPositions().get(j)).getPosition();
            if (point3i.equals((Point3i)cell.getPosition())) {
              flags[5] = true;
              break;
            }
          }
        // Mindestens eine Pacmanstartposition befindet sich unter der Selektion
        if (lab.getAllPacmanStartPositions() != null)
          for (int j = 0; j < lab.getAllPacmanStartPositions().size(); j++) {
            Point3i point3i = ((Position)lab.getAllPacmanStartPositions().get(j)).getPosition();
            if (point3i.equals((Point3i)cell.getPosition())) {
              flags[6] = true;
              break;
            }
          }
        }

      // Mindestens ein Item befindet sich unter der Selektion
      if (cell.isWalkable()) {
        Item[] items = cell.getItems();
        if (items.length >= 1)
          flags[4] = true;
      }
    }
    // Genau eine Zelle wurde selektiert
    if (this.selectedCellsVector.size() == 1)
      flags[7] = true;

    return flags;
  }

  /**
   * Hilfsmethode, die testet, ob es sich bei einem {@link java.awt.event.MouseEvent
   * MouseEvent} um ein <b>PopupTrigger</b> handelt, so da evtl. auf dem
   * 2D-Darstellungsbereich ein Popupmen angezeigt werden soll. Ist dies der Fall,
   * so kmmert sich diese Methode weiterhin um den korrekten Aufbau des anzuzeigenden
   * Popupmens.
   *
   * @param mouseEvent java.awt.event.MouseEvent
   */
  private void maybeShowPopup(MouseEvent mouseEvent) {
    if (mouseEvent.isPopupTrigger()) {
      popup = new JPopupMenu();
      boolean[] flags = this.checkCellTypesAndTheirContents();

      // Popupmen aufbauen:
      JMenuItem menuItem = new JMenuItem("Gewhlten Zellentyp setzen");
      menuItem.setActionCommand("set cell");
      menuItem.addActionListener(this);
      popup.add(menuItem);

      if (flags[0]) {
        menuItem = new JMenuItem("Zelle/n texturieren");
        menuItem.setActionCommand("texture cell");
        menuItem.addActionListener(this);
        popup.add(menuItem);
        menuItem = new JMenuItem("Zelle/n colorieren");
        menuItem.setActionCommand("color cell");
        menuItem.addActionListener(this);
        popup.add(menuItem);
      }
      if (flags[7]) {
        menuItem = new JMenuItem("Alle Zellen des ausgewhlten Typs selektieren");
        menuItem.setActionCommand("experimental");
        menuItem.addActionListener(this);
        popup.add(menuItem);
      }
      if (flags[2]) {
        menuItem = new JMenuItem("Zelleneigenschaften");
        menuItem.setActionCommand("cell property");
        menuItem.addActionListener(this);
        popup.add(menuItem);

      }
      if (this.selectedCellsVector != null && this.selectedCellsVector.size() == 1 && ((Cell)this.selectedCellsVector.get(0)).isWalkable() /*[8]*/) {
        popup.addSeparator();
        menuItem = new JMenuItem("Pacmanstartposition setzen");
        menuItem.setActionCommand("set pacman startposition");
        menuItem.addActionListener(this);
        popup.add(menuItem);
      }
      if (flags[6]) {
        menuItem = new JMenuItem("Pacmanstartposition lschen");
        menuItem.setActionCommand("delete pacman startposition");
        menuItem.addActionListener(this);
        popup.add(menuItem);
      }
      if (flags[1]) {
        popup.addSeparator();
        menuItem = new JMenuItem("Monster setzen");
        menuItem.setActionCommand("set monster");
        menuItem.addActionListener(this);
        popup.add(menuItem);
      }
      if (flags[3]) {
        menuItem = new JMenuItem("Monster lschen");
        menuItem.setActionCommand("delete monster");
        menuItem.addActionListener(this);
        popup.add(menuItem);
      }
      if (flags[1]) {
        menuItem = new JMenuItem("Monsterstartposition setzen");
        menuItem.setActionCommand("set monster startposition");
        menuItem.addActionListener(this);
        popup.add(menuItem);
      }
      if (flags[5]) {
        menuItem = new JMenuItem("Monsterstartposition lschen");
        menuItem.setActionCommand("delete monster startposition");
        menuItem.addActionListener(this);
        popup.add(menuItem);
        popup.addSeparator();
      }
      if (flags[1]) {
        popup.addSeparator();
        menuItem = new JMenuItem("Gewhltes Item setzen");
        menuItem.setActionCommand("set item");
        menuItem.addActionListener(this);
        popup.add(menuItem);
      }
      if (flags[4]) {
        menuItem = new JMenuItem("Item/s lschen");
        menuItem.setActionCommand("delete item");
        menuItem.addActionListener(this);
        popup.add(menuItem);
      }

      // Popupmen anzeigen:
      this.popup.show(mouseEvent.getComponent(), mouseEvent.getX(), mouseEvent.getY());
    }
  }

  /**
   * Methode, um einer Instanz dieser Klasse mitzuteilen, mit Hilfe welcher Klasse
   * Texturen und/oder Farben selektiert werden knnen.
   *
   * @param texSelecterPanel  Instanz der Klasse {@link
   *        pacman3d.labyrinth.leveleditor.TextureSelectionJPanel
   *        TextureSelectionJPanel}, mit deren Hilfe die Textur- bzw. Farbauswahl
   *        erfolgt.
   * @see   pacman3d.labyrinth.leveleditor.TextureSelectionJPanel
   */
  protected void setTextureChooser(TextureSelectionJPanel texSelecterPanel) {
    this.texSelecterPanel = texSelecterPanel;
  }

  /**
   * Liefert aktuel gewhlten Zoomfaktor.
   *
   * @return  Zoomfaktor
   */
  protected int getZoomFac() {
    return zoomFac;
  }

  /**
   * Methode, um die Gre dieser Komponente einzustellen.
   *
   * @param dimension Gewnschte Gre dieser Komponente.
   */
  public void setSize(Dimension dimension) {
    super.setSize(new Dimension(zoomFac*lab.getConstraints().x+1, zoomFac*lab.getConstraints().z+1));
  }

  /**
   *  Methode, um das Gitternetz bei der 2D-Leveleditierung ein- bzw. ausschalten.
   *
   * @param <b>true</b> Grid ein,<br>
   *        <b>false</b> andernfalls.
   */
  public void setGrid(boolean gridOn) {
    this.gridOn = gridOn;
    repaint();
  }

  /**
   * Durch Aufruf dieser Methode werden alle Komponenten mit dem Faktor
   * <b>zoomFac+2</b> multipliziert und entsprechend danach grer
   * auf dem Bildschirm dargestellt.
   *
   * @see LevelEditingJPanel#zoomOut
   * @see #zoomFac
   */
  public void zoomIn() {
    if (zoomFac <= 200) {
      zoomFac = zoomFac + 2;

      // Neue Gre der Komponenten berechnen
      setPreferredSize(new Dimension(zoomFac*lab.getConstraints().x+1, zoomFac*lab.getConstraints().z+1));
      setMaximumSize(new Dimension(zoomFac*lab.getConstraints().x+1, zoomFac*lab.getConstraints().z+1));
      setSize(new Dimension(0,0));

      //  alles neu zeichnen
      repaint();
    }
  }

  /**
   * Durch Aufruf dieser Methode werden alle Komponenten mit dem Faktor
   * <b>zoomFac-2</b> multipliziert und entsprechend danach kleiner auf
   * dem Bildschirm dargestellt.
   *
   * @see LevelEditingJPanel#zoomIn
   * @see #zoomFac
   */
  public void zoomOut() {
    if (zoomFac >= 10) {
      zoomFac = zoomFac - 2;
      setPreferredSize(new Dimension(zoomFac*lab.getConstraints().x+1, zoomFac*lab.getConstraints().z+1));
      setMinimumSize(new Dimension(zoomFac*lab.getConstraints().x+1, zoomFac*lab.getConstraints().z+1));
      setSize(new Dimension(0,0));
      repaint();
    }
  }

  /**
   * Die durch das von dieser Klasse implementierte Interface
   * {@link java.awt.event.MouseListener} geforderte Methode, um auf
   * {@link java.awt.event.MouseEvent MouseEvents} reagieren und somit
   * diverse Interaktion mit dem Benutzer bei der 2D-Leveleditierung realisieren
   * zu knnen.
   *
   * @param mouseEvent {@link java.awt.event.MouseEvent}
   *
   * @see java.awt.event.MouseListener
   */
  public void mouseClicked(MouseEvent mouseEvent) {

    // Setzen einer Pacmanstartposition
    if ((north || east || south || west) && this.selectedCellsVector.size() == 1) {
      if (north) {
        Point3i point3i = (Point3i)((Cell)this.selectedCellsVector.get(0)).getPosition();
        try {
          if (setPacmanStartposition)
            lab.addPacmanStartPosition(new pacman3d.labyrinth.Position(point3i, new javax.vecmath.Vector3d(0, 0, -1)));
        }
        catch (Exception P3DException) {}
      }
      else if (east) {
        Point3i point3i = (Point3i)((Cell)this.selectedCellsVector.get(0)).getPosition();
        try {
          if (setPacmanStartposition)
            lab.addPacmanStartPosition(new pacman3d.labyrinth.Position(point3i, new javax.vecmath.Vector3d(1, 0, 0)));
         }
        catch (Exception P3DException) {}
      }
      else if (south) {
        Point3i point3i = (Point3i)((Cell)this.selectedCellsVector.get(0)).getPosition();
        try {
          if (setPacmanStartposition)
            lab.addPacmanStartPosition(new pacman3d.labyrinth.Position(point3i, new javax.vecmath.Vector3d(0, 0, 1)));
         }
        catch (Exception P3DException) {}
      }
      else if (west) {
        Point3i point3i = (Point3i)((Cell)this.selectedCellsVector.get(0)).getPosition();
        try {
          if (setPacmanStartposition)
            lab.addPacmanStartPosition(new pacman3d.labyrinth.Position(point3i, new javax.vecmath.Vector3d(-1, 0, 0)));
        }
        catch (Exception P3DException) {}
      }

      // Ende Pacmanstartposition setzen
      this.setPacmanStartposition = false;
      this.arrowEndPoint = null;
      // Ende Pacmanstartposition setzen
    }

    // Selektierte Cellen deselektieren, falls kein "STRG" gedrckt ist.
    if (!this.selectedCellsVector.isEmpty() && !mouseEvent.isControlDown()) {
      this.selectedCellsVector.removeAllElements();
      return;
    }
    repaint();



    Point point = mouseEvent.getPoint();

    // MouseClick-Punkt in Cell-Koordinaten umrechnen
    int x = point.x/zoomFac;
    int z = point.y/zoomFac;

    // Checken, ob MouseClick innerhalb der Labyrinthkoordinaten liegt
    if (x < lab.getConstraints().x && z < lab.getConstraints().z) {
      // Ist die STRG-Taste gedrckt, so knnen mehrere Zellen
      // selektiert werden -> TEST
      if (mouseEvent.isControlDown()) {
        // Ist die Zelle bereits selektiert, so wird sie deselektiert.
        if (selectedCellsVector.contains(lab.getCell(x, editLevelNumber, z))) {
          int pos = selectedCellsVector.indexOf(lab.getCell(x, editLevelNumber, z));
          selectedCellsVector.remove(pos);
        }
        // Andernfalls selektiert.
        else {
          selectedCellsVector.add((lab.getCell(x,editLevelNumber, z)));
        }
      }
      // STRG-Tast nicht gedrckt -> alle selektierten Zellen lschen...,
      // da in diesem Modus nur eine Zelle selektiert werden kann...
      else {
        if (!selectedCellsVector.isEmpty()) {
          selectedCellsVector.removeAllElements();
        }

        //
        // Setzen einer Zelle durch einfaches Klicken auf dem 2D-Leveleditierungsbereich:
        // Dabei wird bei jedem Klick zwischen CellFloor und selektiertem Zellentyp
        // gewechselt:
        if (lab.getCell(x, this.editLevelNumber, z) instanceof CellFloor) {
          Cell cell = leveleditor.cellsJPanel.produceSelectedCellType(new Point3i(x, this.editLevelNumber, z));

          String sCellName = CellHandler.getInstance().addCell( cell , true);
          lab.setCell(sCellName, new Point3i(x, this.editLevelNumber, z));
        }
        // andernfalls eine CellFloor erzeugen und an die alte Stelle im Labyrinth
        // setzen
        else {
          String sCellName = CellHandler.getInstance().addCell(new CellFloor(new Point3i(x, editLevelNumber, z)), true);
          lab.setCell( sCellName, new Point3i(x, editLevelNumber, z) );
        }
      }
    }
    repaint();
  }

  /**
   * Die durch das von dieser Klasse implementierte Interface
   * {@link java.awt.event.MouseListener} geforderte Methode, um auf
   * {@link java.awt.event.MouseEvent MouseEvents} reagieren und somit
   * diverse Interaktion mit dem Benutzer bei der 2D-Leveleditierung realisieren
   * zu knnen.<br>
   * Diese Methode berprft speziell, ob es sich bei einem Maus-Klick auf dem
   * 2D-Leveleditierungsbereich um ein <b>Popup-Trigger</b> handelt, so da evtl.
   * ein Popupmen angezeigt werden soll.
   *
   * @param mouseEvent {@link java.awt.event.MouseEvent}
   *
   * @see java.awt.event.MouseListener
   */
  public void mousePressed(MouseEvent mouseEvent) {
    // Punkt merken, an dem die Mouse gedrckt wurde... Knnte sein, da
    // ab hier eine Zellen-Selektion stattfinden soll...
    this.pointStart = mouseEvent.getPoint();

    // Event auf PopupTrigger hin testen...
    maybeShowPopup(mouseEvent);
  }

  /**
   * Die durch das von dieser Klasse implementierte Interface
   * {@link java.awt.event.MouseListener} geforderte Methode, um auf
   * {@link java.awt.event.MouseEvent MouseEvents} reagieren und somit
   * diverse Interaktion mit dem Benutzer bei der 2D-Leveleditierung realisieren
   * zu knnen.<br>
   * Diese Methode berprft speziell, ob es sich bei einem Maus-Klick auf dem
   * 2D-Leveleditierungsbereich um ein <b>Popup-Trigger</b> handelt, so da evtl.
   * ein Popupmen angezeigt werden soll.
   *
   * @param mouseEvent {@link java.awt.event.MouseEvent}
   *
   * @see java.awt.event.MouseListener
   */
  public void mouseReleased(MouseEvent mouseEvent) {
    // Merken, da die Mousetasten losgelassen wurden
    // -> kann jetzt ersteinmal kein MouseDragged auf jeden Fall zu Ende...
    isMouseDragged = false;

    // Event auf PopupTrigger hin testen...
    maybeShowPopup(mouseEvent);
    repaint();
  }

  /**
   * Die durch das von dieser Klasse implementierte Interface
   * {@link java.awt.event.MouseListener} geforderte Methode, um auf
   * {@link java.awt.event.MouseEvent MouseEvents} reagieren und somit
   * diverse Interaktion mit dem Benutzer bei der 2D-Leveleditierung realisieren
   * zu knnen.
   *
   * @param mouseEvent {@link java.awt.event.MouseEvent}
   *
   * @see java.awt.event.MouseListener
   */
  public void mouseEntered(MouseEvent e) {
  }

  /**
   * Die durch das von dieser Klasse implementierte Interface
   * {@link java.awt.event.MouseListener} geforderte Methode, um auf
   * {@link java.awt.event.MouseEvent MouseEvents} reagieren und somit
   * diverse Interaktion mit dem Benutzer bei der 2D-Leveleditierung realisieren
   * zu knnen.<br>
   * Innerhalb dieser Methode erfolgt die Selektierung von Zellen durch den
   * Benutzer mittels <b>Maus-Drag</b>.
   *
   * @param mouseEvent {@link java.awt.event.MouseEvent}
   *
   * @see java.awt.event.MouseListener
   */
  public void mouseDragged(MouseEvent mouseEvent) {
    // Falls eine Pacmanposition gesetzt wird -> return
    if (this.setPacmanStartposition)
      return;


    // Merken, da MouseDrag stattfindet
    isMouseDragged = true;

    // Aktuellen Punkt immer merken...
    pointEnd = mouseEvent.getPoint();

    int x = (pointStart.x > pointEnd.x)? pointEnd.x : pointStart.x;
    int y = (pointStart.y > pointEnd.y)? pointEnd.y : pointStart.y;
    int width   = Math.abs(pointStart.x-pointEnd.x);
    int height  = Math.abs(pointStart.y-pointEnd.y);

    // Aktuell selektierte Levelebene durchlaufen und testen, welche Zellen
    // innerhalb des Rechteckes von (x,y) bis (x+width,y+height) liegen und
    // diese in "selectedCellsVector" ablegen bzw. evtl. nicht mehr selektierte
    // Zellen aus diesem Vector entfernen.
    for (int xx = 0; xx < lab.getConstraints().x; xx++)
      for (int zz = 0; zz < lab.getConstraints().z; zz++) {
        if (containsCell(xx, zz, x, y, width, height)) {
          if (!selectedCellsVector.contains(lab.getCell(xx, this.editLevelNumber, zz))) {
            selectedCellsVector.add(lab.getCell(xx, this.editLevelNumber, zz));
          }
        }
        else if (selectedCellsVector.contains(lab.getCell(xx, this.editLevelNumber, zz))) {
            int pos = selectedCellsVector.indexOf(lab.getCell(xx, this.editLevelNumber, zz));
            selectedCellsVector.remove(pos);
          }
        }
    repaint();
  }


  /**
   * Die durch das von dieser Klasse implementierte Interface
   * {@link java.awt.event.MouseMotionListener} geforderte Methode, um auf
   * {@link java.awt.event.MouseEvent MouseEvents} reagieren und somit
   * diverse Interaktion mit dem Benutzer bei der 2D-Leveleditierung realisieren
   * zu knnen.<br>
   * Innerhalb dieser Methode erfolgt ein Teil der Realisierung des interaktiven
   * Setzens einer Pacmanstartposition samt Blickrichtung.
   *
   * @param mouseEvent {@link java.awt.event.MouseEvent}
   *
   * @see java.awt.event.MouseMotionListener
   */
  public void mouseMoved(MouseEvent e) {
    if ((this.setPacmanStartposition /*|| this.setMonsterStartposition*/) && this.selectedCellsVector.size() == 1) {
      Point3i point3i = (Point3i)((Cell)this.selectedCellsVector.get(0)).getPosition();
      Point   point   = new Point(point3i.x*zoomFac + (int)zoomFac/2, point3i.z*zoomFac + (int)zoomFac/2);
      Point mousePoint = e.getPoint();

      // Blickrichtung WESTEN
      if (mousePoint.x <= point3i.x*zoomFac) {
        if (mousePoint.y >= point3i.z*zoomFac && mousePoint.y <= point3i.z*zoomFac + zoomFac) {
          arrowEndPoint = new Point(point.x-zoomFac, point.y);
          west = true;
          north = south = east = false;
        }
      }
      if (mousePoint.x >= point3i.x*zoomFac && mousePoint.x <= point3i.x*zoomFac + zoomFac) {
        // Blickrichtung NORDEN
        if (mousePoint.y <= point3i.z*zoomFac) {
          arrowEndPoint = new Point(point.x, point.y-zoomFac);
          north = true;
          south = east = west = false;
        }
        // Blickrichtung SDEN
        else if (mousePoint.y >= point3i.z*zoomFac + zoomFac /*&& mousePoint.y <= point3i.z*zoomFac + 2*zoomFac*/) {
          arrowEndPoint = new Point(point.x, point.y+zoomFac);
          south = true;
          north = east = west = false;
        }
      }
      // Blickrichtung OSTEN
      if (mousePoint.x >= point3i.x*zoomFac + zoomFac /*&& mousePoint.x <= point3i.x*zoomFac + 2*zoomFac*/) {
        if (mousePoint.y >= point3i.z*zoomFac && mousePoint.y <= point3i.z*zoomFac + zoomFac) {
          arrowEndPoint = new Point(point.x+zoomFac, point.y);
          east = true;
          north = west = south = false;
        }
      }
    repaint();
  }

  this.requestFocus();
  }

  /**
   * Die durch das von dieser Klasse implementierte Interface
   * {@link java.awt.event.MouseMotionListener} geforderte Methode, um auf
   * {@link java.awt.event.MouseEvent MouseEvents} reagieren und somit
   * diverse Interaktion mit dem Benutzer bei der 2D-Leveleditierung realisieren
   * zu knnen.<br>
   *
   * @param mouseEvent {@link java.awt.event.MouseEvent}
   *
   * @see java.awt.event.MouseMotionListener
   */
  public void mouseExited(MouseEvent e) {
  }

  /**
   * Methode, um die prferierte Gre dieser Komponente einzustellen.
   *
   * @param dimension Gewnschte Gre dieser Komponente.
   */
  public void setPreferredSize(Dimension dimension) {
    super.setPreferredSize(dimension);
  }

  /**
   * Methode, um die minimale Gre dieser Komponente einzustellen.
   *
   * @param dimension Gewnschte Gre dieser Komponente.
   */
  public void setMinimumSize(Dimension dimension) {
    super.setMinimumSize(dimension);
  }

  /**
   * Methode, um die maximale Gre dieser Komponente einzustellen.
   *
   * @param dimension Gewnschte Gre dieser Komponente.
   */
  public void setMaximumSize(Dimension dimension) {
    super.setMaximumSize(dimension);
  }

  /**
   * Methode, die die prferierte Gre dieser Komponente zurckliefert.
   *
   * @return Prferierte Gre dieser Komponente.
   */
  public Dimension getPreferredSize() {
    return new Dimension(zoomFac*lab.getConstraints().x+1, zoomFac*lab.getConstraints().z+1);
  }

  /**
   * Methode, die die minimale Gre dieser Komponente zurckliefert.
   *
   * @return Minimale Gre dieser Komponente.
   */
  public Dimension getMinimumSize() {
    return new Dimension(zoomFac*lab.getConstraints().x+1, zoomFac*lab.getConstraints().z+1);
  }

  /**
   * Methode, die die maximale Gre dieser Komponente zurckliefert.
   *
   * @return Maximale Gre dieser Komponente.
   */
  public Dimension getMaximumSize() {
    return new Dimension(zoomFac*lab.getConstraints().x+1, zoomFac*lab.getConstraints().z+1);
  }

  /**
   * Methode, um die zu editierende Levelebene auszuwhlen und diese dann auf dem
   * Bildschirm darzustellen.
   *
   * @param level Levelebene.
   */
  public void setLevelToEdit(int level) {
    if (0 <= level && level < lab.getConstraints().y) {
      editLevelNumber = level;
      this.selectedCellsVector.removeAllElements();
      repaint();
    }
  }

  /**
   * Die durch das von dieser Klasse implementierte Interface {@link java.awt.event.ActionListener}
   * geforderte Methode, um auf {@link java.awt.event.ActionEvent ActionEvents} reagieren
   * und somit diverse Interaktion mit dem Benutzer des 2D-Leveleditierungsbereich
   * zu realisieren.<br>
   * Alle {@link java.awt.event.ActionEvent ActionEvents}, die hier verarbeitet werden,
   * werden ausschlielich von dem auf dem 2D-Leveleditierungsbereich befindlichen
   * {@link #popup Popupmen} erzeugt.
   *
   * @param actionEvent {@link java.awt.event.ActionEvent}
   *
   * @see java.awt.event.ActionListener
   * @see #popup
   */
  public void actionPerformed(java.awt.event.ActionEvent e) {
    //
    // Begin des Setzens einer Pacmanstartposition
    //
    if (e.getActionCommand().equals("set pacman startposition")) {
      this.setPacmanStartposition = true;
    }

    //
    // Alles Zelltypen selektieren, die demjenigen Zelltypen entsprechen, der
    // gerade aktuell selektiert ist:
    //
    else if (e.getActionCommand().equals("experimental")) {
      Cell oCell_selectedCellType = (Cell)this.selectedCellsVector.get(0);
      String oStrg_selectedCellTypeClassName = oCell_selectedCellType.getClass().getName();
      this.selectedCellsVector.removeAllElements();

      for (int x = 0; x < lab.getConstraints().x; x++) {
        for (int z = 0; z < lab.getConstraints().z; z++) {
        if (lab.getCell(x, this.editLevelNumber, z).getClass().getName().equals(oStrg_selectedCellTypeClassName))
          this.selectedCellsVector.add(lab.getCell(x, this.editLevelNumber, z));
        }
      }
      repaint();
    }

    //
    // Selektierte Zelltypen texturieren
    //
    else if (e.getActionCommand().equals("texture cell")) {
      java.net.URL oURL = texSelecterPanel.image_url;
      String sMimeType = texSelecterPanel.mime_type;
      int iFileSize = (int)texSelecterPanel.size;
      pacman3d.util.ResourceHandler oTextureHandler = pacman3d.util.ResourceHandler.getInstance();
      String sTextureName = oTextureHandler.addTexture(texSelecterPanel.getTexture(), oURL, sMimeType, iFileSize );

      for (int i = 0; i < selectedCellsVector.size(); i++) {
        try {
          ((Cell)selectedCellsVector.get(i)).setTexture(leveleditor.cellsJPanel.getSideMask(), sTextureName, 0);
        }
        catch (Exception exception) {
          Debug.out(this.getClass().getName(), exception);
        }
      }
    }

    //
    // Selektierte Zelltypen kolorieren
    //
    else if (e.getActionCommand().equals("color cell")) {
      for (int i = 0; i < selectedCellsVector.size(); i++) {
        try {
          ((Cell)selectedCellsVector.get(i)).setColor(leveleditor.cellsJPanel.getSideMask(), texSelecterPanel.getColor3f());
        }
        catch (Exception exception) {
          Debug.out(this.getClass().getName(), exception);
        }
      }
    }

    //
    // Eigenschaften der aktuell durch den Benutzer selektierten Zelle abrufen
    // und anzeigen
    //
    else if (e.getActionCommand().equals("cell property")) {
      for (int i = 0; i < selectedCellsVector.size();i++) {
          try {
            leveleditor.cellsJPanel.showAttributesOfSelectedCell((Cell)selectedCellsVector.get(i));
            leveleditor.tabbedPane2.setSelectedIndex(0);
          }
          catch (Exception exception) {
            Debug.out(this.getClass().getName(), exception);
          }
      }
    }

    //
    // Alle selektierten Zellen durch den vom Benutzer selektierten Zellentyp
    // ersetzen:
    //
    else if (e.getActionCommand().equals("set cell")) {
      Cell cell = leveleditor.cellsJPanel.produceSelectedCellType(new Point3i(1,1,1));
      Class cellClass = cell.getClass();
      Cell selectedCell;
      String cellName;

      CellHandler cellHandler = CellHandler.getInstance();

      for (int i = 0; i < this.selectedCellsVector.size(); i++) {
        selectedCell = (Cell)this.selectedCellsVector.get(i);
        // Nur Zellen ersetzten, wenn sie von einem neuen Typ sind:
        if (selectedCell.getClass() == cellClass)
          continue;
        else {
          cellName = cellHandler.addCell(leveleditor.cellsJPanel.produceSelectedCellType((Point3i)selectedCell.getPosition()), true);
          lab.setCell(cellName, (Point3i)selectedCell.getPosition());
          cell = lab.getCell((Point3i)selectedCell.getPosition());

          this.selectedCellsVector.set(i, cell);

          // evtl. gesetzte Items des zu ersetzenden Zellentyps in den neuen
          // umkopieren.
          if (selectedCell.hasItems() && cell.isWalkable()) {
            Item[] items = selectedCell.getItems();
            for (int itemsCount = 0; itemsCount < items.length; itemsCount++) {
              cell.addItem(items[itemsCount]);
            }
          }

          // Evtl. Pacmanstartposition, Monsterstartposition, Monster lschen,
          // falls der neu gesetzte Zellentyp nicht mehr begehbar ist.
          if (!cell.isWalkable()) {
            this.removeMonster((Point3i)cell.getPosition());
            this.lab.removeMonsterStartPosition((Point3i)cell.getPosition());
            this.lab.removePacmanStartPosition((Point3i)cell.getPosition());
          }
        }
      }
    }

    //
    // Items in selektierte Zellen setzen
    //
    else if (e.getActionCommand().equals("set item")) {
      if (!this.selectedCellsVector.isEmpty()) {

        ItemHandler itemHandler = ItemHandler.getInstance();
        String itemName = itemHandler.addItem(leveleditor.itemJPanel.produceSelectedItem());

        for (int i = 0; i < selectedCellsVector.size();i++) {
          Cell cell;
          if ((cell = (Cell)selectedCellsVector.get(i)).isWalkable()) {
            try {
              cell.addItem(itemName);
            }
            catch (Exception exception) {
              Debug.out(this.getClass().getName(), exception);
            }
          }
        }
      }
      /*else {
        Cell cell = (Cell)lab.getCell(this.x, this.editLevelNumber, this.z);
        if (cell.isWalkable()) {
          try {
            cell.addItem(leveleditor.itemJPanel.produceSelectedItem());
          }
          catch(Exception exception) {
            Debug.out(this.getClass().getName(), exception, Debug.LEVEL_WARNING);
          }
        }
      }
      */
    }

    //
    // Items aus den selektierten Zellen lschen
    //
    else if (e.getActionCommand().equals("delete item")) {
      if (!this.selectedCellsVector.isEmpty()) {
        for (int i = 0; i < this.selectedCellsVector.size(); i++) {
          Cell cell;
          if ((cell = (Cell)this.selectedCellsVector.get(i)).isWalkable()) {
            cell.removeAllItems();
          }
        }
      }
    }


    //
    // Monster in selektierte Zellen setzen
    //
    else if (e.getActionCommand().equals("set monster") || e.getActionCommand().equals("set monster startposition")) {
      if (!this.selectedCellsVector.isEmpty()) {
        for (int i = 0; i < selectedCellsVector.size();i++) {
          Cell cell;
          if ((cell = (Cell)selectedCellsVector.get(i)).isWalkable()) {
            try {
              if (e.getActionCommand().equals("set monster")) {
                ID id = new ID();
                Monster monster = new Monster();
                lab.setMonsterPos(id, cell.getPosition().x, cell.getPosition().y, cell.getPosition().z);
                lab.setMonster(id, monster);
              }
              lab.addMonsterStartPosition(new Position((Point3i)cell.getPosition(), new Vector3d()));
            }
            catch (Exception exception) {
              Debug.out(this.getClass().getName(), exception);
            }
          }
        }
      }
    }

    //
    // Monster aus selektierten Zellen lschen
    //
    else if (e.getActionCommand().equals("delete monster")) {
      if (!this.selectedCellsVector.isEmpty()) {
        for (int i = 0; i < this.selectedCellsVector.size(); i++) {
          this.removeMonster((Point3i)((Cell)this.selectedCellsVector.get(i)).getPosition());
        }
      }
    }

    //
    // Monsterstartpositionen aus selektierten Zellen lschen
    //
    else if (e.getActionCommand().equals("delete monster startposition")) {
      if (!this.selectedCellsVector.isEmpty()) {
        for (int i = 0; i < this.selectedCellsVector.size(); i++)
          lab.removeMonsterStartPosition((Point3i)((Cell)this.selectedCellsVector.get(i)).getPosition());
      }
    }

    //
    // Pacmanstartpositionenen aus selektierten Zellen lschen
    //
    else if (e.getActionCommand().equals("delete pacman startposition")) {
      if (!this.selectedCellsVector.isEmpty()) {
        for (int i = 0; i < this.selectedCellsVector.size(); i++)
          lab.removePacmanStartPosition((Point3i)((Cell)this.selectedCellsVector.get(i)).getPosition());
      }
    }
  repaint();
  }


  /**
   * Methode, um gesetzte Items in der 2D-Leveleditierung ein- bzw. auszublenden.
   */
  public void drawItems() {
    drawItems = !drawItems;
    repaint();
  }

  /**
   * Methode, um gesetzte Monster in der 2D-Leveleditierung ein- bzw. auszublenden.
   */
  public void drawMonsters() {
    drawMonsters = !drawMonsters;
    repaint();
  }

  /**
   * Methode, um gesetzte Monster und Monsterstartpositionen in der 2D-Leveleditierung
   * ein- bzw. auszublenden.
   */
  public void drawMonstersAndPossibleStartPositions() {
    drawMonsters = !drawMonsters;
    repaint();
  }

  /**
   * Methode, um gesetzte Pacmanstartpositionen in der 2D-Leveleditierung
   * ein- bzw. auszublenden.
   */
  public void drawPacmanStartPositions() {
    drawPacmanStartPositions = !drawPacmanStartPositions;
    repaint();
  }

  /**
   * Methode, um die Einblendung der begehbaren Zellen der Levelebene unterhalb
   * der aktuell editierten Levelebene in diese zu aktivieren bzw. deaktivieren.
   */
  public void drawLowerLevelFloors() {
    drawLowerLevelFloors = !drawLowerLevelFloors;
    repaint();
  }

  /**
   * Methode, um die Einblendung der nicht begehbaren Zellen der Levelebene unterhalb
   * der aktuell editierten Levelebene in diese zu aktivieren bzw. deaktivieren.
   */
  public void drawLowerLevelWalls() {
    drawLowerLevelWalls = !drawLowerLevelWalls;
    repaint();
  }

  /**
   * Methode, um die Einblendung der begehbaren Zellen der Levelebene oberhalb
   * der aktuell editierten Levelebene in diese zu aktivieren bzw. deaktivieren.
   */
  public void drawUpperLevelFloors() {
    drawUpperLevelFloors = !drawUpperLevelFloors;
    repaint();
  }

  /**
   * Methode, um die Einblendung der nicht begehbaren Zellen der Levelebene oberhalb
   * der aktuell editierten Levelebene in diese zu aktivieren bzw. deaktivieren.
   */
  public void drawUpperLevelWalls() {
    drawUpperLevelWalls = !drawUpperLevelWalls;
    repaint();
  }

  /**
   * Hilfsmethode, die das Zeichnen der nicht begehbarene Zellen der Levelebene
   * unterhalb der aktuell editierten Levelebene in diese bewerkstelligt. Diese
   * Methode wird gegebenfalls automatisch von der Methode {@link #paint paint(Graphics g)}
   * dieser Klasse aufgerufen.
   *
   * @param g Graphics-Instanz mit Referenz auf den Zeichenbereich dieser Komponente.
   * @see #paint
   */
  private final void drawLowerLevelWalls(Graphics g) {
    if (this.editLevelNumber > 0)
      for (int x = 0; x < lab.getConstraints().x; x++)
        for (int z = 0; z < lab.getConstraints().z; z++) {
          if (!lab.getCell(x, this.editLevelNumber-1, z).isWalkable()) {
            g.setColor(this.colorLowerWalls);
            g.drawRect(x*zoomFac+3, z*zoomFac+3, zoomFac-7, zoomFac-7);
          }
        }
  }

  /**
   * Hilfsmethode, die das Zeichnen der begehbarene Zellen der Levelebene
   * unterhalb der aktuell editierten Levelebene in diese bewerkstelligt. Diese
   * Methode wird gegebenfalls automatisch von der Methode {@link #paint paint(Graphics g)}
   * dieser Klasse aufgerufen.
   *
   * @param g Graphics-Instanz mit Referenz auf den Zeichenbereich dieser Komponente.
   * @see #paint
   */
  private final void drawLowerLevelFloors(Graphics g) {
    if (this.editLevelNumber > 0)
      for (int x = 0; x < lab.getConstraints().x; x++)
        for (int z = 0; z < lab.getConstraints().z; z++) {
          if (lab.getCell(x, this.editLevelNumber-1, z).isWalkable()) {
            g.setColor(this.colorLowerFloors);
            g.drawRect(x*zoomFac+3, z*zoomFac+3, zoomFac-7, zoomFac-7);
          }
        }
  }

  /**
   * Hilfsmethode, die das Zeichnen der begehbarene Zellen der Levelebene
   * oberhalb der aktuell editierten Levelebene in diese bewerkstelligt. Diese
   * Methode wird gegebenfalls automatisch von der Methode {@link #paint paint(Graphics g)}
   * dieser Klasse aufgerufen.
   *
   * @param g Graphics-Instanz mit Referenz auf den Zeichenbereich dieser Komponente.
   * @see #paint
   */
  private final void drawUpperLevelFloors(Graphics g) {
    if (this.editLevelNumber < lab.getConstraints().y-1)
      for (int x = 0; x < lab.getConstraints().x; x++)
        for (int z = 0; z < lab.getConstraints().z; z++) {
          if (lab.getCell(x, this.editLevelNumber+1, z).isWalkable()) {
            g.setColor(this.colorUpperFloors);
            g.drawRect(x*zoomFac+6, z*zoomFac+6, zoomFac-13, zoomFac-13);
          }
        }
  }

  /**
   * Hilfsmethode, die das Zeichnen der nicht begehbarene Zellen der Levelebene
   * oberhalb der aktuell editierten Levelebene in diese bewerkstelligt. Diese
   * Methode wird gegebenfalls automatisch von der Methode {@link #paint paint(Graphics g)}
   * dieser Klasse aufgerufen.
   *
   * @param g Graphics-Instanz mit Referenz auf den Zeichenbereich dieser Komponente.
   * @see #paint
   */
  private final void drawUpperLevelWalls(Graphics g) {
    if (this.editLevelNumber < lab.getConstraints().y-1)
      for (int x = 0; x < lab.getConstraints().x; x++)
        for (int z = 0; z < lab.getConstraints().z; z++) {
          if (!lab.getCell(x, this.editLevelNumber+1, z).isWalkable()) {
            g.setColor(this.colorUpperWalls);
            g.drawRect(x*zoomFac+6, z*zoomFac+6, zoomFac-13, zoomFac-13);
          }
        }
  }

  /**
   * Hilfsmethode, die das Zeichnen der in der aktuell editierten Levelebene
   * befindlichen Items bernimmt, falls diese angezeigt werden sollen. Diese
   * Methode wird gegebenfalls automatisch von der Methode
   * {@link #paint paint(Graphics g)} dieser Klasse aufgerufen.
   *
   * @param g Graphics-Instanz mit Referenz auf den Zeichenbereich dieser Komponente.
   * @see #paint
   */
  private final void drawItems(Graphics g) {
    for (int x = 0; x < lab.getConstraints().x; x++)
        for (int z = 0; z < lab.getConstraints().z; z++) {
          if (lab.getCell(x, this.editLevelNumber, z).isWalkable() &&
              lab.getCell(x, this.editLevelNumber, z).hasItems()) {
            g.setColor(this.colorItem);

            // Falls Item selektiert, dann hellere Farbe whlen:
            for (int k = 0; k < this.selectedCellsVector.size(); k++) {
              Point3i point = (Point3i)((Cell)this.selectedCellsVector.get(k)).getPosition();
              if (point.equals(new Point3i(x, this.editLevelNumber, z))) {
                g.setColor(this.colorItem.brighter());
                break;
              }
            }
            g.drawLine(x*zoomFac+1, z*zoomFac+1, x*zoomFac+zoomFac-1, z*zoomFac+zoomFac-1);
            for (int i = 0; i < (int)zoomFac/4; i++) {
              g.drawLine(x*zoomFac+1, z*zoomFac+1+i, x*zoomFac+zoomFac-1-i, z*zoomFac+zoomFac-1);
            }
          }
        }
  }

  /**
   * Hilfsmethode, die das Zeichnen der in der aktuell editierten Levelebene
   * befindlichen Monster bernimmt, falls diese angezeigt werden sollen. Diese
   * Methode wird gegebenfalls automatisch von der Methode
   * {@link #paint paint(Graphics g)} dieser Klasse aufgerufen.
   *
   * @param g Graphics-Instanz mit Referenz auf den Zeichenbereich dieser Komponente.
   * @see #paint
   */
  private final void drawMonsters(Graphics g) {
    if (lab.getMonsterCount() > 0) {
      pacman3d.message.ID id[] =  lab.getAllMonsterIds();
      for (int i = 0; i < lab.getMonsterCount(); i++) {
        Point3i pos = lab.getMonsterPos(id[i]);

        if (pos.y == this.editLevelNumber) {
          g.setColor(this.colorMonster);
          for (int k = 0; k < this.selectedCellsVector.size(); k++) {
            Point3i point = (Point3i)((Cell)this.selectedCellsVector.get(k)).getPosition();

            if (point.equals(pos)) {
              g.setColor(this.colorMonster.brighter());
              break;
            }
          }
          g.drawLine(pos.x*zoomFac+1, pos.z*zoomFac+zoomFac-1, pos.x*zoomFac+zoomFac-1, pos.z*zoomFac+1);
          for (int j = 0; j < (int)zoomFac/4; j++) {
            g.drawLine(pos.x*zoomFac+1, pos.z*zoomFac+zoomFac-1-j, pos.x*zoomFac+zoomFac-1-j, pos.z*zoomFac+1);
          }
        }
      }
    }
  }

  /**
   * Methode, um gesetzte Monsterstartpositionen in der 2D-Leveleditierung ein-
   * bzw. auszublenden.
   */
  public void showMonsterStartpositions() {
    if (vecMonsterStartpositions == null) {
      vecMonsterStartpositions = lab.getAllMonsterStartPositions();
    }
    else {
      vecMonsterStartpositions = null;
    }
    repaint();
  }

  /**
   * Methode, um gesetzte Pacmanstartpositionen in der 2D-Leveleditierung ein-
   * bzw. auszublenden.
   */
  public void showPacmanStartpositions() {
    if (vecPacmanStartpositions == null) {
      vecPacmanStartpositions = lab.getAllPacmanStartPositions();
    }
    else {
      vecPacmanStartpositions = null;
    }
    repaint();
  }

  /**
   * Hilfsmethode, die das Lschen von Monster und Monsterstartposition in der Zelle
   * an der Position <b>point</b> bernimmt.
   *
   * @param point Zellenposition, in welcher ein Monster und Monsterstartposition gelscht
   *              werden sollen.
   */
  private void removeMonster(Point3i point) {
    if (lab.getMonsterCount() > 0 && point != null) {
      pacman3d.message.ID id[] = lab.getAllMonsterIds();
      int numMonster = lab.getMonsterCount();
      for (int i = 0; i < numMonster; i++) {
        Point3i pos = lab.getMonsterPos(id[i]);
        if (pos != null)
          if (point.x == pos.x && point.y == pos.y && point.z == pos.z) {
            lab.removeMonster(id[i]);
          }
      }
    }
  }

  /**
   * Zur Zeit nicht an aktuellen Stand angepat!
   *
   * @deprecated 01.01.2002
   */
  protected void closeFloorsToLower() {
    if (this.editLevelNumber >= 1)
      for (int x = 0; x < lab.getConstraints().x; x++)
        for (int z = 0; z < lab.getConstraints().z; z++)
          if (lab.getCell(x, this.editLevelNumber, z).isWalkable() == true) {
              //removeMonster(new Point3i(x, editLevelNumber-1, z));
          }
    repaint();
  }

  /**
   * Zur Zeit nicht an aktuellen Stand angepat!
   *
   * @deprecated 01.01.2002
   */
  protected void openWallsToLower() {
    if (this.editLevelNumber >= 1)
      for (int x = 0; x < lab.getConstraints().x; x++)
        for (int z = 0; z < lab.getConstraints().z; z++)
          if (lab.getCell(x, this.editLevelNumber, z).isWalkable() == false) {
              //lab.setCell(new CellFloor(new Point3i(x, this.editLevelNumber-1, z)),new Point3i(x, editLevelNumber-1, z));
              //removeMonster(new Point3i(x, editLevelNumber-1, z));
          }
    repaint();
  }

  /**
   * Zur Zeit nicht an aktuellen Stand angepat!
   *
   * @deprecated 01.01.2002
   */
  protected void closeFloorsToUpper() {
    if (this.editLevelNumber < lab.getConstraints().y-1)
      for (int x = 0; x < lab.getConstraints().x; x++)
        for (int z = 0; z < lab.getConstraints().z; z++)
          if (lab.getCell(x, this.editLevelNumber, z).isWalkable() == true) {
              //lab.setCell(new CellWall(new Point3i(x, this.editLevelNumber+1, z)),new Point3i(x, editLevelNumber+1, z));
              //removeMonster(new Point3i(x, editLevelNumber+1, z));
          }
    repaint();
  }

  /**
   * Zur Zeit nicht an aktuellen Stand angepat!
   *
   * @deprecated 01.01.2002
   */
  protected void openWallsToUpper() {
    if (this.editLevelNumber < lab.getConstraints().y)
      for (int x = 0; x < lab.getConstraints().x; x++)
        for (int z = 0; z < lab.getConstraints().z; z++)
          if (lab.getCell(x, this.editLevelNumber, z).isWalkable() == false) {
              //lab.setCell(new CellFloor(new Point3i(x, this.editLevelNumber+1, z)),new Point3i(x, editLevelNumber+1, z));
              //removeMonster(new Point3i(x, editLevelNumber-1, z));
          }
    repaint();
  }

  /**
   * Initialisierung diverser GUI-Einstellungen dieser Komponente. Diese Methode
   * wird automatisch vom Konstruktor dieser Klasse aufgerufen und sollte somit nicht
   * von Interesse fr den Endanwender sein.
   *
   * @see #LevelEditingJPanel
   */
  private void guiInit() {
    // Diese Panel als MouseListener und MouseMotionListener registrieren, da wir
    // z.B. ein MouseDrag zur Editierung realisieren wollen.
    this.addMouseListener(this);
    this.addMouseMotionListener(this);

    // Gre dieser Komponente initial festlegen, damit die Gre nicht von einem
    // LayoutManager eigenwillig verndert wird und somit vieleicht Cells nicht mehr
    // quadratisch angezeigt werden wrden...
    setPreferredSize(new Dimension(zoomFac*lab.getConstraints().x, zoomFac*lab.getConstraints().z));
    setMaximumSize(new Dimension(zoomFac*lab.getConstraints().x, zoomFac*lab.getConstraints().z));
    setMinimumSize(new Dimension(zoomFac*lab.getConstraints().x, zoomFac*lab.getConstraints().z));
    this.setSize(new Dimension(zoomFac*lab.getConstraints().x, zoomFac*lab.getConstraints().z));

    // Farben setzen:
    if (leveleditor.colorChooser2DComponentsInternalFrame != null) {
      Color[] colors = ((ColorChooser2DComponents)leveleditor.colorChooser2DComponentsInternalFrame).colors;
      int i = 0;
      this.colorGrid = colors[i++];
      this.colorLowerFloors = colors[i++];
      this.colorLowerWalls = colors[i++];
      this.colorUpperFloors = colors[i++];
      this.colorUpperWalls = colors[i++];
      this.colorMonster = colors[i++];
      this.colorMonsterStartPosition = colors[i++];
      this.colorPacmanStartPosition = colors[i++];
      this.colorItem = colors[i++];
      this.colorFloor = colors[i++];
      this.colorWall = colors[i++];
      this.colorLadder = colors[i++];
      this.colorSimpleWindow = colors[i++];
      this.colorGlassWall = colors[i++];
      this.colorSimplePillar = colors[i++];
      this.colorCone = colors[i++];
      this.colorArchway = colors[i++];
      this.colorSimpleDoor = colors[i++];
      this.colorQuaterHighWall = colors[i++];
    }
  }

  /**
   * Hilfsmethode, die berprft, ob sich eine Zelle mit Anfangspunkt (anchorX, anchorY)
   * innerhalb des Rechteckes (x, y, width, height) befindet. Ist dies der Fall,
   * so wird <b>true</b> zurckgegeben, andernfalls <b>false</b>. Diese Hilfsmethode
   * wird zur Selektion von Zellen auf dem 2D-Leveleditierungsbereich mittels Maus-Drag
   * verwendet und von der Methode {@link #mouseDragged mouseDragged} aus
   * aufgerufen.
   *
   * @param anchorX x-Koordinate der zu berprfenden Zelle auf dem Bildschirm.
   * @param anchorY y-Koordinate der zu berprfenden Zelle auf dem Bildschirm.
   * @param x x-Koordinate des Anfangs des Mausfangs.
   * @param y y-Koordinate des Anfangs des Mausfangs.
   * @param width Breite des Mausfangs.
   * @param height Hhe des Mausfangs.
   *
   * @return <b>true</b>, falls sich die zu berprfende Zelle innerhalb des Mausfangs
   *         befindet, andernfalls <b>false</b>.
   */
  private final boolean containsCell(int anchorX, int anchorY, int x, int y, int width, int height) {
    // Anfangspunkte der dargestellten Zelle
    anchorX = anchorX * zoomFac;
    anchorY = anchorY * zoomFac;
    // Endpunkte der dargestellten Zelle
    int secondX = anchorX + zoomFac;
    int secondY = anchorY + zoomFac;

    // Fall: Mind. ein Punkt einer Zelle vollstndig enthalten
    if (x <= anchorX && anchorX <= (x+width) && y <= anchorY && anchorY <= (y+height) ||
        x <= secondX && secondX <= (x+width) && y <= anchorY && anchorY <= (y+height) ||
        x <= anchorX && anchorX <= (x+width) && y <= secondY && secondY <= (y+height) ||
        x <= secondX && secondX <= (x+width) && y <= secondY && secondY <= (y+height))
      return true;

    // Vertikaler Fall:
    else if (anchorX <= x && (x+width) <= secondX) {
      // "Erste Zelle"
      if (anchorY <= y && y <= secondY)
        return true;
      // "Letzte Zelle"
      else if (anchorY <= (y+height) && (y+height) <= secondY)
        return true;
      // Zellen zw. der ersten und letzten Zelle :o)
      else if (y <= anchorY && secondY <= (y+height))
        return true;
    }

    // Horizontaler Fall:
    else if (anchorY <= y && (y+height) <= secondY) {
      // "Erste Zelle"
      if (anchorX <= x && x <= secondX)
        return true;
      // "Letzte Zelle"
      else if (anchorX <= (x+width) && (x+width) <= secondX)
        return true;
      // Zellen zw. der ersten und letzten Zelle :o)
      else if (x <= anchorX && secondX <= (x+width))
        return true;
    }
    return false;
  }
}