package pacman3d.labyrinth.leveleditor;

import pacman3d.labyrinth.*;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import javax.vecmath.*;
import pacman3d.labyrinth.items.*;

/**
 * <b>Titel:</b>Pacman 3D - Leveleditor<p>
 *
 * <b>Beschreibung:</b><br>
 * Diese Klasse ist eine Subklasse von {@link javax.swing.JPanel JPanel} und
 * kann als solche wie ein {@link javax.swing.JPanel JPanel} als GUI-Komponente
 * verwendet werden.<br>
 * Diese Subklasse stellt alle dem Leveleditor zur Verfgung stehenden
 * {@link pacman3d.labyrinth.items.Item Items} auf dem Bildschirm dar.
 * Jedem {@link pacman3d.labyrinth.items.Item Item} ist ein eigener
 * {@link javax.swing.JTabbedPane Reiter} zugeordnet, in welchen sich weitere
 * GUI-Komponenten befinden, wie z.B. zur Positionierung eines {@link
 * pacman3d.labyrinth.items.Item Item} innerhalb einer {@link
 * pacman3d.labyrinth.cells.Cell Zelle}, als auch z.B. zur Darstellung der
 * zugewiesenen Lichtfarbe und Blinkintervall, bei dem Itemtyp {@link
 * pacman3d.labyrinth.items.ItemSignalLamp ItemSignalLamp}.<p>
 *
 * <b>Copyright:</b>	Copyright (c) 2001/02<br>
 *
 * @author              Labyrinth-Gruppe<br>
 * @version             1.0 03/15/2002<br>
 */
public class ItemJPanel extends JPanel implements java.awt.event.FocusListener {
  /**
   * Instanz der Klasse {@link javax.swing.JTabbedPane JTabbedPane}, in welchem
   * die durch diese Komponente zur Auswahl stehenden {@link
   * pacman3d.labyrinth.items.Item Items} in Reitern organisiert zur Auswahl
   * stehen.
   */
  protected JTabbedPane tabbedPane = null;

  /**
   * Array vom Typ {@link javax.swing.JTextField JTextField}, welche dazu verwendet
   * werden, um einem zu setzenden  {@link pacman3d.labyrinth.items.Item Item}
   * innerhalb der zu setzenden {@link pacman3d.labyrinth.cells.Cell Zelle} zu
   * positionieren.
   */
  protected JTextField[] itemPositionsTextFields = new JTextField[45];

  /**
   * Instanz der Klasse {@link javax.swing.JCheckBox JCheckBox}, welche bei
   * Auswahl des Reiters <b>Fackel</b> und Aktivierung dieser JCheckBox dazu fhrt,
   * da die zu setzende Instanz der Klasse {@link pacman3d.labyrinth.items.ItemTorch
   * ItemTorch} innerhalb der zu setzenden Zelle nach links ausgerichtet ist.
   *
   * @see #torchRightCheckBox
   * @see #torchFrontCheckBox
   * @see #torchBackCheckBox
   */
  protected JCheckBox torchLeftCheckBox;

  /**
   * Instanz der Klasse {@link javax.swing.JCheckBox JCheckBox}, welche bei
   * Auswahl des Reiters <b>Fackel</b> und Aktivierung dieser JCheckBox dazu fhrt,
   * da die zu setzende Instanz der Klasse {@link pacman3d.labyrinth.items.ItemTorch
   * ItemTorch} innerhalb der zu setzenden Zelle nach rechts ausgerichtet ist.
   *
   * @see #torchLeftCheckBox
   * @see #torchFrontCheckBox
   * @see #torchBackCheckBox
   */
  protected JCheckBox torchRightCheckBox;

  /**
   * Instanz der Klasse {@link javax.swing.JCheckBox JCheckBox}, welche bei
   * Auswahl des Reiters <b>Fackel</b> und Aktivierung dieser JCheckBox dazu fhrt,
   * da die zu setzende Instanz der Klasse {@link pacman3d.labyrinth.items.ItemTorch
   * ItemTorch} innerhalb der zu setzenden Zelle nach vorne ausgerichtet ist.
   *
   * @see #torchLeftCheckBox
   * @see #torchRightCheckBox
   * @see #torchBackCheckBox
   */
  protected JCheckBox torchFrontCheckBox;

  /**
   * Instanz der Klasse {@link javax.swing.JCheckBox JCheckBox}, welche bei
   * Auswahl des Reiters <b>Fackel</b> und Aktivierung dieser JCheckBox dazu fhrt,
   * da die zu setzende Instanz der Klasse {@link pacman3d.labyrinth.items.ItemTorch
   * ItemTorch} innerhalb der zu setzenden Zelle nach hinten ausgerichtet ist.
   *
   * @see #torchLeftCheckBox
   * @see #torchRightCheckBox
   * @see #torchFrontCheckBox
   */
  protected JCheckBox torchBackCheckBox;

  /**
   * Hilfsfeld vom Typ <b>double</b>, in welchem die alte Position bei neuer
   * Positionierung durch den Benutzer abgespeichert wird, da falls wenn dieser
   * eine falsche bzw. unzulssige Eingabe machen sollte, die alte Positionsangabe
   * wieder hergestellt werden kann.
   */
  protected double old_position = 0;

  /**
   * Referenz auf die {@link pacman3d.labyrinth.leveleditor.Leveleditor Hauptklasse
   * des Leveleditors}, welche diese Klasse instantiiert. Diese Referenz dient
   * dazu, um auf diverse andere Klassen innerhalb des Packages des Leveleditors
   * Zugriff zu haben.
   */
  protected Leveleditor leveleditor;

  /**
   * Instanz der Klasse {@link pacman3d.labyrinth.leveleditor.PreviewJLabel
   * PreviewJLabel}, welche hier dazu verwendet wird, um die vom Benutzer
   * gewhlte Blinklichtfarbe fr den Itemtyp {@link
   * pacman3d.labyrinth.items.ItemSignalLamp ItemSignalLamp} innerhalb dieser
   * Komponente darzustellen.
   */
  protected PreviewJLabel signalLampColorLabel;

  /**
   * Instanz der Klasse {@link javax.swing.JTextField JTextField} zur Eingabe
   * des Blinkintervalls des Itemtyps {@link pacman3d.labyrinth.items.ItemSignalLamp
   * ItemSignalLamp}.
   *
   * @see #torchDurationTextField
   */
  protected JTextField signalLampDurationTextField;

  /**
   * Instanz der Klasse {@link javax.swing.JTextField JTextField} zur Eingabe
   * des Flackerintervalls des Itemtyps {@link pacman3d.labyrinth.items.ItemTorch
   * ItemTorch}.
   *
   * @see #signalLampDurationTextField
   */
  protected JTextField torchDurationTextField;

  /**
   * Hilfsfeld vom Typ <b>int</b>, in welchem die alten Werte der JTextFields
   * {@link #signalLampDurationTextField} bzw. {@link #torchDurationTextField} bei
   * Eingabe eines neuen Wertes durch den Benutzer abgespeichtert werden, um
   * bei unzulssiger Eingabe das entsprechende JTextField mit dem alten Wert
   * wieder herzustellen.
   */
  protected int old_duration = 0;


  /**
   * Konstruktor:<p>
   *
   * Zur Initialisierung dieser Klasse ist dieser Konstruktor mit einer Referenz
   * der aufrufenden {@link pacman3d.labyrinth.leveleditor.Leveleditor Hauptklasse
   * <b>leveleditor</b>} aufzurufen. ber diese Referenz erfolgt dann whrend des
   * Programmablaufes ein Teil der "Kommunikation" innerhalb des Leveleditors.
   *
   * @see pacman3d.labyrinth.leveleditor.Leveleditor
   */
  public ItemJPanel(Leveleditor leveleditor) {
    this.leveleditor = leveleditor;
    guiInit();
  }

  /**
   * Initialisierung diverser GUI-Einstellungen dieser Komponente. Diese Methode
   * wird automatisch vom Konstruktor dieser Klasse aufgerufen und sollte somit nicht
   * von Interesse fr den Endanwender sein.
   *
   * @see #ItemJPanel
   */
  protected void guiInit() {
    // Textfelder zur Eingabe der Positionen initialisieren:
    for (int i = 0; i < itemPositionsTextFields.length; i++) {
      itemPositionsTextFields[i] = new JTextField("0.0", 5);
      itemPositionsTextFields[i].setHorizontalAlignment(JTextField.CENTER);
      itemPositionsTextFields[i].addFocusListener(this);
    }

     // Hauptlayout:
    this.setLayout(new BorderLayout());
    // Layout fr die einzelnen Panel der versch. Cell-Typen:
    GridBagLayout gbl       = new GridBagLayout();
    GridBagConstraints gbc  = null;

    // Fr jeden Zellentyp einen eigenen Reiter ->
    tabbedPane = new JTabbedPane(JTabbedPane.TOP);
    this.add(tabbedPane, BorderLayout.CENTER);

    // BallJPanel
    JPanel ballMainPanel = new JPanel(gbl);
    gbc = makeGBC(0, 0, 2, 2);
    gbc.anchor = gbc.CENTER;
    gbc.fill   = gbc.BOTH;
    gbc.weightx = 1.0;
    gbc.weighty = 1.0;
    JPanel panel = createItemPanel("Positionierung", 0);
    gbl.addLayoutComponent(panel,gbc);
    ballMainPanel.add(panel);

    tabbedPane.addTab("Ball", ballMainPanel);

    // Green Apple
    JPanel greenAppleMainPanel = new JPanel(gbl);
    gbc = makeGBC(0, 0, 2, 2);
    gbc.anchor = gbc.CENTER;
    gbc.fill   = gbc.BOTH;
    gbc.weightx = 1.0;
    gbc.weighty = 1.0;
    panel = createItemPanel("Positionierung", 1);
    gbl.addLayoutComponent(panel,gbc);
    greenAppleMainPanel.add(panel);
    tabbedPane.addTab("Grner Apfel", greenAppleMainPanel);

    // Red Apple
    JPanel redAppleMainPanel = new JPanel(gbl);
    gbc = makeGBC(0, 0, 2, 2);
    gbc.anchor = gbc.CENTER;
    gbc.fill   = gbc.BOTH;
    gbc.weightx = 1.0;
    gbc.weighty = 1.0;
    panel = createItemPanel("Positionierung", 2);
    gbl.addLayoutComponent(panel,gbc);
    redAppleMainPanel.add(panel);
    tabbedPane.addTab("Roter Apfel", redAppleMainPanel);

    // Pear
    JPanel pearMainPanel = new JPanel(gbl);
    gbc = makeGBC(0, 0, 2, 2);
    gbc.anchor = gbc.CENTER;
    gbc.fill   = gbc.BOTH;
    gbc.weightx = 1.0;
    gbc.weighty = 1.0;
    panel = createItemPanel("Positionierung", 3);
    gbl.addLayoutComponent(panel,gbc);
    pearMainPanel.add(panel);
    tabbedPane.addTab("Birne", pearMainPanel);

    // Cherry
    JPanel cherryMainPanel = new JPanel(gbl);
    gbc = makeGBC(0, 0, 2, 2);
    gbc.anchor = gbc.CENTER;
    gbc.fill   = gbc.BOTH;
    gbc.weightx = 1.0;
    gbc.weighty = 1.0;
    panel = createItemPanel("Positionierung", 4);
    gbl.addLayoutComponent(panel,gbc);
    cherryMainPanel.add(panel);
    tabbedPane.addTab("Kirsche", cherryMainPanel);

    // Banana
    JPanel bananaMainPanel = new JPanel(gbl);
    gbc = makeGBC(0, 0, 2, 2);
    gbc.anchor = gbc.CENTER;
    gbc.fill   = gbc.BOTH;
    gbc.weightx = 1.0;
    gbc.weighty = 1.0;
    panel = createItemPanel("Positionierung", 5);
    gbl.addLayoutComponent(panel,gbc);
    bananaMainPanel.add(panel);
    tabbedPane.addTab("Banane", bananaMainPanel);

    // Trauben
    JPanel grapeMainPanel = new JPanel(gbl);
    gbc = makeGBC(0, 0, 2, 2);
    gbc.anchor = gbc.CENTER;
    gbc.fill   = gbc.BOTH;
    gbc.weightx = 1.0;
    gbc.weighty = 1.0;
    panel = createItemPanel("Positionierung", 6);
    gbl.addLayoutComponent(panel,gbc);
    grapeMainPanel.add(panel);
    tabbedPane.addTab("Trauben", grapeMainPanel);

    // First-Aid-Box
    JPanel firstAidBoxMainPanel = new JPanel(gbl);
    gbc = makeGBC(0, 0, 2, 2);
    gbc.anchor = gbc.CENTER;
    gbc.fill   = gbc.BOTH;
    gbc.weightx = 1.0;
    gbc.weighty = 1.0;
    panel = createItemPanel("Positionierung", 7);
    gbl.addLayoutComponent(panel,gbc);
    firstAidBoxMainPanel.add(panel);
    tabbedPane.addTab("Erste Hilfe Kasten", firstAidBoxMainPanel);

    // Blue-Pill
    JPanel bluePillMainPanel = new JPanel(gbl);
    gbc = makeGBC(0, 0, 2, 2);
    gbc.anchor = gbc.CENTER;
    gbc.fill   = gbc.BOTH;
    gbc.weightx = 1.0;
    gbc.weighty = 1.0;
    panel = createItemPanel("Positionierung", 8);
    gbl.addLayoutComponent(panel,gbc);
    bluePillMainPanel.add(panel);
    tabbedPane.addTab("Blaue Pille", bluePillMainPanel);

    // Green Pill
    JPanel greenPillMainPanel = new JPanel(gbl);
    gbc = makeGBC(0, 0, 2, 2);
    gbc.anchor = gbc.CENTER;
    gbc.fill   = gbc.BOTH;
    gbc.weightx = 1.0;
    gbc.weighty = 1.0;
    panel = createItemPanel("Positionierung", 9);
    gbl.addLayoutComponent(panel,gbc);
    greenPillMainPanel.add(panel);
    tabbedPane.addTab("Grne Pille", greenPillMainPanel);

    // Orange Pill
    JPanel orangePillMainPanel = new JPanel(gbl);
    gbc = makeGBC(0, 0, 2, 2);
    gbc.anchor = gbc.CENTER;
    gbc.fill   = gbc.BOTH;
    gbc.weightx = 1.0;
    gbc.weighty = 1.0;
    panel = createItemPanel("Positionierung", 10);
    gbl.addLayoutComponent(panel,gbc);
    orangePillMainPanel.add(panel);
    tabbedPane.addTab("Orangene Pille", orangePillMainPanel);

    // Red Pill
    JPanel redPillMainPanel = new JPanel(gbl);
    gbc = makeGBC(0, 0, 2, 2);
    gbc.anchor = gbc.CENTER;
    gbc.fill   = gbc.BOTH;
    gbc.weightx = 1.0;
    gbc.weighty = 1.0;
    panel = createItemPanel("Positionierung", 11);
    gbl.addLayoutComponent(panel,gbc);
    redPillMainPanel.add(panel);
    tabbedPane.addTab("Rote Pille", redPillMainPanel);

    // Transparent Pill
    JPanel transparentPillMainPanel = new JPanel(gbl);
    gbc = makeGBC(0, 0, 2, 2);
    gbc.anchor = gbc.CENTER;
    gbc.fill   = gbc.BOTH;
    gbc.weightx = 1.0;
    gbc.weighty = 1.0;
    panel = createItemPanel("Positionierung", 12);
    gbl.addLayoutComponent(panel,gbc);
    transparentPillMainPanel.add(panel);
    tabbedPane.addTab("Transparente Pill", transparentPillMainPanel);

    // Machine gun
    JPanel machineGunMainPanel = new JPanel(gbl);
    gbc = makeGBC(0, 0, 2, 2);
    gbc.anchor = gbc.CENTER;
    gbc.fill   = gbc.BOTH;
    gbc.weightx = 1.0;
    gbc.weighty = 1.0;
    panel = createItemPanel("Positionierung", 13);
    gbl.addLayoutComponent(panel,gbc);
    machineGunMainPanel.add(panel);
    tabbedPane.addTab("Maschinengewehr", machineGunMainPanel);

    // Hand gun
    JPanel handGunMainPanel = new JPanel(gbl);
    gbc = makeGBC(0, 0, 2, 2);
    gbc.anchor = gbc.CENTER;
    gbc.fill   = gbc.BOTH;
    gbc.weightx = 1.0;
    gbc.weighty = 1.0;
    panel = createItemPanel("Positionierung", 14);
    gbl.addLayoutComponent(panel,gbc);
    handGunMainPanel.add(panel);
    tabbedPane.addTab("Pistole", handGunMainPanel);

    // SignalLamp
    JPanel signalLampMainPanel = new JPanel(gbl);

    // GUI-Elemente fr Farbe:
    JPanel signalLampColorPanel = new JPanel();
    signalLampColorPanel.setBorder(BorderFactory.createTitledBorder("Blinklichtfarbe"));
    signalLampColorLabel = new PreviewJLabel();
    Color3f color3f = leveleditor.textureSelectionPanel.getColor3f();

    signalLampColorLabel.setColor(color3f);
    signalLampColorPanel.add(signalLampColorLabel);

    gbc = makeGBC(0, 0, 2, 2);
    gbc.anchor = gbc.CENTER;
    gbc.fill   = gbc.BOTH;
    gbc.weightx = 0.5;
    gbc.weighty = 0.5;
    gbl.addLayoutComponent(signalLampColorPanel, gbc);
    signalLampMainPanel.add(signalLampColorPanel);

    // GUI-Elemente fr Blinkintervalleinstellung:
    JPanel signalLampDurationPanel = new JPanel();
    signalLampDurationPanel.setBorder(BorderFactory.createTitledBorder("Blinkintervall"));
    signalLampDurationTextField = new JTextField(new ItemSignalLamp().getDuration()+"",5);
    signalLampDurationTextField.addFocusListener(this);
    signalLampDurationTextField.setHorizontalAlignment(JTextField.CENTER);
    signalLampDurationPanel.add(signalLampDurationTextField);
    JLabel signalLampDurationLabel = new JLabel("ms");
    signalLampDurationLabel.setForeground(Color.black);
    signalLampDurationPanel.add(signalLampDurationLabel);

    gbc = makeGBC(0, 2, 1, 1);
    gbc.anchor = gbc.CENTER;
    gbc.fill   = gbc.BOTH;
    gbc.weightx = 0.5;
    gbc.weighty = 0.5;
    gbl.addLayoutComponent(signalLampDurationPanel, gbc);
    signalLampMainPanel.add(signalLampDurationPanel);

    JScrollPane scrollPaneSignalLamp = new JScrollPane(signalLampMainPanel);
    tabbedPane.addTab("Signallampe", scrollPaneSignalLamp);

    // Torch
    JPanel torchMainPanel = new JPanel(gbl);

    // GUI fr Posotionierung der Fackel
    JPanel torchPositionPanel = new JPanel();
    torchPositionPanel.setBorder(BorderFactory.createTitledBorder("Positionierung"));
    torchLeftCheckBox   = new JCheckBox("links");
    torchLeftCheckBox.setSelected(true);
    torchPositionPanel.add(torchLeftCheckBox);

    torchFrontCheckBox  = new JCheckBox("vorne");
    torchPositionPanel.add(torchFrontCheckBox);

    torchRightCheckBox  = new JCheckBox("rechts");
    torchPositionPanel.add(torchRightCheckBox);

    torchBackCheckBox   = new JCheckBox("hinten");
    torchPositionPanel.add(torchBackCheckBox);

    ButtonGroup torchPositionButtonGroup = new ButtonGroup();
    torchPositionButtonGroup.add(torchLeftCheckBox);
    torchPositionButtonGroup.add(torchFrontCheckBox);
    torchPositionButtonGroup.add(torchRightCheckBox);
    torchPositionButtonGroup.add(torchBackCheckBox);

    gbc = makeGBC(0, 0, 2, 2);
    gbc.anchor = gbc.CENTER;
    gbc.fill   = gbc.BOTH;
    gbc.weightx = 0.5;
    gbc.weighty = 0.5;
    gbl.addLayoutComponent(torchPositionPanel, gbc);
    torchMainPanel.add(torchPositionPanel);

    JPanel torchDurationPanel = new JPanel();
    torchDurationPanel.setBorder(BorderFactory.createTitledBorder("Flackerabstand"));
    torchDurationTextField = new JTextField(new ItemTorch().getDuration()+"",5);
    torchDurationTextField.addFocusListener(this);
    torchDurationTextField.setHorizontalAlignment(JTextField.CENTER);
    torchDurationPanel.add(torchDurationTextField);
    JLabel torchDurationLabel = new JLabel("ms");
    torchDurationLabel.setForeground(Color.black);
    torchDurationPanel.add(torchDurationLabel);

    gbc = makeGBC(0, 2, 1, 1);
    gbc.anchor = gbc.CENTER;
    gbc.fill   = gbc.BOTH;
    gbc.weightx = 0.5;
    gbc.weighty = 0.5;
    gbl.addLayoutComponent(torchDurationPanel, gbc);
    torchMainPanel.add(torchDurationPanel);

    tabbedPane.addTab("Fackel", torchMainPanel);
  }

 /**
  * Je nach dem, welcher Reiter der GUI-Komponeten {@link #tabbedPane} zur Auswahl
  * eines Items gewhlt wurde, wird der entsprechende Itemtyp instantiiert
  * und eine Referenz auf diese zurckgeliefert.
  *
  * @return    Instanz des selektierten Itemtyps.
  */
  protected Item produceSelectedItem() {
    double x, y, z;

    switch (tabbedPane.getSelectedIndex()) {
      case 0: return new ItemBall(getPosition(0));

      case 1: return new ItemGreenApple(getPosition(1));

      case 2: return new ItemRedApple(getPosition(2));

      case 3: return new ItemPear(getPosition(3));

      case 4: return new ItemCherry(getPosition(4));

      case 5: return new ItemBanana(getPosition(5));

      case 6: return new ItemGrape(getPosition(6));

      case 7: return new ItemFirstAidBox(getPosition(7));

      case 8: return new ItemBluePill(getPosition(8));

      case 9: return new ItemPillGreen(getPosition(9));

      case 10: return new ItemOrangePill(getPosition(10));

      case 11: return new ItemRedPill(getPosition(11));

      case 12: return new ItemTransparentPill(getPosition(12));

      case 13: return new ItemMachineGun(getPosition(13));

      case 14: return new ItemHandGun(getPosition(14));

      case 15: ItemSignalLamp signalLamp = new ItemSignalLamp(signalLampColorLabel.getColor3f());
               signalLamp.setDuration(Integer.parseInt(signalLampDurationTextField.getText()));
               return signalLamp;

      case 16:  try { ItemTorch itemTorch = null;
                  if (torchLeftCheckBox.isSelected()) {
                    itemTorch = new ItemTorch(ItemTorch.SIDE_LEFT);
                    itemTorch.setDuration(Integer.parseInt(torchDurationTextField.getText()));
                    return itemTorch;
                  }
                  else if (torchRightCheckBox.isSelected()) {
                    itemTorch = new ItemTorch(ItemTorch.SIDE_RIGHT);
                    itemTorch.setDuration(Integer.parseInt(torchDurationTextField.getText()));
                    return itemTorch;
                  }
                  else if (torchFrontCheckBox.isSelected()) {
                    itemTorch = new ItemTorch(ItemTorch.SIDE_BACK);
                    itemTorch.setDuration(Integer.parseInt(torchDurationTextField.getText()));
                    return itemTorch;
                  }
                  else if (torchBackCheckBox.isSelected()) {
                    itemTorch = new ItemTorch(ItemTorch.SIDE_FRONT);
                    itemTorch.setDuration(Integer.parseInt(torchDurationTextField.getText()));
                    return itemTorch;
                  }
                  // Sollte nicht eintreten... Aber man wei ja nie ;)
                  else {
                    itemTorch = new ItemTorch();
                    itemTorch.setDuration(Integer.parseInt(torchDurationTextField.getText()));
                    return itemTorch;
                  }
                }
                catch (Exception exception) {
                }
    }
    return null;
  }

  /**
   * Hilfsmethode, welche eine Instanz der Klasse {@link javax.swing.JPanel
   * JPanel} zurckliefert, gefllt mit 3 Feldern des Array
   * {@link #itemPositionsTextFields} zur Eingabe von Positionsangabe. Dabei bekommt
   * das zurckgelieferte JPanel einen Rahmen mit dem Title <b>title</b> und
   * der zweite bergebene Parameter <b>pos</b> dient dazu, um innerhalb des
   * Array {@link #itemPositionsTextFields} die richtigen Arrayelemente auszuwhlen.
   *
   * @param title Title des Rahmens des zurckgelieferten JPanels.
   * @param pos   Anfang der zu initialisierenden JTextField innerhalb des
   *              Arrays {@link #itemPositionsTextFields}.
   *
   * @return Instanz der Klasse {@link javax.swing.JPanel JPanel}, gefllt
   *         mit drei JTextFields und einem Rahmen mit dem Title <b>title</b>.
   */
  private JPanel createItemPanel(String title, int pos) {
    JPanel jpanel = new JPanel();
    jpanel.setBorder(BorderFactory.createTitledBorder(title));
    //jpanel.add(new JLabel("Position"));
    JLabel label = new JLabel("x");
    label.setForeground(Color.black);
    jpanel.add(label);
    jpanel.add(this.itemPositionsTextFields[pos*3]);
    label = new JLabel("y");
    label.setForeground(Color.black);
    jpanel.add(label);
    jpanel.add(this.itemPositionsTextFields[pos*3+1]);
    label = new JLabel("z");
    label.setForeground(Color.black);
    jpanel.add(label);
    jpanel.add(this.itemPositionsTextFields[pos*3+2]);
    return jpanel;
  }

  /**
   * Hilfsmethode, um das GridBagLayout fr diese Methode zu verwenden.
   *
   * @param x       x-Anfangskoordinate im Gitternetz
   * @param y       y-Anfangskoordinate im Gitternetz
   * @param width   Erstreckung der Komponente in horizontaler Richtung auf dem Gitternetz
   * @param height  Erstreckung der KOmponente in vertikaler Richtung auf dem Gitternetz
   */
  protected GridBagConstraints makeGBC(int x, int y, int width, int height) {
    GridBagConstraints gbc = new GridBagConstraints();
    gbc.gridx = x;
    gbc.gridy = y;
    gbc.gridwidth = width;
    gbc.gridheight = height;
    gbc.insets = new Insets(1, 1, 1, 1);
    return gbc;
  }

  /**
   * Diese Methode wird immer dann aufgerufen, wenn einer der GUI-Komponenten
   * aus {@link #itemPositionsTextFields} oder {@link #signalLampDurationTextField}
   * oder {@link #torchDurationTextField} den Eingabefocus erhaltne. Ist dies der
   * Fall, so wird der alte Wert der focuserhaltenden Komponente abgespeichert, um
   * diesen evtl. fr eine vom Benutzer falsch gemachte Eingabe wieder einzusetzten.
   *
   * @param focusEvent java.awt.event.FocusEvent
   *
   * @see #focusLost
   */
  public void focusGained(java.awt.event.FocusEvent focusEvent) {
    if (focusEvent.getSource() instanceof JTextField) {
      if (focusEvent.getSource() == signalLampDurationTextField)
        old_duration = Integer.parseInt(signalLampDurationTextField.getText());
      else if (focusEvent.getSource() == torchDurationTextField)
        old_duration = Integer.parseInt(torchDurationTextField.getText());
      else
        old_position = Double.parseDouble(((JTextField)focusEvent.getSource()).getText());
    }
  }

  /**
   * Diese Methode wird immer dann aufgerufen, wenn einer der GUI-Komponenten
   * aus {@link #itemPositionsTextFields} oder {@link #signalLampDurationTextField}
   * oder {@link #torchDurationTextField} den Eingabefocus verliert. Ist dies der
   * Fall, so wird innerhalb dieser Methode berprft, ob es sich bei der durch
   * den Bentuzer gemachten Eingabe um eine zulssige Eingabe handelt. Ist dies
   * nicht der Fall, so wird die unzulssige Eingabe durch die zurvor berschriebene
   * Eingabe wieder ersetzt.
   *
   * @param focusEvent java.awt.event.FocusEvent
   *
   * @see #focusGained
   */
  public void focusLost(java.awt.event.FocusEvent focusEvent) {
    if (focusEvent.getSource() instanceof JTextField) {
      if (focusEvent.getSource() == signalLampDurationTextField) {
        try {
          int duration = Integer.parseInt(signalLampDurationTextField.getText());
          if (duration >= 0)
            signalLampDurationTextField.setText(duration+"");
          else
            signalLampDurationTextField.setText(old_duration+"");
        }
        catch (Exception exception) {
          signalLampDurationTextField.setText(old_duration+"");
        }
      }
      else if (focusEvent.getSource() == torchDurationTextField) {
        try {
          int duration = Integer.parseInt(torchDurationTextField.getText());
          if (duration >= 0)
            torchDurationTextField.setText(duration+"");
          else
            torchDurationTextField.setText(old_duration+"");
        }
        catch (Exception exception) {
          torchDurationTextField.setText(old_duration+"");
        }
      }
      else
        try {
          double value = Double.parseDouble(((JTextField)focusEvent.getSource()).getText());
          if (value < -0.5d || value > 0.5d)
            ((JTextField)focusEvent.getSource()).setText(old_position+"");
          else
            ((JTextField)focusEvent.getSource()).setText(value+"");
        }
        catch (Exception exception) {
          ((JTextField)focusEvent.getSource()).setText(old_position+"");
        }
    }
  }

  /**
   * Diese Hilfsmethode liefert die Inhalte der JTextField
   * {@link #itemPositionsTextFields itemPositionsTextFields[index*3], itemPositionsTextFields[index*3+1]
   * und itemPositionsTextFields[index*3+2]} als Instanz der Klasse {@link
   * javax.vecmath.Point3d Point3d} zurck.
   *
   * @param index Positionsangabe innerhalb des Arrays {@link #itemPositionsTextFields},
   *              ab welcher die nachfolgenden drei JTextFields ausgelesen werden sollen.
   *
   * @return Instanz der Klasse {@link javax.vecmath.Point3d Point3d} initialisiert
   *         mit den Werten der JTextFields {@link #itemPositionsTextFields
   *         itemPositionsTextFields[index*3], itemPositionsTextFields[index*3+1]
   *         und itemPositionsTextFields[index*3+2]}.
   */
  private Point3d getPosition(int index) {
    return new Point3d(Double.parseDouble(this.itemPositionsTextFields[index*3].getText()),
                       Double.parseDouble(this.itemPositionsTextFields[index*3+1].getText()),
                       Double.parseDouble(this.itemPositionsTextFields[index*3+2].getText()));
  }
}