package pacman3d.labyrinth.leveleditor;

import pacman3d.labyrinth.*;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import javax.media.j3d.*;
import java.awt.image.*;
import javax.vecmath.*;

/**
 * <b>Titel:</b>Pacman 3D - Leveleditor<p>
 *
 * <b>Beschreibung:</b><br>
 * Diese Klasse ist eine Subklasse von {@link javax.swing.JPanel JPanel} und
 * kann als solche GUI-Komponente verwendet werden.<br>
 * Diese Subklasse verwendet je nach {@link pacman3d.labyrinth.cells.Cell
 * Zellentyp} mehrere Instanzen der Klasse {@link
 * pacman3d.labyrinth.leveleditor.PreviewJLabel PreviewJLabel}, um die evtl.
 * einer {@link pacman3d.labyrinth.cells.Cell Zelle} zugewiesenen
 * Texturen bzw. Farben auf dem Bildschirm darzustellen.<p>
 *
 * <b>Copyright:</b>	Copyright (c) 2001/02<br>
 *
 * @author              Labyrinth-Gruppe<br>
 * @version             1.0 03/15/2002<br>
 */
public class AppearanceJPanel extends JPanel {

  /**
   * Instanzen der Klasse {@link pacman3d.labyrinth.leveleditor.PreviewJLabel
   * PreviewJLabel} zur Darstellung der anzuzeigenden Textur bzw. Farbe.
   */
  protected PreviewJLabel[] previewLabels;

  /**
   * Diejenigen Instanzen der Klasse {@link javax.media.j3d.Appearance Appearance},
   * welche von der selektierten Zelle {@link #selectedCell} dargestellt werden
   * sollen.
   */
  protected Appearance[] appearances;

  /**
   * Referenz auf denjenigen Zellentyp, dessen Textur(en) bzw. Farbe(en) durch
   * diese Komponente dargestellt werden sollen.
   */
  protected pacman3d.labyrinth.cells.Cell selectedCell;

  /**
   * Referenz auf diejenige Instanz der Klasse
   * {@link pacman3d.labyrinth.leveleditor.TextureSelectionJPanel
   * TextureSelectionJPanel}, welche von {@link
   * pacman3d.labyrinth.leveleditor.Leveleditor Hauptklasse des Leveleditors}
   * zur Auswahl von Texturen bzw. Farben verwendet wird.
   */
  protected TextureSelectionJPanel textureSelectionPanel;

  /**
   * Konstruktor:<br>
   *
   * Zur Initialisierung dieser Komponente ist ein Array vom Typ {@link
   * java.lang.String String} als Parameter zu bergeben. Die Lnge diese Arrays
   * bestimmt die Anzahl der fr die Textur- bzw. Farbdarstellung verwendeten
   * Instanzen der Klasse {@link pacman3d.labyrinth.leveleditor.PreviewJLabel
   * PreviewJLabel} und die entsprechend initialisierten String-Instanzen die
   * verwendeten Bezeichnungen in der Darstellung auf dem Bildschirm.
   *
   * @param descriptions Bezeichner der die Texturen bzw. Farben darstellenden
   *        Komponenten.
   */
  public AppearanceJPanel(String[] descriptions) {
    guiInit(descriptions);
  }

  /**
   * Dieser Methode ist derjenige Zellentyp, d.h. Subklasse der Klasse
   * {@link pacman3d.labyrinth.cells.Cell Cell}, zu bergeben, dessen Textur bzw.
   * Farbe durch diese Komponente auf dem Bildschirm dargestellt werden soll.
   *
   * @param selectedCell  Referenz auf denjenigen Zellentyp, dessen gesetzte(n) Textur(en)
   *                      bzw. Farbe(n) durch diese Komponente auf dem Bildschirm angezeigt
   *                      werden soll.
   */
  protected void updateAppearances(pacman3d.labyrinth.cells.Cell selectedCell) {
    this.selectedCell = selectedCell;

    if (selectedCell != null)
      this.appearances = selectedCell.getAppearance();
    else
      this.appearances = null;

    // Natrlich machen wir nur was, wenn das Array != null ist, denn
    // sonst -> Peng :)
    if (appearances != null) {
      for (int i = 0; i < previewLabels.length; i++) {
        if (appearances[i] != null) {
          // Falls Textur gesetzt,
          if (appearances[i].getTexture() != null) {
            // so holen wir diese und
            BufferedImage bufferedImage = ((ImageComponent2D)appearances[i].getTexture().getImage(0)).getImage();
            // stellen sie dar:
            previewLabels[i].setImage(bufferedImage);
          }
          // Andernfalls, falls nur eine Farbe zugewiesen wurde, so
          else if (appearances[i].getMaterial() != null) {
            // holen wir diese und
            Color3f color3f = new Color3f();
            appearances[i].getMaterial().getAmbientColor(color3f);
            Color color = new Color(color3f.x, color3f.y, color3f.z);
            // stellen sie dar:
            previewLabels[i].setColor(color);
          }
          // Ist die Appearance != null, wurde aber keinen Textur/Farbe gesetzt,
          // so zeigen wir dies an:
          else
            previewLabels[i].setText("leer");
        }
        // Ist appearances[i] == null ->
        else {
          // Anzeigen, da keine Appearance fr entsprechende Seite gesetzt wurde:
          previewLabels[i].setText("leer");
        }
      }
    }
    // Das Array appearances ist == null ->
    else {
      // berall anzeigen, da keine Textur/Farbe gesetzt wurde...
      for (int i = 0; i < previewLabels.length; i++) {
        previewLabels[i].setText("leer");
      }
    }
  }


  /**
   * Initialisierung diverser GUI-Einstellungen dieser Komponente. Diese Methode
   * wird automatisch vom Konstruktor dieser Klasse aufgerufen und sollte somit nicht
   * von Interesse fr den Endanwender sein.
   *
   * @see #AppearanceJPanel
   */
  protected void guiInit(String[] descriptions) {
    // LayoutManager
    GridBagLayout gbl = new GridBagLayout();
    GridBagConstraints gbc;
    this.setLayout(gbl);

    // Komponenten zur Darstellung der selektierten Textur/Farbe
    previewLabels = new PreviewJLabel[descriptions.length];

    for (int i = 0; i < descriptions.length; i++)
      previewLabels[i] = new PreviewJLabel("leer", PreviewJLabel.CENTER);

    // Komponenten einfgen

    int ypos = 0, xpos = 0;

    for (int pos = 1; pos < descriptions.length+1; pos++) {
      gbc = makeGBC(xpos++, ypos, 1, 1);
      gbc.weightx = 0;
      gbc.weighty = 0;
      gbc.anchor = GridBagConstraints.CENTER;
      gbc.fill = GridBagConstraints.BOTH;
      JPanel panel = new JPanel();
      panel.setLayout(new BorderLayout());
      gbl.setConstraints(panel, gbc);
      panel.setBorder(BorderFactory.createTitledBorder(descriptions[pos-1]));
      this.add(panel);

      panel.add(previewLabels[pos-1], BorderLayout.CENTER);

      ypos = (pos % 3 == 0) ? ++ypos:ypos;
      xpos = (xpos == 3) ? 0:xpos;
    }
  }

  /**
   * Hilfsmethode, um das GridBagLayout fr diese Methode zu verwenden.
   *
   * @param x       x-Anfangskoordinate im Gitternetz
   * @param y       y-Anfangskoordinate im Gitternetz
   * @param width   Erstreckung der Komponente in horizontaler Richtung auf dem Gitternetz
   * @param height  Erstreckung der KOmponente in vertikaler Richtung auf dem Gitternetz
   */
  private GridBagConstraints makeGBC(int x, int y, int width, int height) {
    GridBagConstraints gbc = new GridBagConstraints();
    gbc.gridx = x;
    gbc.gridy = y;
    gbc.gridwidth = width;
    gbc.gridheight = height;
    gbc.insets = new Insets(1, 1, 1, 1);
    return gbc;
  }
}