package pacman3d.labyrinth.cells;

import javax.media.j3d.*;
import javax.vecmath.*;
import com.sun.j3d.utils.geometry.Box;
import com.sun.j3d.utils.geometry.ColorCube;

import org.w3c.dom.*;

import pacman3d.util.*;
import pacman3d.labyrinth.*;

/**
 * <b>Titel:</b>Pacman 3D - Labyrinthzellen<p>
 *
 * <b>Beschreibung:</b><br>
 * <p>
 *
 * <b>Copyright:</b>	Copyright (c) 2001/02<br>
 *
 * @author              Labyrinth-Gruppe<br>
 * @version             1.0 03/15/2002<br>
 */
public class CellGlassWall extends Cell
{ // Der Java3D Knoten
  private TransformGroup view;
  // Die Position der Zelle
  private Tuple3i position;
  // Die Seiten die nicht gezeichnet werden
  private int sideMask = 0;
  // Die Seiten
  private Shape3D[] sides = new Shape3D[6];
  // Die Texturen
  private long[] texIds = new long[6];
  // Die Texturtypen
  private String[] m_sTextureTypeNames = new String[6];

  /** Direction for setting Texture and SideMask
   *  Right <=>    X Direction
   */
  static public final int SIDE_RIGHT =  1; //0

  /** Direction for setting Texture and SideMask
   *  Left  <=>  - X Direction
   */
  static public final int SIDE_LEFT  =  4; //2

  /** Direction for setting Texture and SideMask
   *  Front <=>  - Z Direction
   */
  static public final int SIDE_FRONT =  2; //1

  /** Direction for setting Texture and SideMask
   *  Back  <=>    Z Direction
   */
  static public final int SIDE_BACK  =  8; //3

  /** Direction for setting Texture and SideMask
   *  Up    <=>    Y Direction
   */
  static public final int SIDE_UP    = 16; //4

  /** Direction for setting Texture and SideMask
   *  Down  <=>  - Y Direction
   */
  static public final int SIDE_DOWN  = 32; //5

  /** All Sides
   */
  static public final int SIDES_ALL   = 63;

  /** All Sides without Up and Down
   */
  static public final int SIDES_LEVEL   = 15;


  /** Leerer Konstruktor
   *  initialisiert die Zelle
   */
  public CellGlassWall()
  { super(false);
    for(int i=0;i<6;i++)
      sides[i] = getSide();
    initParameterHandling();
  }

  /** Konstruktor
   *  @param pos Position der Zelle
   */
  public CellGlassWall(Tuple3i pos)
  { super(false);
    for(int i=0;i<6;i++)
      sides[i] = getSide();
    setPosition(pos);
    initParameterHandling();
  }

  /**
   * Initialisiert die Parameterliste der Klasse.
   */
  private void initParameterHandling()
  { for (int i=1; i<=6; i++) {
      m_oParameterNames.add( "texture" + i );
      m_oParameterTypes.add( new String() );
      m_oParameterDescriptions.add( "Typname von Textur #" +
        i );
    }
  }

  public void setParameter( String sName, Object oValue )
  { try {
      if (sName.equalsIgnoreCase( "texture1" )) {
        this.setTexture( 1, (String) oValue, 0 );
      } else if (sName.equalsIgnoreCase( "texture2" )) {
        this.setTexture( 2, (String) oValue, 0 );
      } else if (sName.equalsIgnoreCase( "texture3" )) {
        this.setTexture( 4, (String) oValue, 0 );
      } else if (sName.equalsIgnoreCase( "texture4" )) {
        this.setTexture( 8, (String) oValue, 0 );
      } else if (sName.equalsIgnoreCase( "texture5" )) {
        this.setTexture( 16, (String) oValue, 0 );
      } else if (sName.equalsIgnoreCase( "texture6" )) {
        this.setTexture( 32, (String) oValue, 0 );
      } else {
        super.setParameter( sName, oValue );
      }
    } catch( Exception oException ) {
        Debug.out( getClass().getName(), oException, Debug.LEVEL_WARNING );
    }
  }

  public Object getParameter( String sName )
  { if (sName.equalsIgnoreCase( "texture1" )) {
      return m_sTextureTypeNames[ 0 ];
    } else if (sName.equalsIgnoreCase( "texture2" )) {
      return m_sTextureTypeNames[ 1 ];
    } else if (sName.equalsIgnoreCase( "texture3" )) {
      return m_sTextureTypeNames[ 2 ];
    } else if (sName.equalsIgnoreCase( "texture4" )) {
      return m_sTextureTypeNames[ 3 ];
    } else if (sName.equalsIgnoreCase( "texture5" )) {
      return m_sTextureTypeNames[ 4 ];
    } else if (sName.equalsIgnoreCase( "texture6" )) {
      return m_sTextureTypeNames[ 5 ];
    } else {
      return super.getParameter( sName );
    }
  }


  /** Methode setzt die Position der Zelle
   *  @param pos Position der Zelle
   */
  public void setPosition(Tuple3i pos)
  { position = pos;
  }

  /**
   * Methode liefert die gesetzte Position der Zelle im Labyrinth.
   *
   * @return Position der Zelle im Labyrinth.
   */
  public Tuple3i getPosition() {
    return position;
  }

  /** Methode setzt die Seiten Maske der Zelle
   *  @param sideMask  Flags der Seiten die <b>nicht</b> erzeugt werden sollen
   *                   (Teil)Summe der folgenden Flags:
   *                   SIDE_RIGHT, SIDE_FRONT, SIDE_LEFT, SIDE_BACK
   */
  public void setSideMask(int sideMask) throws Exception
  { if(sideMask>15 || sideMask<0)
      throw new P3DException("Unerlaubte sideMask in CellWall.getJ3DGroup() angegeben");
    this.sideMask = sideMask;
  }

  /** Gibt den Java3D Knoten zurck
   *  @return Den Java3d Knoten mit der Ansicht der Zelle.
   */
  public javax.media.j3d.Node getJ3DGroup()
  { build();
    return view;
  }


  /** Erzeugt den Java3D Knoten
   */
  private void build()
  { Transform3D tr = new Transform3D();;
    if(null!=position)
      tr.setTranslation(new Vector3f(position.x,position.y,position.z));
    view = new TransformGroup(tr);

    AxisAngle4f[] rotationen = { new AxisAngle4f(0.0f,0.0f,0.0f,0.0f),
                                 new AxisAngle4f(0.0f,1.0f,0.0f,1.570796f),
                                 new AxisAngle4f(0.0f,1.0f,0.0f,3.141593f),
                                 new AxisAngle4f(0.0f,1.0f,0.0f,4.712389f),
                                 new AxisAngle4f(0.0f,0.0f,1.0f,1.570796f),
                                 new AxisAngle4f(0.0f,0.0f,1.0f,4.712389f)
                               };
    // Die Seiten oben und Untern werden immer erzeugt...
    String hs = Integer.toBinaryString(64+sideMask);
    tr = new Transform3D();
    TransformGroup trGr;
    Appearance appearance;
    for(int i=0;i<6;i++)
    { if(hs.substring(6-i,7-i).equals("0"))
      { tr.setRotation(rotationen[i]);
        trGr = new TransformGroup(tr);
        if(sides[i].getParent()!=null) {
          Shape3D shape = (Shape3D)sides[i].cloneNode(true);

          appearance = shape.getAppearance();

          if (appearance != null)
            appearance.setTransparencyAttributes(new TransparencyAttributes(TransparencyAttributes.FASTEST, 0.7f));

          trGr.addChild(shape);
        }
        else {
          appearance = sides[i].getAppearance();

          if (appearance != null)
            appearance.setTransparencyAttributes(new TransparencyAttributes(TransparencyAttributes.FASTEST, 0.7f));

          trGr.addChild(sides[i]);
        }
        view.addChild(trGr);
      }
    }
  }

  /** Setzt eine Textur auf eine Seite
   *  @param side               Flag der Seite(n), die Texturiert werden soll(en).
   *                            Fr mehrere Seiten Summe der Flags.
   *  @param sTextureType       Der Name des Texturtyps, wie er fuer den
   *                            Zugriff auf den <code>ResourceHandler</code>
   *                            benoetigt wird.
   *  @param texId              Die Textur-Id
   *  @see                      pacman3d.util.ResourceHandler
   */
  public void setTexture(int side, String sTextureName, long texId) throws Exception
  { if(side>64 || side<0)
      throw new P3DException("Unerlaubte Seite in CellWall.setTexture() angegeben");
    side = this.SIDES_ALL;
    // Textur vom ResourceHandler holen
    ResourceHandler oResourceHandler = ResourceHandler.getInstance();
    Texture tex = oResourceHandler.getTexture( sTextureName );

    String sNull = (tex == null) ? "texture " + sTextureName +
        " not available" : "texture " + sTextureName + " loaded";
    Debug.out (getClass().getName(), sNull, Debug.LEVEL_NOTICE);

    Appearance app = new Appearance();
    app.setCapability(Appearance.ALLOW_TEXTURE_READ);
    // Textur attribute
    TextureAttributes texAttr = new TextureAttributes();
    texAttr.setTextureMode(TextureAttributes.MODULATE);
    app.setTextureAttributes(texAttr);
    app.setTexture(tex);
    // Farbe und Material werden gesetzt
    app.setMaterial(new Material(white,black,white,white,10.0f));
    // Wenn einer von euch eine besser Idee hat, die binr darstellung einer Zahl
    // zu analysieren, nur raus mit der Sprache
    String hs = Integer.toBinaryString(64+side);
    for(int i=0;i<6;i++)
    { if(hs.substring(6-i,7-i).equals("1"))
      { sides[i].setAppearance(app);
        texIds[i] = texId;
        m_sTextureTypeNames[i] = sTextureName;
      }
    }
  }

  /** Setzt eine Farbe auf eine Seiten
   *  @param side   Flag der Seite(n), die Coloriert werden soll(en).
   *                Fr mehrere Seiten Summe der Flags.
   *  @param color  Die Farbe
   *  @param texId  Die Textur-Id
   */
  public void setColor(int side, Color3f color) throws Exception {
    side = this.SIDES_ALL;
    if(side>64 || side<0) throw new P3DException("Unerlaubte Seite in CellWall.setColor() angegeben");

    Appearance app = new Appearance();
    app.setCapability(Appearance.ALLOW_MATERIAL_READ);
    // Farbe und Material werden gesetzt
    Material material = new Material(color,black,white,white,10.0f);
    material.setCapability(Material.ALLOW_COMPONENT_READ);
    material.setLightingEnable(true);
    app.setMaterial(material);
    // Wenn einer von euch eine besser Idee hat, die binr darstellung einer Zahl
    // zu analysieren, nur raus mit der Sprache
    String hs = Integer.toBinaryString(64+side);
    for(int i=0;i<6;i++)
    { if(hs.substring(6-i,7-i).equals("1"))
        sides[i].setAppearance(app);
    }
  }

  /** Erzeugt eine Seite
   */
  private Shape3D getSide()
  { IndexedQuadArray side = new IndexedQuadArray(4,IndexedQuadArray.COORDINATES |
                                                   IndexedQuadArray.TEXTURE_COORDINATE_2 |
                                                   IndexedQuadArray.NORMALS
                                                 ,4);
    Point3f[] sideCoordinates = { new Point3f(0.5f,-0.5f,-0.5f),
                                  new Point3f(0.5f, 0.5f,-0.5f),
                                  new Point3f(0.5f, 0.5f, 0.5f),
                                  new Point3f(0.5f,-0.5f, 0.5f)
                                 };
    TexCoord2f[] texCoord = { new TexCoord2f(0.0f,0.0f),
                              new TexCoord2f(0.0f,1.0f),
                              new TexCoord2f(1.0f,1.0f),
                              new TexCoord2f(1.0f,0.0f)
                            };
    Vector3f[] normals = { new Vector3f(1.0f,0.0f,0.0f)
                         };
    int[] CoordIndices    = {0,1,2,3};
    int[] texIndices      = {3,2,1,0};
    int[] normalsIndicies = {0,0,0,0};
    side.setCoordinates(0,sideCoordinates);
    side.setCoordinateIndices(0,CoordIndices);
    side.setTextureCoordinates(0,0,texCoord);
    side.setTextureCoordinateIndices(0,0,texIndices);
    side.setNormals(0,normals);
    side.setNormalIndices(0,normalsIndicies);
    Shape3D shape = new Shape3D(side);
    shape.setAppearance(super.DUMMY_APPEARANCE);
    shape.setCapability(Shape3D.ALLOW_APPEARANCE_READ);

    //return new Shape3D(side);
    return shape;
  }

	/**
	 * liefert ein Array der Appearance-Objekte der sechs Seiten einer CellWall zurck
	 * @return ein sechselementiges Array der Appearance-Objekte
	 */
	public Appearance[] getAppearance() {
		Appearance[] appearance = new Appearance[6];
		for (int i = 0; i < 6; i++) {
		        appearance[i] = sides[i].getAppearance();
		}
		return appearance;
	}

	/**
	 * Nimmt ein XML Element entgegen, und deserialisiert den Zustand des
	 * eigenen Objektes.
	 *
	 * @param oElement      Das XML-Element, unterhalb dessen ein Objekt
	 *                      seinen Zustand speichern darf.
	 */
	public void loadFromDOM( org.w3c.dom.Element oElement ) {

		// zunchst die Elternmethode aufrufen und sie werkeln lassen
	        super.loadFromDOM(oElement);

		// wenn kein Elemenmt bergeben wurde, dann sofort zurckspringen
		if (oElement == null) { return; }

		// wir brauchen die Instanz des ResourcenHandlers zum Holen von Texturen
		ResourceHandler oResourceHandler = pacman3d.util.ResourceHandler.getInstance();

		// jetzt holen wir uns unsere Parameter fr die Texturen (entweder ist
		// das bergebene oElement selbst <params> oder ein Kind von ihm)
		Element elParams = null;
		if (oElement.getTagName().equals(LevelXmlSyntax.TAG_PARAMS)) {
			elParams = oElement;
		} else {
		        elParams = XmlUtils.getChildElement(oElement, LevelXmlSyntax.TAG_PARAMS);
		}
		if (elParams != null) {
			if (elParams.hasChildNodes()) {
				NodeList oChildren = elParams.getChildNodes();

				// gehe durch alle Parameter durch und lade Texturen ein
				for (int i=0; i < oChildren.getLength(); i++) {
					org.w3c.dom.Node ndCurrent = oChildren.item(i);
					if (ndCurrent.getNodeType() == org.w3c.dom.Node.ELEMENT_NODE) {
						if (((Element)ndCurrent).getTagName().equals("texture")) {

							// <texture side="1"><rsrclink xlink:type="simple" xlink:href="#res_wallX"/></texture>

							// job 1 ist das Holen der Seitennummer (iSide)
							int iSide = 0;
							try {
							        String sSide = ((Element)ndCurrent).getAttribute("side");
	        						if (sSide != null) { iSide = new Integer(sSide).intValue(); }
							} catch (java.lang.NumberFormatException ex) {
							        Debug.out (this.getClass().getName(), "keine Zahl vorgefunden", Debug.LEVEL_WARNING);
							}

							// job 2 ist das Holen des Texturnamens
							Element elResLink = XmlUtils.getChildElement((Element)ndCurrent, LevelXmlSyntax.TAG_RESOURCELINK);
							String sResName = "";
							if (elResLink != null) {
								// hole den Namen der Resourcen aus dem <rsrclink>-Element und schneide
								// das fhrende # des Namens ab (im <rscrlink>-Element ist der Name der
								// Resource im href-Attribut als ein XPointer-Verweis auf eine folgende
								// Resourcendefinition abgelegt. Die Syntax fr einen solchen lokalen
								// Verweis ist: '#' Name. Dementspreched mu das fhrende # abgeschnitten
								// werden, bevor der ResourceHandler die Anweisung zum Laden bekommt
								sResName = elResLink.getAttribute(LevelXmlSyntax.ATTR_RESOURCELINK_HREF);
								if (sResName.startsWith("#")) {
								        sResName = sResName.substring(1);
								}
								// Debug.out(this.getClass().getName(), "" + sResName);

								// lade die Textur vom ResourcenHandler
								// javax.media.j3d.Texture2D txLoaded = null;
								// txLoaded = oResourceHandler.getTexture(sResName);
								// String sNull = (txLoaded == null) ? "texture " + sResName + " not available" : "texture " + sResName + " loaded";
								// Debug.out (this.getClass().getName(), sNull, Debug.LEVEL_NOTICE);

								try {
									this.setTexture(iSide, sResName, 0);
								} catch (java.lang.Exception ex) {
									Debug.out (this.getClass().getName(), "error assigning texture", Debug.LEVEL_WARNING);
								}
							}
						}
					}
				}
			}
		}
	}

       /**
         * Serialisiert den Zustand der Instanz in ein XML-Element, hier insbesondere
	 * die Parameter zur Darstellung der Zelle (Texturen)
         *
	 * @param oDoc das Document, in das die Daten eingefgt werden sollen
	 * (wird fr das Erzeugen von Elementen nach dem Factory-Konzept bentigt)
         * @return Valide XML-Daten in String-Form.
         */
        public org.w3c.dom.Element saveToDOM(org.w3c.dom.Document oDoc) {

		// zunchst die Elternmethode aufrufen und sie werkeln lassen
	        Element oCell = super.saveToDOM(oDoc);

		// Einfgen der Parameter (hier mu eine Prfung rein, ob schon
		// ein <params>-Tag vorhanden ist
		Element elParams = oDoc.createElement(LevelXmlSyntax.TAG_PARAMS);

		// wir brauchen die Instanz des ResourcenHandlers zum Ablegen von Texturen
		ResourceHandler oResourceHandler = pacman3d.util.ResourceHandler.getInstance();

		// alle sechs Seiten durchlaufen
		for (int iSide = 0; iSide < 6; iSide++) {

			// die Adresse der Textur der aktuellen Seite holen{
			// String sTextureName = sides[iSide].getAppearance().getTexture().toString();
                        String sTextureName = oResourceHandler.addTexture( sides[iSide].getAppearance().getTexture() );
			// oResourceHandler.addTexture(sides[iSide].getAppearance().getTexture());

			// eine <texture side="."><rsrclink ... xlink:href=""/></texture> Struktur aufbauen
			Element elTexture = oDoc.createElement (LevelXmlSyntax.TAG_TEX);
			elTexture.setAttribute(LevelXmlSyntax.ATTR_TEX_SIDE, "" + iSide);
			Element elResourceLink = oDoc.createElement(LevelXmlSyntax.TAG_RESOURCELINK);
			elResourceLink.setAttribute("xlink:type", "simple");
			elResourceLink.setAttribute(LevelXmlSyntax.ATTR_RESOURCELINK_HREF, "#" + sTextureName);
			elTexture.appendChild(elResourceLink);
			elParams.appendChild(elTexture);

		}

		oCell.appendChild(elParams);

                return oCell;
        }

	/**
	 * klont das Objekt und liefert es zurck
	 * @return der Klon der Instanz mit dem gleichen Typ (aber per
	 * Definition als Objekt gecastet)
	 */
	public Object clone () {

	        return super.clone();

	}

	/**
	 * vergleicht die Zelle mit dem angegebenen Objekt und liefert true
	 * bei Gleichheit, sonst falsch.
	 * @param obj das mit sich selbst zu vergleichende Objekt
	 * @return true bei Gleichheit, sonst false
	 */
	public boolean equals (Object obj) {

		// die Elternklasse erstmal zum Vergleich rannlassen
	        if (!super.equals(obj)) { return false; }

		CellGlassWall oCellWall = (CellGlassWall)obj;
		Appearance[] aoAppearance = oCellWall.getAppearance();

		// alle sechs Seiten durchlaufen und Texturen berprfen
		for (int iSide = 0; iSide < 6; iSide++) {

                        if (aoAppearance[iSide] == null) {
                                if (sides[iSide].getAppearance() != null) {
                                        return false;
                                } else continue;
                        } else {
                                if (sides[iSide].getAppearance() == null) {
                                        return false;
                                }
                        }
                        if (aoAppearance[iSide].getTexture() == null) {
                                if (sides[iSide].getAppearance().getTexture() != null) {
                                        return false;
                                } else continue;
                        } else {
                                if (sides[iSide].getAppearance().getTexture() == null) {
                                        return false;
                                }
                        }

                        if (!aoAppearance[iSide].getTexture().equals(sides[iSide].getAppearance().getTexture())) {
                                return false;
                        }
		}

		return true;
	}

}