package pacman3d.labyrinth.cells;

import pacman3d.labyrinth.items.*;
import javax.vecmath.*;
import org.w3c.dom.*;
import pacman3d.util.*;
import java.util.Vector;


/**
 * <b>Titel:</b>Pacman 3D - Labyrinthzellen<p>
 *
 * <b>Beschreibung:</b><br>
 * Diese Klasse stellt die abstrakte Superklasse aller in einem Labyrinth
 * verwendbaren Zellentypen dar.
 * <p>
 *
 * <b>Copyright:</b>	Copyright (c) 2001/02<br>
 *
 * @author              Labyrinth-Gruppe<br>
 * @version             1.0 03/15/2002<br>
 */
public abstract class Cell implements pacman3d.util.XMLSerializeable,
	java.lang.Cloneable, pacman3d.util.Introspectable {

  /**
   * Interner Flag von Typ <b>boolean</b>, der angibt, ob eine Zelle begehbar
   * oder nicht begehbar ist. Trgt dieser Flag den Wert <b>true</b>, so ist
   * die durch diese Klasse bzw. Subklasse reprsentierte Zelle begehbar, andernfalls
   * nicht.
   */
  private boolean walkable = false;

  /**
   * Instanz der Klasse {@link java.util.Vector Vector}, welche die Referenzen
   * auf alle in einer Zelle befindlichen {@link pacman3d.labyrinth.items Items}
   * enthlt.
   */
  protected java.util.Vector items;

  /**
   * Instanz einer Subklasse der abstrakten Superklasse {@link javax.vecmath.Tuple3i
   * Tuple3i}. Diese Instanz ist initialisiert mit der Position im
   * Java3D-Koordinatensystem, welche eine durch eine Subklasse dieser Klasse
   * dargestellten Zelle hat.
   */
  private Tuple3i position;

  /** der Templatetyp der Zelle.*/
  protected String m_sTemplateType = "";

  /** Vector der Parameterwerte */
  protected java.util.Hashtable m_oParameterValues;

  /** Vector der Parameternamen */
  protected Vector m_oParameterNames;

  /** Vector der Parameterbeschreibungen */
  protected Vector m_oParameterDescriptions;

  /** Vector der Parametertypen */
  protected Vector m_oParameterTypes;

  /**
   * Grundlegende Farbe, um Appearance bzw. Material zu erzeugen.
   *
   * @see #white
   */
  static public javax.vecmath.Color3f black    = new javax.vecmath.Color3f(0.0f,0.0f,0.0f);

  /**
   * Grundlegende Farbe, um Appearance bzw. Material zu erzeugen.
   *
   * @see black
   */
  static public javax.vecmath.Color3f white    = new javax.vecmath.Color3f(1.0f,1.0f,1.0f);

  /**
   * Instanz der Klasse {@link javax.media.j3d.Appearance Appearance}, welche
   * allen Java3D-Knoten eines darzustellenden Zellentyps bei Erzeugung initial
   * zugewiesen wird
   */
  protected static final javax.media.j3d.Appearance DUMMY_APPEARANCE = new javax.media.j3d.Appearance();

  // Statischer Initialisierer, der weitere "Einstellungen" an der Instanz
  // "DUMMY_APPEARANCE" vornimmt.
  static {
    javax.media.j3d.Material dummy_material     = new javax.media.j3d.Material(white, black, white, white, 10.0f);
    dummy_material.setCapability(javax.media.j3d.Material.ALLOW_COMPONENT_READ);
    DUMMY_APPEARANCE.setCapability(javax.media.j3d.Appearance.ALLOW_MATERIAL_READ);
    DUMMY_APPEARANCE.setCapability(javax.media.j3d.Appearance.ALLOW_TEXTURE_READ);
    DUMMY_APPEARANCE.setMaterial(dummy_material);
  }

  /**
   * Boolscher Flag, der angibt, ob ein Labyrinth im "Multiplayermodus" oder
   * "Singleplayermodus" gespielt wird. Befindet sich der Spielzustand im
   * "Singleplayermodus", so trgt dieser Flag den Wert <b>true</b>, andernfalls
   * <b>false</b>.
   */
  private boolean m_bMultiplayerGame = false;

  /**
   * Boolscher Flag, der angibt, ob sich die Zelle in einem "Serverlabyrinth"
   * befindet oder nicht. Befindet sich die entsprechende Zelle in einem
   * "Serverlabyrinth", so trgt dieser Flag den Wert <b>true</b>, andernfalls
   * <b>false</b>
   */
  private boolean m_bWeAreServer = false;

  /**
   * Konstruktor:<br>
   *
   * Dieser Konstruktor instantiiert eine neue Instanz dieser Klasse. Da diese
   * Klasse allerdings die abstrakte Superklasse aller in einem Labyrinth
   * verwendbaren Zellentypen ist, sollte diese Klasse nicht eigenstndig
   * instantiiert werden. Die Instantiierung erfolgt von den verschiedenen
   * Subklassen dieser Klasse aus, die hierzu den boolschen Parameter
   * <b>walkable</b> bergeben. Trgt dieser den Wert <b>true</b>, so ist die
   * durch eine Subklasse dieser Klasse reprsentierte Zelle begehbar, andernfalls
   * nicht.
   *
   * @param walkable <b>true</b>, falls die Zelle begehbar sein soll, andernfalls
   *                 <b>false</b>.
   *
   * @see pacman3d.labyrinth.cells.CellArchwayConnection#CellArchwayConnection
   * @see pacman3d.labyrinth.cells.CellArchwayHorizontal#CellArchwayHorizontal
   * @see pacman3d.labyrinth.cells.CellArchwayVertical#CellArchwayVertical
   * @see pacman3d.labyrinth.cells.CellFloor#CellFloor
   * @see pacman3d.labyrinth.cells.CellGlassWall#CellGlassWall
   * @see pacman3d.labyrinth.cells.CellHalfWall#CellHalfWall
   * @see pacman3d.labyrinth.cells.CellLadderEast#CellLadderEast
   * @see pacman3d.labyrinth.cells.CellLadderNorth#CellLadderNorth
   * @see pacman3d.labyrinth.cells.CellLadderSouth#CellLadderSouth
   * @see pacman3d.labyrinth.cells.CellLadderWest#CellLadderWest
   * @see pacman3d.labyrinth.cells.CellSimpleCone#CellSimpleCone
   * @see pacman3d.labyrinth.cells.CellSimpleDoorWest#CellSimpleDoorWest
   * @see pacman3d.labyrinth.cells.CellSimpleDoorNorth#CellSimpleDoorNorth
   * @see pacman3d.labyrinth.cells.CellSimpleDoorEast#CellSimpleDoorEast
   * @see pacman3d.labyrinth.cells.CellSimpleDoorSouth#CellSimpleDoorSouth
   * @see pacman3d.labyrinth.cells.CellSimplePillar#CellSimplePillar
   * @see pacman3d.labyrinth.cells.CellSimpleWindowHorizontal#CellSimpleDoorHorizontal
   * @see pacman3d.labyrinth.cells.CellSimpleWindowVertical#CellSimpleDoorVertical
   * @see pacman3d.labyrinth.cells.CellWall#CellWall
   */
  public Cell(boolean walkable)
  { this.walkable = walkable;
    this.m_oParameterDescriptions = new Vector();
    this.m_oParameterNames = new Vector();
    this.m_oParameterTypes = new Vector();
    this.m_oParameterValues = new java.util.Hashtable();
    items = new java.util.Vector();
  }

  public void setParameter( String sName, Object oValue )
  {
  }

  public Object getParameter( String sName )
  { return null;
  }

  public int getParameterCount ()
  { return m_oParameterNames.size();
  }

  public String getParameterName( int iPos )
  { return (String) m_oParameterNames.get( iPos );
  }

  public String getParameterDescription( int iPos )
  { return (String) m_oParameterDescriptions.get( iPos );
  }

  public Object getParameterType( int iPos )
  { return m_oParameterTypes.get( iPos );
  }

  /**
   * wird nach dem Initialisieren der Zelle aufgerufen und nimmt den Status
   * Multiplayer/Singleplayer und Server/Client entgegen. Der Status ist abhngig
   * vom Labyrinth, in dem die Zelle platziert wird.
   *
   * @param bMultiplayerGame true, wenn Multiplayer-Spiel, false wenn Singleplayer
   * @param bWeAreServer true, wenn diese Applikation der Server ist, false andernfalls
   */
  public void setGameMode (boolean bMultiplayerGame, boolean bWeAreServer) {
	m_bMultiplayerGame = bMultiplayerGame;
	m_bWeAreServer = bWeAreServer;
  }

  /**
   * Mittles dieser Methode kann die Position einer Zelle im
   * Java3D-Koordinatensystem gesetzt werden.
   *
   * @param pos Position der Zelle im Java3D-Koordinatensystem.
   */
  public void setPosition(Tuple3i pos)
  { position = pos;
  }

  /**
   * Diese Methode liefert die gesetzte Position der Zelle im
   * Java3D-Koordinatensystem.
   *
   * @return Position der Zelle im Java3D-Koordinatensystem.
   */
  public Tuple3i getPosition() {
    return position;
  }

  /**
   * Mittels dieser Methode kann abgefragt werden, ob die gegebene Zelle
   * - reprsentiert durch eine Instanz von <b>Cell</b> - begehbar ist oder
   * nicht.
   *
   * @return  <b>true</b>, falls die Zelle begehbar ist, andernfalls
   *	      <b>false</b>.
   */
  public boolean isWalkable()
  { return walkable;
  }

  /**
   * Mittels dieser Methode kann abgefragt werden, ob die gegebene Zelle
   * {@link pacman3d.labyrinth.items.Items Item} enthlt oder nicht.
   *
   * @return  <b>true</b>, falls die gegebene Zelle mindestens ein
   *          {@link pacman3d.labyrinth.items.Item Item} enthlt, andernfalls
   *	      <b>false</b>.
   */
  public boolean hasItems()
  { return (getItemsCount()!=0);
  }

  /**
   * Diese Methode liefert alle Referenzen auf die in der gegebenen Zelle
   * befindlichen {@link pacman3d.labyrinth.items.Item Items} zurck.
   *
   * @return  Array vom Typ {@link pacman3d.labyrinth.items.Item Item}, mit Referenzen
   *          auf alle der in der gegebenen Zelle befindlichen
   *          {@link pacman3d.labyrinth.items.Item Item}.
   */
  public Item[] getItems()
  { Item[] i = new Item[getItemsCount()];
    for(int n=0;n<getItemsCount();n++)
      i[n]=(Item)items.elementAt(n);
    return i;
  }

  /**
   * Diese Methode liefert eine Referenz auf eine Instanz der Klasse
   * {@link java.util.Vector Vector}. Diese enthlt alle Referenzen der in einer
   * Zelle befindlichen {@link pacman3d.labyrinth.items.Item Items}.
   *
   * @return  Referenz auf Instanz der Klasse {@link java.util.Vector Vector},
   *          welche alle Referenzen auf die in einer Zelle befindlichen
   *          {@link pacman3d.labyrinth.items.Item Items} enthlt.
   */
  public Vector getItemsVector()
  {
    return items;
  }

  /**
   * Hilfsmethode, die den Vector {@link #items} klont. Diese Methode wird
   * vom {@link pacman3d.util.CellHanlder CellHandler} bentigt und ist somit
   * nicht von weiterem Interesse fr den Endanwender.
   */
  public void cloneItems()
  {
	Vector vecItemsCloned = new Vector();
	int iCount = items.size();
	for (int i=0; i<iCount; i++) {
		Item oItem = (Item) ((Item) items.get( i )).clone();
		pacman3d.message.MessageService.getInstance().addMessageListener(
			oItem );
		vecItemsCloned.add( oItem );
	}
	this.items = vecItemsCloned;
  }

  /**
   * Diese Methode fgt der Zelle ein neues {@link pacman3d.labyrinth.items.Item
   * Item} hinzu.
   *
   * @param  item Referenz auf dasjenige {@link pacman3d.labyrinth.items.Item Item},
   *              welches der Zelle hinzugefgt werden soll.
   */
  public void addItem(Item item)
  { if(item != null)
    { items.addElement(item);
    }
  }

  /**
   * Diese Methode fgt der Zelle ein {@link pacman3d.labyrinth.items.Item Item}
   * hinzu, welches bereits im {@link pacman3d.util.ItemHandler ItemHandler}
   * hinterlegt ist. Dazu ist der Name als Instanz der Klasse {@link
   * java.lang.String String} zu bergeben. Dieser Name ist vom {@link
   * pacman3d.util.ItemHandler ItemHandler} zuvor zu erfragen.
   *
   * @param  sItemName  Der Name desjenigen [@link pacman3d.labyritnh.items.Item
   *                    Items}, welches der Zelle hinzugefuegt werden soll.
   *
   * @see pacman3d.util.ItemHandler
   */
  public void addItem( String sItemName )
  { items.addElement( pacman3d.util.ItemHandler.getInstance().getItemInstance(
      sItemName ) );
  }

  /**
   * Diese Methode liefert die Anzahl der in der gegebenen Zelle befindlichen
   * {@link pacman3d.labyrinth.items.Item Items} zurck.
   *
   * @return  Anzahl der in der gegebenen Zelle befindlichen {@link
   *          pacman3d.labyrinth.items.Item Items}.
   */
  public int getItemsCount()
  {
	return items.size();
  }

  /**
   * Liefert die Anzahl der in der Zelle befindlichen Score-Items
   * zurck (solche Items, die Punkte bringen und bei deren vlliger Abgrasung
   * das Spielziel erreicht ist)
   *
   * @return Anzahl der in der Spielfeldzelle enthaltenen Score-Items
   */
  public long getScoreItemsCount()
  {
	long lScoreItemsCount = 0;
	for (int i = 0; i < items.size(); i++) {
		if (((Item)(items.get(i))).isCoreItem()) {
			lScoreItemsCount++;
		}
	}
	return lScoreItemsCount;
  }

  /**
   * Diese Methode speichert eine Referenz auf das {@link
   * pacman3d.labyrinth.Labyrinth Labyrinth}, in dem sich diese Zelle und
   * vor allem ihre {@link #items Items} befinden (fr die Cell->Labyrinth und
   * Item->Labyrinth- Kommunikation, die bislang in diese Richtung nur in passiver
   * Form mglich war)
   *
   * @param oLabyrinth Referenz auf diejenige Instanz der Klasse
   *                   {@link pacman3d.labyrinth.Labyrinth Labyrinth}, in welcher
   *                   sich die Zelle befindet.
   */
  public void setLabyrinth(pacman3d.labyrinth.Labyrinth oLabyrinth)
  {
	// eine eigene Speicherung ist bislang noch nicht ntig:
	// m_oLabyrinth = oLabyrinth;

	// in den Items mu die Referenz aber vorhandensein
	for (int i = 0; i < items.size(); i++) {
		((Item)(items.get(i))).setLabyrinth(oLabyrinth);
	}
  }

  /**
   * Diese Methode liefert eine Referenz auf diejenige <b>Item</b>-Instanz, welche
   * in der gegebenen Spielfeldzelle als <b>itemNr</b>tes <b>Item</b> enthalten ist.
   *
   * @param itemNr  Nummer desjenigen <b>Item</b>, auf welches von der gegebenen
   *		    Spielfeldzelle eine Referenz zurckgeliefert werden soll.
   *
   * @return        Referenz auf Instanz des <b>itemNr</b>ten Item in der gegebenen
   *                Spielfeldzelle.
   */
  public Item getItem(int itemNr)
  { return (Item)items.elementAt(itemNr);
  }

  /**
   * Diese Methode lscht all diejenigen <b>Items</b>-Instanzen, welche in der gegebenen
   * Spielfeldzelle enthalten sind.
   */
  public void removeAllItems() {
    items.removeAllElements();
  }

  /**
   * Diese Methode lscht die dieser Methode bergebenen <b>Item</b>-Instanz aus der
   * gegebenen Spielfeldzelle, insofern diese natrlich in dieser enthalten ist.
   *
   * @param item  Zu lschende <b>Item</b>-Instanz.
   */
  public void removeItem(Item item) {
    items.remove(item);
  }

  /**
   * Diese Methode lscht diejenige <b>Item</b>-Instanz, welche in der gegebenen Spielfeldzelle
   * als <b>itemNr</b>tes <b>Item</b> enthalten ist.
   *
   * @param itemNr  Nummer desjenigen <b>Item</b>, welches in der gegebenen Spielfeldzelle
   *                gelscht werden soll.
   */
  public void removeItem(int itemNr) {
    items.remove(itemNr);
  }

  /**
   * Liefert den <b>Java3D-Node</b> zurck, der die Spielfeldzelle darstellt.
   * Ein Objekt vom Typ <i>TransformGroup</i>.
   *
   * @return ein Java3D-Node-Objekt, das zum Rendern der Zelle in der Szene bentigt wird.
   */
  public javax.media.j3d.Node getJ3DGroup()
  { // Funktionalitt in Subklassen definiert!
    return null;
  }

	/**
	 * Nimmt ein XML Element entgegen, und deserialisiert den Zustand des
	 * eigenen Objektes.
	 *
	 * @param oElement      Das XML-Element, unterhalb dessen ein Objekt
	 *                      seinen Zustand speichern darf.
	 */
	public void loadFromDOM( org.w3c.dom.Element oElement ) {

		if (oElement == null) {
			// wenn wir kein Element als Parameter bekommen, dann ist hier nichts zu tun
			return;
		}

		// wir brauchen die Instanz des ItemHandlers zum Holen von Items (wofr sonst :)
		ItemHandler oItemHandler = pacman3d.util.ItemHandler.getInstance();

		// Einlesen von Itemdaten (wenn Server- oder Singleplayermode)
		if ((!m_bMultiplayerGame) || (m_bWeAreServer)) {

			Element elItems = XmlUtils.getChildElement(oElement, LevelXmlSyntax.TAG_ITEMS);
			if (elItems != null) {
				if (elItems.hasChildNodes()) {
					NodeList oChildren = elItems.getChildNodes();
					// Debug.out("cell", "anzahl items in aktueller Zelle = "+oChildren.getLength());

					// gehe durch alle items durch und Erzeuge Instanzen
					for (int i=0; i < oChildren.getLength(); i++) {
						Node ndCurrent = oChildren.item(i);
						if (ndCurrent.getNodeType() == Node.ELEMENT_NODE) {
							if (((Element)ndCurrent).getTagName().equals(LevelXmlSyntax.TAG_ITEM)) {

								// hole itemtype-bezeichenr
								String sItemType = ((Element)ndCurrent).getAttribute(LevelXmlSyntax.ATTR_TYPE);

								// Hole Instanz diesen Typs vom ItemHandler
								if (!sItemType.equals("")) {
									Item oCurrentItem = oItemHandler.getItemInstance(sItemType);
									if (oCurrentItem != null) {
										this.addItem((Item)oCurrentItem);
									} else {
										Debug.out (this.getClass().getName(), "could not instantiate item of type: " + sItemType, Debug.LEVEL_WARNING);
									}
								}
							}
						}
					}
				}
			}
		}

		//Debug.out ("LLASLLSSL", "" + this.getItemsCount());

		// Einlesen von Parameterdaten
		Element elParams = XmlUtils.getChildElement(oElement, LevelXmlSyntax.TAG_PARAMS);

		return;
	}

       /**
	 * Serialisiert den Zustand der Instanz in ein XML-Element.
	 *
	 * @param oDoc das Document, in das die Daten eingefgt werden sollen
	 * (wird fr das Erzeugen von Elementen nach dem Factory-Konzept bentigt)
	 * @return Valide XML-Daten in String-Form.
	 */
	public org.w3c.dom.Element saveToDOM(org.w3c.dom.Document oDoc) {

		// Zellelement erzeugen (<cell>)
		Element oCell = oDoc.createElement(LevelXmlSyntax.TAG_CELL);
		oCell.setAttribute(LevelXmlSyntax.ATTR_JAVACLASS, this.getClass().getName());

		// Speichern der Items

		// sind Items vorhanden?
		int iItemsCount = items.size();
		if (iItemsCount > 0) {

			// ein Unterelement fr die Items erzeugen und hinzufuegen
			Element oItemsElement = oDoc.createElement(
				LevelXmlSyntax.TAG_ITEMS ) ;
			oCell.appendChild( oItemsElement );

			// durch alle Items iterieren
			for (int i=0; i<iItemsCount; i++) {
				// i-tes Item laden
				Item oItem = (Item) items.get( i );
				// Item dem ItemHandler hinzufuegen
				String sItemName = ItemHandler.getInstance().addItem( oItem );

				// ein neues Element fuer das Item erzeugen und hinzufuegen
				Element oItemElement = oDoc.createElement(
					LevelXmlSyntax.TAG_ITEM );
				oItemsElement.appendChild( oItemElement );

				// ein neues Attribut fr die Itemklasse
				// erzeugen und hinzufgen
				oItemElement.setAttribute(
					LevelXmlSyntax.ATTR_TYPE, sItemName );

				// eine eindeutige ID fuer das Item waere zwar
				// noch denkbar, bringt aber im Moment keinen
				// konkreten Nutzen.
			}
		}

		return oCell;
	}

	/**
	 * klont das Objekt und liefert es zurck
	 * @return der Klon der Instanz mit dem gleichen Typ (aber per
	 * Definition als Objekt gecastet)
	 */
	public Object clone () {

		try {
			Object obj = super.clone();
			Cell oCell = (Cell)obj;
			oCell.cloneItems();

			return (Object)oCell;
		} catch (java.lang.CloneNotSupportedException ex) {
			Debug.out (this.getClass().getName(), ex, Debug.LEVEL_CRITICAL);
		}

		return null;
	}

	/**
	 * vergleicht die Zelle mit dem angegebenen Objekt und liefert true
	 * bei Gleichheit, sonst falsch.
	 * @param obj das mit sich selbst zu vergleichende Objekt
	 * @return true bei Gleichheit, sonst false
	 */
	public boolean equals (Object obj) {

		// sind die Klassen gleich? die erste wichtige Eigenschaft!
		if (!obj.getClass().getName().equals(this.getClass().getName())) {
			return false;
		}

		// Liste der Items vergleichen, dazu zunchst items von obj holen
		Cell oCell = (Cell)obj;
		Vector vecObjItems = oCell.getItemsVector();
		if (vecObjItems.size() != this.items.size()) {
			return false;
		}

		// detailiertes Vergleichen mit eigenen Items
		for (int i = 0; i < items.size(); i++) {
			Item oCurrentItem = (Item)items.get(i);
			if (!vecObjItems.contains(oCurrentItem)) {
				return false;
			}
		}

		return true;
	}

   /** Setzt eine Textur auf eine Seite
   *  @param side               Flag der Seite(n), die Texturiert werden soll(en).
   *                            Fr mehrere Seiten Summe der Flags.
   *  @param sTextureType       Der Name des Texturtyps, wie er fuer den
   *                            Zugriff auf den <code>ResourceHandler</code>
   *                            benoetigt wird.
   *  @param texId              Die Textur-Id
   *  @see                      pacman3d.util.ResourceHandler
   */
  public abstract void setTexture(int side, String sTextureName, long texId) throws Exception;
  public abstract javax.media.j3d.Appearance[] getAppearance();
  public abstract void setColor(int side, javax.vecmath.Color3f color3f) throws Exception;
}