package pacman3d.labyrinth;

import pacman3d.Game;
import pacman3d.message.*;
import pacman3d.monster.*;
import pacman3d.pacman.*;
import pacman3d.util.*;
import pacman3d.camera.*;
import pacman3d.labyrinth.cells.*;

import javax.media.j3d.*;
import javax.vecmath.*;
import com.sun.j3d.utils.image.TextureLoader;

import org.w3c.dom.*;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.DocumentBuilder;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import java.awt.*;
import java.awt.image.*;
import java.net.URL;
import java.io.*;
import java.util.Vector;


/**
 * <b>Title:</b>      	Pacman 3D - Nicht wirklich ;)</br>
 * <b>Description:</b><br>
 *
 * Das Labyrinth.<br>
 *
 * <b>Copyright:</b>	Copyright (c) 2001<br>
 *
 * @author              Labyrinth-Gruppe<br>
 * @version             $Version$ $Date: 2002/03/27 09:12:43 $<br>
 */
public class Labyrinth
{

  /* Array nimmt die Zellinstanzen auf (3D-Orientierung im Array ist: Y,X,Z) */
  private Cell[][][] labyrinth;
  private String[][][] m_asCellTypes;

  /* Container fr die Monster Positionen */
  private LabyrinthContainer positionMonsters = new LabyrinthContainer("javax.vecmath.Point3i");

  /* Container fr die Monster */
  private LabyrinthContainer monsters = new LabyrinthContainer("pacman3d.monster.Monster");

  /* Container fr die Pacman Positionen */
  private LabyrinthContainer positionPacman = new LabyrinthContainer("javax.vecmath.Point3i");

  /* Container fr die Pacman */
  private LabyrinthContainer pacman = new LabyrinthContainer("pacman3d.pacman.Pacman");

  /* Container fr die Pacman-Branchgroups */
  private LabyrinthContainer pacmanBranchGroups = new LabyrinthContainer("javax.media.j3d.BranchGroup");

  /* Flag, ob im Level Gravitation vorliegt oder nicht */
  private boolean m_bGravity = true;

  /* Ohne Kommentar */
  private Switch[] floors; // vorlufig fr den Test

  /* die ID des activen Pacman-Objekts, also des Pacmans, der den lokalen
  Spieler reprsentiert */
  private ID m_idActivePacman;

  /* die LevelMetaInfo des aktuell geladenen Levels */
  private LevelMetaInfo m_oLevelMetaInfo = null;

  /* die Abmessungen des Labyrinths in Zellenanzahlen (also diskrete Werte) */
  private Point3i m_p3iConstraints;

  /* eine Instanz auf ein Overview-Objekt zur Visualisierung einer bersichtskarte.
  Ist dieses Variable ungleich null wird bei Vernderungen am Labyrinth die
  repaint-Methoden der overview-Objekts aufgerufen (das ein Nachfahre von
  java.awt.Canvas ist */
  private Overview m_oOverview;

  /* das Fenster, in dem das Labyrinth angezeigt wird */
  private GameFrame m_oGameFrame;

  /* das Spielobjekt, das den Rahmen fr Pacman3d bildet und Labyrinth instanziert */
  private Game m_oGame;

  /* ContentBranch */
  private BranchGroup contentBranch;

  /* eine Referenz auf das aktive Camera-Objekt */
  private Camera m_oCamera;

  /* Flag gibt an, ob Wireframes oder echte Zellen gezeichnet werden sollen */
  private boolean m_bWireFrameMode = false;

  /* dieses Objekt nimmt die Liste der Startpositionen und LineOfSights fr
  Pacmans auf (siehe dazu auch der Zeiger iNextPacmanStartPosition) */
  private Vector m_vecPacmanStartPositions = null;

  /* Zeiger auf die nchste Startposition, die bei einer Anfrage an
  die Methode getNextPacmanStartPosition() geliefert werden soll. */
  private int m_iNextPacmanStartPosition = 0;

  /* dieses Objekt nimmt die Liste der Startpositionen fr die Monster
  auf (siehe dazu auch der Zeiger iNextMonsterStartPosition) */
  private Vector m_vecMonsterStartPositions = null;

  /* Zeiger auf die nchste Startposition, die bei einer Anfrage an
  die Methode getNextMonsterStartPosition() geliefert werden soll. */
  private int m_iNextMonsterStartPosition = 0;

  /* Flag, ob Multiplayermodus (true) oder Singleplayermodus (false) */
  private boolean m_bMultiplayerGame = false;

  /* Anzahl der Spieler (mal sehen, fr was wir das noch brauchen knnen) */
  private int m_iNumberOfPlayers = 0;

  /* Flag, ob wir Server sind oder nicht */
  private boolean m_bWeAreServer = false;

  /* Flag, das true ist, wenn Applikation im Demomodus, false im normalen
  Spielmodus (Singleplayer wie Multiplayer) */
  private boolean m_bDemoMode = false;

  /* Anzahl von (S)core-Items im Level (sind alle dieser Score-Items von Pacmans
  eingesammelt, dann ist das Ziel erreicht und das Spiel beendet) */
  private long m_lScoreItemsMax = 0;

  /* Anzahl der bislang eingesammelten (S)core-Items im Level */
  private long m_lScoreItemsCollected = 0;



  /**
   *  Konstruktor:
   *  erzeugt ein neues, leeres Labyrinthobjekt
   */
  public Labyrinth() {

  }

  /**
   *  Konstruktor:
   *  Instantiierung eines neuen Labyrinths mit x*y*z Cell-Instanzen.
   *  Alle Cell-Instanzen werden auf "nicht begehbar" gesetzt.
   */
  public Labyrinth(int x, int y, int z) {

    /* Da in Java3D die y-Achse die "Hhenachse" darstellt, "kommt y an die
    erste Position", so da das Cell-Array mit Durchlauf der y-Werte von 0 bis
    y alle Levelebenen vertical durchluft. */
    labyrinth = new Cell[y][x][z];
    m_asCellTypes = new String[y][x][z];

    /* Cell-Array mit Cell-Instanzen fllen */
    for (int yTmp = 0; yTmp < y; yTmp++)
      for (int xTmp = 0; xTmp < x; xTmp++)
	for (int zTmp = 0; zTmp < z; zTmp++) {
	  labyrinth[yTmp][xTmp][zTmp] = new CellFloor(new Point3i(xTmp, yTmp, zTmp));
	  m_asCellTypes[yTmp][xTmp][zTmp] = "";
	}

    /* die Abmessungen in der Instanzvariable setzen */
    m_p3iConstraints = new Point3i(labyrinth[1].length, labyrinth.length, labyrinth[1][1].length);

  }

  /**
   *  Methode um eine Zelle im Labyrinth auszutauschen...
   */

  /**
   *  Setzt den Zellentyp fuer eine bestimmte Koordinate im Labyrinth. Der
   *  entsprechende Zellentyp muss zuvor beim <code>CellHandler</code> bekannt
   *  gemacht worden sein.
   *  @param sCellType  Der Name des Zelltyps. Muss zuvor beim <code>CellHandler</code>
   *                    bekannt gemacht worden sein.
   *  @param position   Die Koordinate, auf die eine entsprechende Cell-Instanz
   *                    gesetzt werden soll.
   *  @see pacman3d.util.CellHandler
   */
  public void setCell(String sCellType, Point3i position) {
    if (labyrinth != null) {
      Cell cell = CellHandler.getInstance().getCellInstance(sCellType);
      cell.setPosition(position);
      labyrinth[position.y][position.x][position.z] = cell;
	//CellHandler.getInstance().getCellInstance( sCellType );
    }
      m_asCellTypes[position.y][position.x][position.z] = sCellType;

  }

  /**
   * Diese Methode liefert zum bergebenen Parameter <b>position</b>
   * eine Referenz auf die dort im Java3D-Koordinatensystem befindliche
   * Spielfeldzelle.
   *
   * @param   position  Punkt im Java3D-Koordinatensystem, von dessen Position
   *                    eine Referenz auf die dort befindliche Spielfeldzelle
   *                    erwnscht wird. (super formuliert ;)
   *
   * @return  Referenz auf die Spielfeldzelle an der Position <b>position</b>.
   *
   * @see     pacman3d.labyrinth.Labyrinth#getCell
   */
  public Cell getCell(Point3i position) {
    return getCell(position.x,position.y,position.z);
  }

  /**
   * Diese Methode liefert zu der bergebenen <b>x</b>-, <b>y</b>- und
   * <b>z</b>-Position eine Referenz auf die dort im Java3D-Koordinatensystem
   * befindliche Spielfeldzelle.
   *
   * @param x   x-Position
   * @param y   y-Position
   * @param z   z-Position
   *
   * @return    Referenz auf Spielfeldzelle an der Position (x,y,z)
   *
   * @see	pacman3d.labyrinth.Labyrinth#getCell(Point3i)
   */
  public Cell getCell(int x, int y, int z) {
    if (labyrinth != null &&
	0 <= x && x < this.m_p3iConstraints.x &&
	0 <= y && y < this.m_p3iConstraints.y &&
	0 <= z && z < this.m_p3iConstraints.z)
      return labyrinth[y][x][z];
    else return new CellWall(new Point3i(x, y, z));
  }

  /**
   * Diese Methode teilt dem Labyrinth die Pacman-Referenz <b>pacman</b>
   * als neuen Spieler unter der Bezeichnung <b>id</b> mit.
   *
   * @param id      Pacmanbezeichnung
   * @param pacman  Referenz auf Pacman-Instanz, welche dem Labyrinth
   *                bergeben werden soll.
   *
   * @see           pacman3d.labyrinth.Labyrinth#getPacman
   */
  public void setPacman(ID id, Pacman pacman)
  { try
    { this.pacman.add(id,pacman);
    }
    catch(Exception e)
    { // kann nicht auftreten!!!
      System.out.println ("error:" + e.getMessage());
    }
  }

  /**
   * Diese Methode liefert eine Referenz auf diejenige Pacman-Instanz,
   * welche im Labyrinth unter der Bezeichnung <b>id</b> gefhrt wird.
   *
   * @param   id  Bezeichnung des Pacman, auf dessen Instanz eine Referenz
   *		  erwnscht ist.
   *
   * @return      Referenz auf Instanz des Pacman mit der Bezeichnung <b>id</b>.
   *
   * @see         pacman3d.labyrinth.Labyrinth#setPacman
   */
  public Pacman getPacman(ID id)
  { return (Pacman)pacman.get(id);
  }

  /**
   * Diese Methode teilt dem Labyrinth die Branchgroup des Pacman mit,
   * in der er gerendert wird (primr fr Schatten-pacmans von Interesse)
   *
   * @param id        Pacmanbezeichnung
   * @param bgPacman  Pacman-Branchgroup
   *
   * @see           pacman3d.labyrinth.Labyrinth#getPacman
   */
  public void setPacmanBranchGroup(ID id, BranchGroup bgPacman)
  { try
    { this.pacmanBranchGroups.add(id,bgPacman);
    }
    catch(Exception e)
    { // kann nicht auftreten!!!
      System.out.println ("error:" + e.getMessage());
    }
  }

  /**
   * Diese Methode liefert die Branchgroup des Pacman mit der angegebenen
   * <b>id</b> zurck
   *
   * @param   id  Bezeichnung des Pacman, von dessen Instanz die BranchGroup
   *		  erwnscht ist.
   *
   * @return      Referenz auf die BranchGroup des Pacman mit der Bezeichnung <b>id</b>.
   *
   * @see         pacman3d.labyrinth.Labyrinth#setPacman
   */
  public BranchGroup getPacmanBranchGroup(ID id)
  { return (BranchGroup)pacmanBranchGroups.get(id);
  }

  /**
   * Speichert die bergebene <b>id</b> als die <b>id</b> des aktiven Pacman,
   * also des Pacman-Objekts, das den lokalen Spieler reprsentiert.
   *
   * @param id die <b>id</b> des neuen aktiven Pacman
   */
  public void setActivePacmanID (ID id) {
	m_idActivePacman = id;
  }

  /**
   * Liefert die <b>id</b> des als aktiv markierten Pacmans aus einer
   * Instanzvariable zurck.
   *
   * @return die <b>id</b> des aktiv markierten Pacman
   */
  public ID getActivePacmanID () {
	return m_idActivePacman;
  }

  /**
   * Diese Methode teilt dem Labyrinth die Monster-Referenz <b>monster</b>
   * als neues Monster unter der Bezeichnung <b>id</b> mit.
   *
   * @param id      Monsterbezeichnung
   * @param pacman  Referenz auf Monster-Instanz, welche dem Labyrinth
   *                bergeben werden soll.
   *
   * @see           pacman3d.labyrinth.Labyrinth#getMonster
   */
  public void setMonster(ID id, Monster monster)
  { try
    { monsters.add(id,monster);
    }
    catch(Exception e)
    { // kann nicht auftreten!!!
      System.out.println ("error:" + e.getMessage());
    }
  }

  /**
   * Diese Methode liefert eine Referenz auf diejenige Monster-Instanz,
   * welche im Labyrinth unter der Bezeichnung <b>id</b> gefhrt wird.
   *
   * @param   id  Bezeichnung des Monsters, auf dessen Instanz eine Referenz
   *              erwnscht ist.
   *
   * @return      Referenz auf Instanz des Monsters mit der Bezeichnung <b>id</b>.
   *
   * @see         pacman3d.labyrinth.Labyrinth#setMonster
   */
  public Monster getMonster(ID id)
  { return (Monster)monsters.get(id);
  }

  /**
   * Diese Methode teilt dem Labyrinth die neue Position des Pacman
   * mit der Bezeichnung <b>id</b> mit. Dabei ist die Positionsangabe
   * im Koordinatensystem von Java3D vorzunehmen.
   *
   * @param id        Pacmanbezeichnung
   * @param position  Neue Position des Pacman mit der Bezeichnung <b>id</b>
   *                  im Labyrinth.
   *
   * @see             pacman3d.labyrinth.Labyrinth#getPacmanPos
   */
  public void setPacmanPos(ID id, Point3i position) throws P3DException
  {
    /* berprfen, ob angegebene Position valide, dann setzen */
    checkPosition(position);

    /* wenn die id noch nicht im Position-Container enthalten, dann wird sie
    hinzugefgt, ansonsten mit dem neuen Wert geupdatet */
    positionPacman.add(id,position);

    /* positionsvernderung im Overview */
    if(null!=m_oOverview) m_oOverview.repaint();
  }

  /**
   * Diese Methode teilt dem Labyrinth die neue Position des Pacman
   * mit der Bezeichnung <b>id</b> mit. Dabei ist die Positionsangabe
   * im Koordinatensystem von Java3D vorzunehmen.
   *
   * @param id  Pacmanbezeichnung
   * @param x   x-Position
   * @param y   y-Position
   * @param z   z-Position
   *
   * @see	pacman3d.labyrinth.Labyrinth#getPacmanPos
   */
  public void setPacmanPos(ID id, int x, int y, int z) throws P3DException
  { setPacmanPos(id,new Point3i(x,y,z));
  }

  /**
   * Diese Methode liefert die Position des Pacman mit der
   * Bezeichnung <b>id</b> im Labyrinth. Die Positionsangabe
   * erfolgt dabei als Punkt im Java3D-Koordinatensystem.
   *
   * @param   id  Bezeichnung des Pacman, dessen Position im Labyrinth erfragt wird.
   * @return      Position des Pacman mit der Bezeichnung <b>id</b> im Labyrinth.
   *
   * @see         pacman3d.labyrinth.Labyrinth#setPacmanPos
   */
  public Point3i getPacmanPos(ID id)
  { return (Point3i)positionPacman.get(id);
  }

	/**
	 * liefert eine "neue" Startposition fr Pacmans zurck (diese wird aus
	 * dem Pool der Startpositionen rausgeholt und mehr oder weniger zufllig
	 * vergeben)
	 * @param id Bezeichnung des Pacman, der eine neue Position haben will
	 * @return eine neue Startposition vom Typ Position
	 */
	public Position getNextPacmanStartPosition(ID id) {

		// Position aus dem Pool holen, Zeiger hochzhlen und zurckgeben
		Position oNewPosition = (Position)(m_vecPacmanStartPositions.get(m_iNextPacmanStartPosition));
		m_iNextPacmanStartPosition++;
		if (m_iNextPacmanStartPosition >= m_vecPacmanStartPositions.size()) {
			m_iNextPacmanStartPosition = 0;
		}

		return oNewPosition;
	}

	/**
	 * liefert die Anzahl der im Level vordefinierten Pacmans zurck (das ist
	 * insbesondere fr die Rotation der Startpositionen wichtig) (im Multiplayer-
	 * Spiel knnen auch mehr Pacmans auftreten!)
	 *
	 * @return die Anzahl der im Level definierten Pacmans
	 */
	public int getPacmanDefinitionCount () {
		return (m_vecPacmanStartPositions != null) ? m_vecPacmanStartPositions.size() : 0;
	}

	/**
	 * fgt dem Pool von Pacman-Startpositionen eine "neue" Startposition
	 * hinzu {@link #getNextPacmanStartPosition}
	 *
	 * @param oPosition eine neue Startposition fr Pacmans (bestehend aus
	 * Position und Blickrichtung)
	 */
	public void addPacmanStartPosition(Position oPosition) {
		if (m_vecPacmanStartPositions == null) {
			m_vecPacmanStartPositions = new Vector();
		}
		m_vecPacmanStartPositions.add(oPosition);
	}

	public Vector getAllMonsterStartPositions() {
	  return m_vecMonsterStartPositions;
	}
	public Vector getAllPacmanStartPositions() {
	  return this.m_vecPacmanStartPositions;
	}
	public void removePacmanStartPosition(Point3i point) {
	  if (this.m_vecPacmanStartPositions != null)
	    for (int i = 0; i < this.m_vecPacmanStartPositions.size(); i++) {
	      Position pos = (Position)this.m_vecPacmanStartPositions.get(i);
	      if (pos.getPosition().x == point.x &&
		  pos.getPosition().y == point.y &&
		  pos.getPosition().z == point.z)
		this.m_vecPacmanStartPositions.remove(i--);
	    }
	}
	public void removeMonsterStartPosition(Point3i point) {
	  if (this.m_vecMonsterStartPositions != null)
	    for (int i = 0; i < this.m_vecMonsterStartPositions.size(); i++) {
	      Position pos = (Position)this.m_vecMonsterStartPositions.get(i);
	      if (pos.getPosition().x == point.x &&
		  pos.getPosition().y == point.y &&
		  pos.getPosition().z == point.z)
		this.m_vecMonsterStartPositions.remove(i--);
	    }
	}

	/**
	 * liefert eine "neue" Startposition fr Monster zurck (diese wird aus
	 * dem Pool der Startpositionen rausgeholt und mehr oder weniger zufllig
	 * vergeben)
	 *
	 * @param id Bezeichnung des Monsters, das eine neue Position haben will
	 * @return eine neue Startposition vom Typ Position
	 */
	public Position getNextMonsterStartPosition(ID id) {

		// Position aus dem Pool holen, Zeiger hochzhlen und zurckgeben
		Position oNewPosition = (Position)(m_vecMonsterStartPositions.get(m_iNextMonsterStartPosition));
		m_iNextMonsterStartPosition++;
		if (m_iNextMonsterStartPosition >= m_vecMonsterStartPositions.size()) {
			m_iNextMonsterStartPosition = 0;
		}

		return oNewPosition;
	}

	/**
	 * liefert die Anzahl der im Level vordefinierten Monster zurck (so viele
	 * sollten maximal gleichzeitig im Level dargestellt werden)
	 *
	 * @return die Anzahl der im Level definierten Monster
	 */
	public int getMonsterDefinitionCount () {
		return (m_vecMonsterStartPositions != null) ? m_vecMonsterStartPositions.size() : 0;
	}

	/**
	 * fgt dem Pool von Monster-Startpositionen eine "neue" Startposition
	 * hinzu {@link #getNextMonsterStartPosition}.
	 *
	 * @param oPosition eine neue Startposition fr Monster (nur Startposition
	 * der Wert des Blickrichtung-Members ist im Moment unbedeutend)
	 */
	public void addMonsterStartPosition(Position oPosition) {
		if (m_vecMonsterStartPositions == null) {
			m_vecMonsterStartPositions = new Vector();
		}
		m_vecMonsterStartPositions.add(oPosition);
	}

  /**
   * Diese Methode liefert die Anzahl der aktuell im Labyrinth befindlichen Pacmen.
   *
   * @return Anzahl der im Labyrinth befindlichen Pacmen.
   */
  public int getPacmanCount()
  { return pacman.count();
  }

  /**
   * Diese Methode liefert alle ID's der aktuell im Labyrinth befindlichen Pacman.
   *
   * @return  Alle ID's der im Labyrinth befindlichen Pacman.
   */
  public ID[] getAllPacmanIds()
  { //return pacman.getAllIds();
    return positionPacman.getAllIds();
  }

  /**
   * Diese Methode teilt dem Labyrinth die neue Position des Monsters
   * mit der Bezeichnung <b>id</b> mit. Dabei ist die Positionsangabe
   * im Koordinatensystem von Java3D vorzunehmen.
   *
   * @param id        Monsterbezeichnung
   * @param position  Neue Position des Monsters mit der Bezeichnung <b>id</b>
   *                  im Labyrinth.
   *
   * @see             pacman3d.labyrinth.Labyrinth#getMonsterPos
   */
  public void setMonsterPos(ID id, Point3i position) throws P3DException
  { checkPosition(position);
    positionMonsters.add(id,position);
    if(null!=m_oOverview) m_oOverview.repaint();
  }

  /**
   * Diese Methode teilt dem Labyrinth die neue Position des Monsters
   * mit der Bezeichnung <b>id</b> mit. Dabei ist die Positionsangabe
   * im Koordinatensystem von Java3D vorzunehmen.
   *
   * @param id  Monsterbezeichnung
   * @param x   x-Position
   * @param y   y-Position
   * @param z   z-Position
   *
   * @see	pacman3d.labyrinth.Labyrinth#getMonsterPos
   */
  public void setMonsterPos(ID id, int x, int y, int z) throws P3DException
  { setMonsterPos(id, new Point3i(x,y,z));
  }

  /**
   * Diese Methode liefert die Position des Monsters mit der
   * Bezeichnung <b>id</b> im Labyrinth. Die Positionsangabe
   * erfolgt dabei als Punkt im Java3D-Koordinatensystem.
   *
   * @param   id  Bezeichnung des Monsters, dessen Position im Labyrinth erfragt wird.
   * @return      Position des Monsters mit der Bezeichnung <b>id</b> im Labyrinth.
   *
   * @see         pacman3d.labyrinth.Labyrinth#setMonsterPos
   */
  public Point3i getMonsterPos(ID id)
  { return (Point3i)positionMonsters.get(id);
  }

  /**
   * Diese Methode liefert die Anzahl der aktuell im Labyrinth befindlichen Monster.
   *
   * @return Anzahl der im Labyrinth befindlichen Monster.
   */
  public int getMonsterCount()
  { return monsters.count();
  }

  /**
   * Diese Methode liefert alle ID's der aktuell im Labyrinth befindlichen Monster.
   *
   * @return  Alle ID's der im Labyrinth befindlichen Monster.
   */
  public ID[] getAllMonsterIds()
  { //return monsters.getAllIds();
    return positionMonsters.getAllIds();
  }

  /**
   *  Methode um Monster mit der ID <code>id</code>
   *  aus dem Labyrinth zu entfernen.
   *
   * @param id  ID des zu entfernenden Monster.
   */
  public void removeMonster(ID id) {
    monsters.del(id);
    positionMonsters.del(id);
  }

  /**
   *  Methode um Pacman mit der ID <code>id</code>
   *  aus dem Labyrinth zu entfernen.
   *
   * @param id  ID des zu entfernenden Pacman.
   */
  public void removePacman(ID id) {
    pacman.del(id);
    positionPacman.del(id);
    pacmanBranchGroups.del(id);
  }

  /**
   * Diese Methode liefert die "Gre" des Labyrinths.<br>
   * Das Labyrinth erstreckt sich von (0,0,0) in Richtung pos. x-, y- und z-Achse.
   * Die von dieser Methode gelieferte Point3i-Instanz bezeichnet dabei den
   * Punkt mit dem grten x-, y, und z-Wert.
   *
   * // vielleicht'n Bildchen :)
   *
   * @return  Gibt die "Gre" des Labyrinths als Instanz von <b>Point3i</b> zurck.
   */
  public Point3i getConstraints() {
    // Funktionalitt
    if (labyrinth != null)
      return new Point3i(labyrinth[1].length, labyrinth.length, labyrinth[1][1].length);
    else
      return new Point3i(0,0,0);
  }

  /**
   * Mittels dieser Methode kann die Labyrinthebenen <b>id</b> visuell ein-
   * und ausgeblendet werden. Dazu ist als zweiter Parameter im Falle einer
   * gewnschten Darstellung durch Java3D <b>true</b> und andernfalls
   * <b>false</b> zu bergeben.<br>
   * Es ist zu beachten, da Decken bzw. Bden ebenfalls als Labyrinthebene
   * gezhlt werden.
   *
   * @param	id	ID der Labyrinthebenen, die visuell ein- oder ausgeblendet
   *			werden soll.
   *
   * @param 	state	<b>true</b> Labyrinthebene <b>id</b> soll durch Java3D dargestellt,<br>
   *			<b>false</b> 	andernfalls.
   *
   * @see		pacman3d.labyrinth.Labyrinth#setFloorVisible
   * @see		pacman3d.labyrinth.Labyrinth#isFloorVisible
   * @see		pacman3d.labyrinth.Labyrinth#getFloorCount
   */
  public void setFloorVisible(int id, boolean state)
  {
    if (floors[id] != null) {
	floors[id].setWhichChild(state?0:1);
    }
  }

  /**
   * Mittels dieser Methode kann abgefragt werden, ob die Labyrinthebene
   * <b>id</b> visuell durch Java3D dargestellt wird.
   *
   * @param   id  ID derjenigen Labyrinthebene, von der angefragt wird,
   *              ob diese durch Java3D visuell dargestellt wird.
   *
   * @return  <b>true</b>   Labyrinthebene <b>id</b> wird durch Java3D visuell dargestellt,<br>
   *          <b>false</b>  andernfalls.
   *
   * @see		pacman3d.labyrinth.Labyrinth#isFloorVisible
   * @see		pacman3d.labyrinth.Labyrinth#setFloorVisible
   * @see		pacman3d.labyrinth.Labyrinth#getFloorCount
   */
  public boolean isFloorVisible(int id)
  { return (floors[id].getWhichChild()==0);
  }

  /**
   * Diese Methode liefert die Gesamtanzahl der Labyrinthebenen. Hierbei
   * ist zu beachten, da Decken bzw. Bden ebenfalls als Labyrinthebene
   * gezhlt werden.
   *
   * @return  Anzahl der Labyrinthebenen.
   *
   * @see     pacman3d.labyrinth.Labyrinth#setFloorVisible
   * @see     pacman3d.labyrinth.Labyrinth#isFloorVisible
   */
  public int getFloorCount()
  { return m_p3iConstraints.y;
  }

  /**
   * IM MOMENT NICHT VORGESEHEN!!!
   * <p>
   * Mittels dieser Methode kann abgefragt werden, ob im Labyrinth
   * Gravitation vorherrscht.
   *
   * @return  <b>true</b>   Im Labyrinth herrscht Gravitation<br>,
   *          <b>false</b>  andernfalls.
   */
  public boolean hasGravity()
  { return m_bGravity;
  }

  /**
    * gibt zurck, ob es sich bei der aktuellen Spielesitzung um ein Singleplayer
    * oder Multiplayer-Spiel handelt (relevant fr das Gameplay bei pacmans+monsters)
    *
    * @return true, wenn Multiplayer, false wenn Singleplayer
    */
  public boolean isMultiplayerGame() {
	return m_bMultiplayerGame;
  }

  /**
    * gibt zurck, ob wir in einem Multiplayerspiel der Server sind oder nicht.
    *
    * @return true, wenn Server, false wenn Client
    */
  public boolean isServer() {
	return m_bWeAreServer;
  }

  /**
   * wird nach dem Initialisieren des Labyrinths aufgerufen und nimmt den Status
   * Multiplayer/Singleplayer und die Anzahl der Spieler entgegen (soweit verfgbar!)
   *
   * @param bMultiplayerGame true, wenn Multiplayer-Spiel, false wenn Singleplayer
   * @param iNumberOfPlayers Anzahl der Spieler (soweit Info verfgbar)
   * @param bWeAreServer true, wenn diese Applikation der Server ist, false andernfalls
   * @param bDemoMode true, wenn Applikation im Demomode luft, im Normalbetrieb false
   */
  public void setGameMode (boolean bMultiplayerGame, int iNumberOfPlayers,
    boolean bWeAreServer, boolean bDemoMode) {
	m_bMultiplayerGame = bMultiplayerGame;
	m_iNumberOfPlayers = iNumberOfPlayers;
	m_bWeAreServer = bWeAreServer;
	m_bDemoMode = bDemoMode;
  }

  /**
   * setzt das LevelMetaInfo-Objekt des aktuellen Labyrinths
   *
   * @param oLevelMetaInfo die Levelbeschreibung fr das aktuell im Labyrinth
   * abgebildete Level
   */
  public void setLevelMetaInfo (LevelMetaInfo oLevelMetaInfo) {
	m_oLevelMetaInfo = oLevelMetaInfo;
  }

  /**
   * liefert das LevelMetaInfo-Objekt des aktuellen Labyrinths (also die
   * Beschreibung des geladenen Levels)
   *
   * @return die Levelbeschreibung fr das aktuell im Labyrinth abgebildete Level
   */
  public LevelMetaInfo getLevelMetaInfo () {
	return m_oLevelMetaInfo;
  }

  /**
   * Diese Methode liefert eine Referenz auf eine Instanz von
   * java.awt.Canvas. In dieser Komponente wird eine bersicht
   * ber das Labyrinth gezeichnet, um z.B. die eigenen Position
   * und die der Monster im Labyrinth darzustellen.
   *
   * @return  Referenz auf eine java.awt.Canvas-Instanz, welche ein bersicht
   *          ber das Labyrinth darstellt.
   */
  public Overview getOverview()
  { return m_oOverview;
  }

  /**
   * Diese Methode speichert eine Referenz auf eine Instanz des Overview-Objekts
   * in einer Instanzvariable des Labyrinths. Bei Vernderungen an den
   * Labyrinth-Daten wie Pacman oder Monsterposition wird dann vom Labyrinth
   * die repaint-Methode des gespeicherten Overviewobjekts aufgerufen.
   *
   * @param oOverview Referenz auf eine Overview-Instanz, welche ein bersicht
   * ber das Labyrinth darstellt.
   */
  public void setOverview (Overview oOverview)
  {
	m_oOverview = oOverview;
  }

  /**
   * liefert die Nachrichten-ID des ScorePanels (wenn bekannt)
   * @return die gewnschte ID
   */
  public ID getPanelScoreID() {
	if (m_oGameFrame != null) {
		return m_oGameFrame.panelScore.getID();
	} else {
		return null;
	}
  }

  /**
   * liefert die Nachrichten-ID des Inventory-Panels (wenn bekannt)
   * @return die gewnschte ID
   */
  public ID getPanelInventoryID() {
	if (m_oGameFrame != null) {
		return m_oGameFrame.panelInventory.getID();
	} else {
		return null;
	}
  }

  /**
   * erhht den Zhler der eingesammelten Punkteitems um die spezifizierte Anzahl.
   * Dabei wird das ScorePanel (in dem die Items-Left-Info angezeigt wird)
   * zum Neu-Zeichnen aufgerufen und geprft, ob alle Items aufgesammelt wurden...
   * @param iAmount die Menge der eingesammelten Punkteitems (sollte 1 sein!)
   */
  public void addCollectedItem (int iAmount) {

	// nur im Server- oder Singleplayermodus ausfhren
	if ((!m_bWeAreServer) && (m_bMultiplayerGame)) { return; }

	// mit Itemzhler runtergehen
	m_lScoreItemsCollected += iAmount;

	// neue Nachricht bauen und mit der ID von Game an das GameScorePanel verschicken
	try {
		LabyrinthInfoContent oLabyrinthInfoContent = new LabyrinthInfoContent (m_lScoreItemsMax - m_lScoreItemsCollected);
		// Message oMessage = new Message (m_oGame.getID(), oLabyrinthInfoContent, getPanelScoreID());
		// MessageService.getInstance().sendMessage(oMessage, MessageService.PROPAGATE_LOCAL);
		Message oMessage = new Message (m_oGame.getID(), oLabyrinthInfoContent, Message.RECEIVER_ALL_GAMES);
		MessageService.getInstance().sendMessage(oMessage, MessageService.PROPAGATE_LOCAL + MessageService.PROPAGATE_NETWORK);
	} catch (Exception ex) {
		Debug.out(this.getClass().getName(), ex, Debug.LEVEL_WARNING);
	}

	// wenn Maximalanzahl von Items erreicht, dann Spielende!
	if (m_lScoreItemsCollected >= m_lScoreItemsMax) {
		this.endGame();
	}

  }


  /**
   * setzt das Spielobjekt, in dem das Labyrinth luft (das ist zwar eine
   * oo-like nicht optimale Verflechtung, aber das rhrt von der grundstzlich nicht
   * so optimalen Verknpfung von Labyrinthdarstellung und Steuerung her).
   * @param ein Objekt vom Typ Game
   */
  public void setGame (Game oGame) {
	m_oGame = oGame;
	m_oGameFrame = oGame.getGameFrame();
  }

  /**
   * liefert das Spielobjekt, in dem das Labyrinth luft, zurck
   * @return ein Objekt vom Typ Game
   */
  public Game getGame () {
	return m_oGame;
  }

  /**
   * liefert das Spielfenster, in dem das Labyrinth angezeigt wird
   * @return ein Objekt vom Typ pacman3d.util.GameFrame, falls das Fenster
   * bereits gesetzt wurde.
   */
  public GameFrame getGameFrame () {
	return m_oGameFrame;
  }

  /**
   * setzt die Kamera-Instanz, die das Labyrinth beobachtet (das Labyrinth braucht diese
   * Information im Prinzip nicht zur Darstellung, da dem Labyrinth aber auch eine
   * Steueraufgabe von Pacmans und Monstern zugeteilt wurde, ist sie hier ntig).
   * @param Die zu verwendende Kamera-Instanz
   */
  public void setCamera (Camera oCamera) {
	m_oCamera = oCamera;
  }

  /**
   * liefert die Kamera-Instanz, die das Labyrinth beobachtet {@link #setCamera}.
   * @return die verwendete Kamera-Instanz
   */
  public Camera getCamera () {
	return m_oCamera;
  }

  /**
   * setzt ein Flag, ob Wireframes oder echte Zellen gezeichnet werden sollen
   * @param bWireFrameMode wenn true, werden nur Linien des Labyrinths gezeichnet,
   * bei false die "echten" Zellenwnde.
   */
  public void setWireFrameMode (boolean bWireFrameMode) {
	m_bWireFrameMode = bWireFrameMode;
  }

  /**
   * liefert zurck, ob Wireframes oder echte Zellen gezeichnet werden sollen
   * @return wenn true, werden nur Linien des Labyrinths gezeichnet,
   * bei false die "echten" Zellenwnde.
   */
  public boolean getWireFrameMode () {
	return m_bWireFrameMode;
  }

  /**
   * wird vom aktiven Pacman aufgerufen, um das Spiel zu beenden. Dabei wird
   * auch der erreichte Punktestand bermittelt.
   * @param oPacmanId die ID des aufrufenden Pacmans
   * @param iPacmanScore der erreichte Punktestand des aufrufenden Pacmans
   */
  public void endGame (ID oPacmanId, int iPacmanScore) {

	// Spielende-Methode in Gameklasse aufrufen, wenn es sich um den aktiven
	// Pacman handelt, der das hier aufruft!
	if (this.m_idActivePacman.equals(oPacmanId)) {
		m_oGame.stopGame();
	}

  }

  /**
   * hlt das Spiel an (wird vom Singleplayer- oder Serverlabyrinth (also uns
   * selbst :) aufgerufen, wenn das Spiel wegen Erreichen des maximalen
   * Punktestandes zu Ende ist)
   */
  public void endGame () {

	m_oGame.stopGame();

  }

  /** Die Methode berprft ob sich die bergebene Position innerhalb des Labyrinthes befindet
   *  und wirft eine Exception ansonsten!
   *  @param pos Die zu berprfende Position
   */
  private void checkPosition(Point3i pos) throws P3DException
  { if( pos.x > m_p3iConstraints.x || pos.x < 0 ||
	pos.y > m_p3iConstraints.y || pos.x < 0 ||
	pos.z > m_p3iConstraints.z || pos.x < 0)
      throw new P3DException("Illegal Placement: Position " + pos.toString() + " is not in the Labyrinth");
  }

	/**
	 * erzeugt ein shape3d-Objekt, dass eine Gitterebene aus Linien darstellt
	 * @return shape3d-Objekt mit den Liniendaten
	 */
	private Shape3D createLineGrid() {

		LineArray landGeom = new LineArray(44, GeometryArray.COORDINATES
						    | GeometryArray.COLOR_3);
		float l = 0.0f;
		for(int c = 0; c < 44; c+=4) {
			landGeom.setCoordinate( c+0, new Point3f( -50.0f, 0.0f,  l ));
			landGeom.setCoordinate( c+1, new Point3f(  50.0f, 0.0f,  l ));
			landGeom.setCoordinate( c+2, new Point3f(   l   , 0.0f, -50.0f ));
			landGeom.setCoordinate( c+3, new Point3f(   l   , 0.0f,  50.0f ));
			l += 1.0f;
		}

		Color3f c = new Color3f(0.0f, 0.0f, 0.8f);
		for(int i = 0; i < 44; i++) landGeom.setColor( i, c);

		return new Shape3D(landGeom);
	}

	/**
	 * erzeugt einen Linienkubus mit Kantenlnge 1
	 * @param x die x-Koordinate des Wrfelmittelpunkts
	 * @param y die y-Koordinate des Wrfelmittelpunkts
	 * @param z die z-Koordinate des Wrfelmittelpunkts
	 * @return shape3d-Objekt des Linienkubuses
	 */
	private Shape3D createWireCube(int x, int y, int z) {

		LineArray landGeom = new LineArray(40, GeometryArray.COORDINATES
						    | GeometryArray.COLOR_3);

		float fX = (float)x, fY = (float)y, fZ = (float)z;

		// Kantenlinien des Wrfels
		landGeom.setCoordinate(  0, new Point3f( -0.5f+fX, -0.5f+fY,  -0.5f+fZ ));
		landGeom.setCoordinate(  1, new Point3f(  0.5f+fX, -0.5f+fY,  -0.5f+fZ ));
		landGeom.setCoordinate(  2, new Point3f( -0.5f+fX,  0.5f+fY,  -0.5f+fZ ));
		landGeom.setCoordinate(  3, new Point3f(  0.5f+fX,  0.5f+fY,  -0.5f+fZ ));
		landGeom.setCoordinate(  4, new Point3f( -0.5f+fX, -0.5f+fY,  -0.5f+fZ ));
		landGeom.setCoordinate(  5, new Point3f( -0.5f+fX,  0.5f+fY,  -0.5f+fZ ));
		landGeom.setCoordinate(  6, new Point3f(  0.5f+fX, -0.5f+fY,  -0.5f+fZ ));
		landGeom.setCoordinate(  7, new Point3f(  0.5f+fX,  0.5f+fY,  -0.5f+fZ ));

		landGeom.setCoordinate(  8, new Point3f( -0.5f+fX, -0.5f+fY,  0.5f+fZ ));
		landGeom.setCoordinate(  9, new Point3f(  0.5f+fX, -0.5f+fY,  0.5f+fZ ));
		landGeom.setCoordinate( 10, new Point3f( -0.5f+fX,  0.5f+fY,  0.5f+fZ ));
		landGeom.setCoordinate( 11, new Point3f(  0.5f+fX,  0.5f+fY,  0.5f+fZ ));
		landGeom.setCoordinate( 12, new Point3f( -0.5f+fX, -0.5f+fY,  0.5f+fZ ));
		landGeom.setCoordinate( 13, new Point3f( -0.5f+fX,  0.5f+fY,  0.5f+fZ ));
		landGeom.setCoordinate( 14, new Point3f(  0.5f+fX, -0.5f+fY,  0.5f+fZ ));
		landGeom.setCoordinate( 15, new Point3f(  0.5f+fX,  0.5f+fY,  0.5f+fZ ));

		landGeom.setCoordinate( 16, new Point3f( -0.5f+fX, -0.5f+fY, -0.5f+fZ ));
		landGeom.setCoordinate( 17, new Point3f( -0.5f+fX, -0.5f+fY,  0.5f+fZ ));
		landGeom.setCoordinate( 18, new Point3f(  0.5f+fX, -0.5f+fY, -0.5f+fZ ));
		landGeom.setCoordinate( 19, new Point3f(  0.5f+fX, -0.5f+fY,  0.5f+fZ ));
		landGeom.setCoordinate( 20, new Point3f( -0.5f+fX,  0.5f+fY, -0.5f+fZ ));
		landGeom.setCoordinate( 21, new Point3f( -0.5f+fX,  0.5f+fY,  0.5f+fZ ));
		landGeom.setCoordinate( 22, new Point3f(  0.5f+fX,  0.5f+fY, -0.5f+fZ ));
		landGeom.setCoordinate( 23, new Point3f(  0.5f+fX,  0.5f+fY,  0.5f+fZ ));

		// gekreuzte Linien
		landGeom.setCoordinate( 24, new Point3f( -0.5f+fX, -0.5f+fY, -0.5f+fZ ));
		landGeom.setCoordinate( 25, new Point3f(  0.5f+fX,  0.5f+fY, -0.5f+fZ ));
		landGeom.setCoordinate( 26, new Point3f( -0.5f+fX,  0.5f+fY, -0.5f+fZ ));
		landGeom.setCoordinate( 27, new Point3f(  0.5f+fX, -0.5f+fY, -0.5f+fZ ));
		landGeom.setCoordinate( 28, new Point3f( -0.5f+fX, -0.5f+fY,  0.5f+fZ ));
		landGeom.setCoordinate( 29, new Point3f(  0.5f+fX,  0.5f+fY,  0.5f+fZ ));
		landGeom.setCoordinate( 30, new Point3f( -0.5f+fX,  0.5f+fY,  0.5f+fZ ));
		landGeom.setCoordinate( 31, new Point3f(  0.5f+fX, -0.5f+fY,  0.5f+fZ ));

		landGeom.setCoordinate( 32, new Point3f(  0.5f+fX, -0.5f+fY, -0.5f+fZ ));
		landGeom.setCoordinate( 33, new Point3f(  0.5f+fX,  0.5f+fY,  0.5f+fZ ));
		landGeom.setCoordinate( 34, new Point3f(  0.5f+fX,  0.5f+fY, -0.5f+fZ ));
		landGeom.setCoordinate( 35, new Point3f(  0.5f+fX, -0.5f+fY,  0.5f+fZ ));
		landGeom.setCoordinate( 36, new Point3f( -0.5f+fX, -0.5f+fY, -0.5f+fZ ));
		landGeom.setCoordinate( 37, new Point3f( -0.5f+fX,  0.5f+fY,  0.5f+fZ ));
		landGeom.setCoordinate( 38, new Point3f( -0.5f+fX,  0.5f+fY, -0.5f+fZ ));
		landGeom.setCoordinate( 39, new Point3f( -0.5f+fX, -0.5f+fY,  0.5f+fZ ));

		// Farbe der Linien setzen
		Color3f c = new Color3f(0.0f, 0.0f, 0.8f);
		for(int i = 0; i < 40; i++) {
			landGeom.setColor( i, c);
		}

		return new Shape3D(landGeom);
	}

	/**
	 * erzeugt den Contentbranch des Labyrinths
	 * @return eine Branchgroup, die die 3D-Elemente des Labyrinths enthlt
	 */
	public BranchGroup buildContentBranch()	{

		contentBranch = new BranchGroup();
		floors = new Switch[m_p3iConstraints.y];
		TransformGroup trGr;
		Transform3D tr = new Transform3D();
		int xOffset[] = {1,0,-1,0};
		int zOffset[] = {0,-1,0,1};
		int masks[]   = {1,2,4,8};
		int sideMask;

		for(int y=0; y<m_p3iConstraints.y; y++) {

			floors[y] = new Switch(0);
			floors[y].setCapability(Switch.ALLOW_SWITCH_WRITE);
			floors[y].setCapability(Switch.ALLOW_SWITCH_READ);
			trGr = new TransformGroup();

			// Unterscheidung zwischen Linien- und Polygon-Modus
			if (this.m_bWireFrameMode) {

				for (int x=0; x < m_p3iConstraints.x; x++)
				 for (int z=0; z < m_p3iConstraints.z; z++) {

					// fr die nicht passierbaren Zellen wird ein
					// Linenienkubus erzeugt (wireframe-mode)
					if (labyrinth[y][x][z].isWalkable()) {
						trGr.addChild(labyrinth[y][x][z].getJ3DGroup());
					} else {
						trGr.addChild( createWireCube(x,y,z) );
					}

				}

			} else {

				for (int x=0; x < m_p3iConstraints.x; x++)
				 for (int z=0; z < m_p3iConstraints.z; z++) {

					if(false) { //labyrinth[y][x][z] instanceof CellWall) {

						// Optimierung: seiten zwischen Zellen ausblenden
						sideMask = 0;
						for(int i=0;i<4;i++) {
							try {
								if (labyrinth[y][x+xOffset[i]][z+zOffset[i]] instanceof CellWall)
								sideMask += masks[i];
							} catch(ArrayIndexOutOfBoundsException e) {
								// Zelle am Rand - wird ignoriert
							}
						}

						try {
							((CellWall)labyrinth[y][x][z]).setSideMask(sideMask);
						} catch (Exception e) {
							// will not happen
						}

					}

					// Zelle hinzufgen
					if (labyrinth[y][x][z] != null) {
						trGr.addChild(labyrinth[y][x][z].getJ3DGroup());
					} else {
						Debug.out( getClass().getName(), "cell is null at (" +
						(new Integer(y)).toString() + ","  +
						(new Integer(x)).toString() + ","  +
						(new Integer(z)).toString() + ")", Debug.LEVEL_CRITICAL );
					}

				}

			}

			floors[y].addChild(trGr);
			floors[y].addChild(null);
			contentBranch.addChild(floors[y]);
		}

		// Lichtquelle erzeugen
		AmbientLight ambientLight = new AmbientLight(new Color3f(0.5f,0.5f,0.5f));
		ambientLight.setInfluencingBounds(new BoundingSphere(new Point3d(m_p3iConstraints.x/2,m_p3iConstraints.y/2,m_p3iConstraints.z/2),100));
		contentBranch.addChild(ambientLight);

		// Content Branch optimieren
		contentBranch.compile();

		return contentBranch;
	}

  /**
   * ldt das Level aus der bergebenen URL
   * @param urlLevel die URL des zu ladenden Levels
   */
  public void loadLevel (URL urlLevel)
	throws pacman3d.util.InvalidLevelFileException, java.io.IOException {

	try {
		loadLevel(urlLevel.openStream());
		floors = new Switch[m_p3iConstraints.y];
	} catch (java.io.IOException ex) {
		Debug.out (this.getClass().getName(), ex, Debug.LEVEL_CRITICAL);
		throw ex;
	}
  }

  /**
   * ldt das Level aus dem bergebenen InputStream
   * @param oLevelStream der Inputstrom des zu ladenden Levels
   */
  public void loadLevel (InputStream oLevelStream)
	throws pacman3d.util.InvalidLevelFileException {

	// Einladen der Leveldaten aus dem angegebenen Inputstream (oLevelStream)
	int iMaxX = 10, iMaxY = 10, iMaxZ = 10;
	Document doc;
	try {
		Debug.out (this.getClass().getName(), "loading level: "+oLevelStream.toString(), Debug.LEVEL_NOTICE);

		// Erzeugen des DOM-Parsers und Einlesen des XML-Stroms
		DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
		DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
		doc = docBuilder.parse (oLevelStream);

		// normalize text representation
		doc.getDocumentElement ().normalize ();
		Element elDoc = doc.getDocumentElement();

		// hier mu ein Dispatcher die ntigen den xml-tags entsprechenden
		// Elemente instanzieren (auch rekursiv teilweise, so dass hier nur
		// die obersten Tags verarbeitet werden!) und dann die entsprechenden
		// loadFromDOM-Methoden mit den Elementknoten aufrufen

		// suche das <common>-Element, instanziere ein LevelMetaInfo-Objekt
		// und lade es mit den Daten
		Element elCommon = XmlUtils.getChildElement (elDoc, LevelXmlSyntax.TAG_COMMON);
		m_oLevelMetaInfo = new LevelMetaInfo ();
		m_oLevelMetaInfo.loadFromDOM (elCommon);
		Debug.out (this.getClass().getName(), "title of level: "+m_oLevelMetaInfo.getTitle().toString(), Debug.LEVEL_NOTICE);

		// suche das <resources>-Element und gebe es an den Resourcenhandler
		Element elResources = XmlUtils.getChildElement (elDoc, LevelXmlSyntax.TAG_RESOURCES);
		ResourceHandler oResourceHandler = pacman3d.util.ResourceHandler.getInstance();
		oResourceHandler.loadFromDOM(elResources);

		// suche das <itemtypes>-Element und gebe es an den ItemHandler
		Element elItemTypes = XmlUtils.getChildElement (elDoc, LevelXmlSyntax.TAG_ITEMTYPES);
		ItemHandler oItemHandler = ItemHandler.getInstance();
		oItemHandler.loadFromDOM(elItemTypes);

		// suche das <celltypes>-Element und gebe es an den CellHandler
		Element elCellTypes = XmlUtils.getChildElement (elDoc, LevelXmlSyntax.TAG_CELLTYPES);
		CellHandler oCellHandler = CellHandler.getInstance();
		oCellHandler.loadFromDOM(elCellTypes);

		// suche das <levelstructure>-Element und beginne den Aufbau des Levels
		Element elLevelStructure = XmlUtils.getChildElement (elDoc, LevelXmlSyntax.TAG_LEVELSTRUCTURE);
		if (elLevelStructure == null) {
			// das <levelstructure>-Element darf nicht fehlen!
			throw new pacman3d.util.InvalidLevelFileException (
				"mandatory tag for the level structure is missing." );
		}

		// Einlesen des Gravitationsattributs (im Moment wird hier nur zwischen
		// 0 und irgendetwas anderem unterschieden)
		String sGravity = elLevelStructure.getAttribute(LevelXmlSyntax.ATTR_GRAVITY);
		if (sGravity != null) {
			if (sGravity.equals("0")) {
				m_bGravity = false;
			} else {
				m_bGravity = true;
			}
		} else {
			// Defaulteinstellung ist Gravitt = true
			m_bGravity = true;
		}

		// das <levelstructure>-Element hat eine Reihe von Kinderelementen. Dazu zhlen
		// <pacmen>, <monsters> und <floors>. Zunchst wenden wir uns hier den floors
		// - also den eigentlichen Labyrinth-Konstruktionsdaten - zu. Das <floors>-Element
		// enthlt seinerseits fr jede Ebene ein <floor>-Element, in dem eine Reihe von
		// Zellen stehen knnen (<cell>). Die Zellen werden linear aneinandergereiht,
		// die Hhe und Breite ergibt sich durch die festgesetzten Grenangaben des Levels.
		Element elFloors = XmlUtils.getChildElement (elLevelStructure, LevelXmlSyntax.TAG_FLOORS);
		NodeList oFloors = elFloors.getElementsByTagName(LevelXmlSyntax.TAG_FLOOR);
		iMaxY = oFloors.getLength();
		if (iMaxY < 1) {
			// exception werfen, weil das level mindestens einen Flur haben mu!
			throw new pacman3d.util.InvalidLevelFileException (
				"level must contain at least one floor." );
		}

		// hole die Constraints aus den Attributen rows und cols des Floors-Elements
		iMaxZ = 0; iMaxX = 0;
		try {
			String sCols = elFloors.getAttribute(LevelXmlSyntax.ATTR_COLS);
			iMaxX = new Integer(sCols).intValue();
			String sRows = elFloors.getAttribute(LevelXmlSyntax.ATTR_ROWS);
			iMaxZ = new Integer(sRows).intValue();
			Debug.out(this.getClass().toString(), "constraints of the loaded level are: MaxY=" + iMaxY + ", MaxZ=" + iMaxZ + ", MaxX=" + iMaxX+"", Debug.LEVEL_NOTICE);
		} catch (Exception ex) {
			// exception werfen, dass level nicht geladen werden kann, weil Constraints fehlen!
			throw new pacman3d.util.InvalidLevelFileException (
				"could not read level constraints." );
		}

		// Erzeugen der bentigten Labyrinth-internen Speicehr und die Abmessungen
		// in der Instanzvariable setzen
		labyrinth = new Cell[iMaxY][iMaxX][iMaxZ];
		m_asCellTypes = new String[iMaxY][iMaxX][iMaxZ];
		m_p3iConstraints = new Point3i(labyrinth[1].length, labyrinth.length, labyrinth[1][1].length);

		m_lScoreItemsMax = 0;
		m_lScoreItemsCollected = 0;

		// durchlaufe die Flure und schaue, was da fr Zellen drin sind

		if (oFloors.getLength() > 0) {

			// Durchlaufen der Flure Flur fr Flur
			for (int iFloorCounter = 0; iFloorCounter < oFloors.getLength(); iFloorCounter++) {
				Element elCurrentFloor = (Element)oFloors.item(iFloorCounter);

				// durchlaufe den Flur und erzeuge Zelleninstanzen
				NodeList oCells = elCurrentFloor.getElementsByTagName(LevelXmlSyntax.TAG_CELL);

				////// Fr rechteckige Labyrinthe
				int x = 0;
				//////

				if (oCells.getLength() > 0) {
					for (int iCellCounter = 0; iCellCounter < oCells.getLength(); iCellCounter++) {
						Element elCurrentCell = (Element)oCells.item(iCellCounter);
						String sId = elCurrentCell.getAttribute(LevelXmlSyntax.ATTR_ID);
						String sType = elCurrentCell.getAttribute(LevelXmlSyntax.ATTR_TYPE);
						// Debug.out(this.getClass().toString(), "cell (id=" + sId + ", type=" + sType+")", Debug.LEVEL_NOTICE);

						// x,y,z-Koordinaten des aktuellen xml-Zellen-Elements
						// im labyrinth[]-Array ermitteln
						int y = iFloorCounter;

						////// Fr rechteckige Labyrinthe
						//int x = iCellCounter / iMaxX;
						//int z = iCellCounter % iMaxX;
						int z = iCellCounter % iMaxZ;
						////// Fr rechteckige Labyrinthe



						// get an instance of the class
						Cell oCell = oCellHandler.getCellInstance(sType);

						if (oCell != null) {
							oCell.setPosition (new Point3i(x,y,z));
							oCell.setGameMode(m_bMultiplayerGame, m_bWeAreServer);
							oCell.loadFromDOM(elCurrentCell);
							m_lScoreItemsMax += oCell.getScoreItemsCount();
							oCell.setLabyrinth(this);

							this.labyrinth[y][x][z] = oCell;
							this.m_asCellTypes[y][x][z] = sType;
							if (z >= iMaxZ-1)
							  x++;
						} else {
							// werfe Exception, dass eine Zelle nicht geliefert
							// werden kann (das wrde zu einer Exception whrend
							// des Spiels fhren!)
							Debug.out (this.getClass().getName(), "celltype was not found in cell repository.", Debug.LEVEL_CRITICAL);
							throw new pacman3d.util.InvalidLevelFileException();
						}


					}
				} else {
					// execption!  -> keine Zellen im aktuellen Flur definiert!
					Debug.out (this.getClass().getName(), "empty floor found. There must be cells in labyrinth floors.", Debug.LEVEL_CRITICAL);
					throw new pacman3d.util.InvalidLevelFileException();
				}
			}
		}

		// Debug-Ausgabe, wieviele der berchtigen Score-Items gefunden wurden...
		Debug.out (this.getClass().getName(), "score items found: " + m_lScoreItemsMax, Debug.LEVEL_NOTICE);

		// an dieser Stelle wurden die meisten Leveldaten bereits eingelesen.
		// Insbesondere die Positionen der Zellen sind drin, was beim
		// folgenden Auslesen der Daten von Pacmans und Monstern zur
		// Fehlerprfung relevant ist.
		Element elPacmans = XmlUtils.getChildElement (elLevelStructure, LevelXmlSyntax.TAG_PACMANS);
		NodeList oPacmans = elPacmans.getElementsByTagName(LevelXmlSyntax.TAG_PACMAN);
		m_vecPacmanStartPositions = new Vector();
		if (oPacmans.getLength() > 0) {
			for (int iCurrentPacmanNo = 0; iCurrentPacmanNo < oPacmans.getLength(); iCurrentPacmanNo++) {

				// aus jedem <pacman>-Element die startposition- und lineofsight-Info rauslesen
				Element elCurrentPacman = (Element)(oPacmans.item(iCurrentPacmanNo));
				if (elCurrentPacman != null) {

					Position oPosition = null;
					Point3i oStartPosition = null;
					Vector3d oLineOfSight = null;

					// zunchst die startposition holen
					Element elStartPosition = XmlUtils.getChildElement (elCurrentPacman, LevelXmlSyntax.TAG_STARTPOSITION);
					int iX = 0, iY = 0, iZ = 0;
					if (elStartPosition != null) {
						try {
							String sX = elStartPosition.getAttribute(LevelXmlSyntax.ATTR_X);
							if (sX != null) { iX = new Integer(sX).intValue(); }
							String sY = elStartPosition.getAttribute(LevelXmlSyntax.ATTR_Y);
							if (sY != null) { iY = new Integer(sY).intValue(); }
							String sZ = elStartPosition.getAttribute(LevelXmlSyntax.ATTR_Z);
							if (sZ != null) { iZ = new Integer(sZ).intValue(); }
							oStartPosition = new Point3i(iX, iY, iZ);
							Debug.out( this.getClass().getName(), "pacman " + iCurrentPacmanNo + " has startposition: x=" + iX + "  y=" + iY + "  z=" + iZ, Debug.LEVEL_NOTICE);
						} catch (java.lang.NumberFormatException ex) {
							Debug.out (this.getClass().getName(), "keine Zahlen in den x/y/z-Attributen vorgefunden.", Debug.LEVEL_WARNING);
						}
					}

					// als nchstes die lineofsight holen
					Element elLineOfSight = XmlUtils.getChildElement (elCurrentPacman, LevelXmlSyntax.TAG_LINEOFSIGHT);
					double dLosX = 0.0, dLosY = 0.0, dLosZ = 1.0;
					if (elLineOfSight != null) {
						try {
							String sX = elLineOfSight.getAttribute(LevelXmlSyntax.ATTR_X);
							if (sX != null) { dLosX = new Double(sX).doubleValue(); }
							String sY = elLineOfSight.getAttribute(LevelXmlSyntax.ATTR_Y);
							if (sY != null) { dLosY = new Double(sY).doubleValue(); }
							String sZ = elLineOfSight.getAttribute(LevelXmlSyntax.ATTR_Z);
							if (sZ != null) { dLosZ = new Double(sZ).doubleValue(); }
							oLineOfSight = new Vector3d (dLosX, dLosY, dLosZ);
							Debug.out( this.getClass().getName(), "pacman " + iCurrentPacmanNo + " has lineofsight: x=" + dLosX + "  y=" + dLosY + "  z=" + dLosZ, Debug.LEVEL_NOTICE);
						} catch (java.lang.NumberFormatException ex) {
							Debug.out (this.getClass().getName(), "keine Zahlen in den x/y/z-Attributen vorgefunden.", Debug.LEVEL_WARNING);
						}
					}

					// nur wenn beide Informationen eingelesen werden konnten, dann ein
					// Position-Objekt erstellen und in den Vector reinlegen
					if ((oLineOfSight != null) && (oStartPosition != null)) {
						oPosition = new Position (oStartPosition, oLineOfSight);
						this.m_vecPacmanStartPositions.add (oPosition);
					}

					// wenn es sich um das erste Pacman-Element handelt,
					// dann prfen, ob ein <demomode>-Element enthalten ist. Ist
					// dies der Fall, wird eine Pacman-Instanz erzeugt und
					// dessen LoadFromDOM-Methode mit dem Knoten aufgerufen
					// wird dieses dann eingeladen
					if (m_bDemoMode) {
						if (iCurrentPacmanNo == 0) {
							Element elDemoMode = XmlUtils.getChildElement (elCurrentPacman, "demomode");
							Element elMeta = XmlUtils.getChildElement (elDemoMode, "meta");
							String sPacmanModel = elMeta.getAttribute("pacmanmodel");

							Pacman oPacman = new Pacman(this, sPacmanModel, true);
							this.setPacman(oPacman.getID(), oPacman);
							this.setActivePacmanID(oPacman.getID());
							if (oPosition == null) {
								oPosition = this.getNextPacmanStartPosition(oPacman.getID());
							}
							oPacman.setLineOfSight (oPosition.getLineOfSight());
							oPacman.setPacmanPos (oPosition.getPosition());
							this.setPacmanPos (oPacman.getID(), oPosition.getPosition());

							oPacman.loadFromDOM(elDemoMode);
						}
					}

				}
			}
		} else {
			throw new pacman3d.util.InvalidLevelFileException (
				"at least one pacman-tag has to be defined in level." );
		}

		// Setzen der Zeigervariable auf die nchste Startposition
		if (this.m_vecPacmanStartPositions.size() > 0) {
			m_iNextPacmanStartPosition = 0;
		} else {
			throw new pacman3d.util.InvalidLevelFileException (
				"at least one pacman has to be available in level." );
		}

		// und hnlich wie bei den Pacman-Startpositionen auch die Monster-
		// Startpositionen ermitteln und abspeichern
		Element elMonsters = XmlUtils.getChildElement (elLevelStructure, LevelXmlSyntax.TAG_MONSTERS);
		NodeList oMonsters = elMonsters.getElementsByTagName(LevelXmlSyntax.TAG_MONSTER);
		m_vecMonsterStartPositions = new Vector();
		if (oMonsters.getLength() > 0) {
			for (int iCurrentMonsterNo = 0; iCurrentMonsterNo < oMonsters.getLength(); iCurrentMonsterNo++) {

				// aus jedem <pacman>-Element die startposition- und lineofsight-Info rauslesen
				Element elCurrentMonster = (Element)(oMonsters.item(iCurrentMonsterNo));
				if (elCurrentMonster != null) {

					Position oPosition = null;
					Point3i oStartPosition = null;

					// die startposition holen
					Element elStartPosition = XmlUtils.getChildElement (elCurrentMonster, LevelXmlSyntax.TAG_STARTPOSITION);
					int iX = 0, iY = 0, iZ = 0;
					if (elStartPosition != null) {
						try {
							String sX = elStartPosition.getAttribute(LevelXmlSyntax.ATTR_X);
							if (sX != null) { iX = new Integer(sX).intValue(); }
							String sY = elStartPosition.getAttribute(LevelXmlSyntax.ATTR_Y);
							if (sY != null) { iY = new Integer(sY).intValue(); }
							String sZ = elStartPosition.getAttribute(LevelXmlSyntax.ATTR_Z);
							if (sZ != null) { iZ = new Integer(sZ).intValue(); }
							oStartPosition = new Point3i(iX, iY, iZ);
							Debug.out( this.getClass().getName(), "monster " + iCurrentMonsterNo + " has startposition: x=" + iX + "  y=" + iY + "  z=" + iZ, Debug.LEVEL_NOTICE);
						} catch (java.lang.NumberFormatException ex) {
							Debug.out (this.getClass().getName(), "keine Zahlen in den x/y/z-Attributen vorgefunden.", Debug.LEVEL_WARNING);
						}
					}

					// bei den Monstern gibt es im Moment noch kein lineofsight,
					// mglicherweise wird die Monstergruppe das noch anbieten

					// nur wenn die Information erfolgreich eingelesen werden konnte,
					// dann ein Position-Objekt erstellen und in den Vector reinlegen
					if (oStartPosition != null) {
						oPosition = new Position (oStartPosition, new Vector3d(0.0, 0.0, 0.0));
						this.m_vecMonsterStartPositions.add (oPosition);
					}
				}
			}
		}

	} catch (SAXParseException err) {
		Debug.out (this.getClass().getName(), "** Parsing error"
		  + ", line " + err.getLineNumber ()
		  + ", uri " + err.getSystemId (), Debug.LEVEL_CRITICAL);
		Debug.out (this.getClass().getName(), "" + err.getMessage ()
		  , Debug.LEVEL_CRITICAL);

		throw new pacman3d.util.InvalidLevelFileException();
	} catch (SAXException e) {
		Exception x = e.getException ();
		((x == null) ? e : x).printStackTrace ();
		throw new pacman3d.util.InvalidLevelFileException();
	} catch (Throwable t) {
		t.printStackTrace ();
		throw new pacman3d.util.InvalidLevelFileException();
	}

	Debug.out (this.getClass().getName(), "level loaded successfully.", Debug.LEVEL_NOTICE);
  }

  /**
   * speichert das Level in einer Datei mit dem bergebenen Dateinamen
   * @param sLevelFileName der zur Speicherung zu verwendende Dateiname
   */
  public void saveLevel (String sLevelFileName) {

	// build some xml structures
	Document doc;
	try {
		DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
		DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
		doc = docBuilder.newDocument ();

		// das Dokumentelement erzeugen (<gamelevel>)
		Element elGameLevel = doc.createElement (LevelXmlSyntax.TAG_GAMELEVEL);
		doc.appendChild (elGameLevel);

		elGameLevel.setAttribute( "xmlns:xlink", "http://www.w3.org/1999/xlink" );

		// CellHandler-Instanz holen
		CellHandler oCellHandler = CellHandler.getInstance();

		// schreiben der level meta infos
		Element elLevelMetaInfo = null;
		if (m_oLevelMetaInfo != null) {
			elLevelMetaInfo = m_oLevelMetaInfo.saveToDOM(doc);
		} else {
			LevelMetaInfo oLevelMetaInfo = new LevelMetaInfo();
			elLevelMetaInfo = oLevelMetaInfo.saveToDOM(doc);
		}
		elGameLevel.appendChild(elLevelMetaInfo);

		// schreiben der aktuellen Cell-Instanzen
		Element elLevelStructure = doc.createElement (LevelXmlSyntax.TAG_LEVELSTRUCTURE);
		elLevelStructure.setAttribute(LevelXmlSyntax.ATTR_GRAVITY, "" + (m_bGravity ? 1 : 0));
		elGameLevel.appendChild (elLevelStructure);

		// schreiben der Pacmen
		// erzeugen und einhngen des XML-Elementes
		Element oPacmansElement = doc.createElement(
			LevelXmlSyntax.TAG_PACMANS );
		elLevelStructure.appendChild( oPacmansElement );

		// ermitteln der Anzahl von Pacman-Positionen
		int iPacmanCount = m_vecPacmanStartPositions.size();
		for (int i=0; i<iPacmanCount; i++) {

			// ermitteln der Position und Richtung
			Position oPosition =
				(Position) m_vecPacmanStartPositions.get( i );

			// erzeugen und einfuegen der XML-Elemente
			Element oPacmanElement = doc.createElement(
				LevelXmlSyntax.TAG_PACMAN );
			oPacmansElement.appendChild( oPacmanElement );

			// die Position XML-serialisieren
			Point3i oPoint = oPosition.getPosition();

			// erzeugen und einfuegen der XML-Elemente
			Element oPositionElement = doc.createElement(
				LevelXmlSyntax.TAG_STARTPOSITION );
			oPacmanElement.appendChild( oPositionElement );

			oPositionElement.setAttribute( LevelXmlSyntax.ATTR_X,
				Integer.toString( oPoint.x ) );
			oPositionElement.setAttribute( LevelXmlSyntax.ATTR_Y,
				Integer.toString( oPoint.y ) );
			oPositionElement.setAttribute( LevelXmlSyntax.ATTR_Z,
				Integer.toString( oPoint.z ) );

			Element oSightlineElement = doc.createElement(
				LevelXmlSyntax.TAG_LINEOFSIGHT );
			oPacmanElement.appendChild( oSightlineElement );

			Vector3d oSightline = oPosition.getLineOfSight();

			oSightlineElement.setAttribute( LevelXmlSyntax.ATTR_X,
				Double.toString( oSightline.x ) );
			oSightlineElement.setAttribute( LevelXmlSyntax.ATTR_Y,
				Double.toString( oSightline.y ) );
			oSightlineElement.setAttribute( LevelXmlSyntax.ATTR_Z,
				Double.toString( oSightline.z ) );

			// der erste Pacman serialisiert seine Bewegungen im
			// <demomode>-Element
			if (i == 0) {
				ID idActivePacman = getActivePacmanID();
				if (idActivePacman != null) {
					Pacman oActivePacman = getPacman(getActivePacmanID());

					if (oActivePacman != null) {
						oPacmanElement.appendChild(
							getPacman(getActivePacmanID()).saveToDOM(doc) );
					}
				}
			}

		}

		// schreiben der Monster
		// erzeugen und einhngen des XML-Elementes
		Element oMonstersElement = doc.createElement(
			LevelXmlSyntax.TAG_MONSTERS );
		elLevelStructure.appendChild( oMonstersElement );

		// ermitteln der Anzahl von Monster-Positionen
		int iMonsterCount = m_vecMonsterStartPositions.size();
		for (int i=0; i<iMonsterCount; i++) {

			// ermitteln der Position und Richtung
			Position oPosition =
				(Position) m_vecMonsterStartPositions.get( i );

			// erzeugen und einfuegen der XML-Elemente
			Element oMonsterElement = doc.createElement(
				LevelXmlSyntax.TAG_MONSTER );
			oMonstersElement.appendChild( oMonsterElement );

			// die Position XML-serialisieren
			Point3i oPoint = oPosition.getPosition();

			// erzeugen und einfuegen der XML-Elemente
			Element oPositionElement = doc.createElement(
				LevelXmlSyntax.TAG_STARTPOSITION );
			oMonsterElement.appendChild( oPositionElement );

			oPositionElement.setAttribute( LevelXmlSyntax.ATTR_X,
				Integer.toString( oPoint.x ) );
			oPositionElement.setAttribute( LevelXmlSyntax.ATTR_Y,
				Integer.toString( oPoint.y ) );
			oPositionElement.setAttribute( LevelXmlSyntax.ATTR_Z,
				Integer.toString( oPoint.z ) );

			Element oSightlineElement = doc.createElement(
				LevelXmlSyntax.TAG_LINEOFSIGHT );
			oMonsterElement.appendChild( oSightlineElement );

			Vector3d oSightline = oPosition.getLineOfSight();

			oSightlineElement.setAttribute( LevelXmlSyntax.ATTR_X,
				Double.toString( oSightline.x ) );
			oSightlineElement.setAttribute( LevelXmlSyntax.ATTR_Y,
				Double.toString( oSightline.y ) );
			oSightlineElement.setAttribute( LevelXmlSyntax.ATTR_Z,
				Double.toString( oSightline.z ) );
		}

		// die Flure Zelle fr Zelle dumpen
		Element elFloors = doc.createElement (LevelXmlSyntax.TAG_FLOORS);
		elLevelStructure.appendChild (elFloors);

		// hole die Labyrinth-Constraints und setze Attribute
		int iMaxX, iMaxY, iMaxZ;
		iMaxY = m_p3iConstraints.y;
		iMaxX = m_p3iConstraints.x;
		iMaxZ = m_p3iConstraints.z;
		Debug.out (this.getClass().getName(), "iMaxY = " + iMaxY + ", iMaxX = " + iMaxX + ", iMaxZ = " + iMaxZ, Debug.LEVEL_NOTICE);
		elFloors.setAttribute(LevelXmlSyntax.ATTR_COLS, ""+iMaxX);
		elFloors.setAttribute(LevelXmlSyntax.ATTR_ROWS, ""+iMaxZ);

		// CellHandler zuruecksetzen!
		oCellHandler.reset();

		// ItemHandler zuruecksetzen!
		ItemHandler.getInstance().reset();

		// labyrinth = new Cell[iMaxY][iMaxX][iMaxZ];
		// m_p3iConstraints = new Point3i(labyrinth[1].length, labyrinth.length, labyrinth[1][1].length);
		Cell oCell = null;  Element elCell = null;
		for (int iY = 0; iY < iMaxY; iY++) {

			// neues Floor-Element erzeugen
			Element elCurrentFloor = doc.createElement(LevelXmlSyntax.TAG_FLOOR);

			for (int iX = 0; iX < iMaxX; iX++) {
				for (int iZ = 0; iZ < iMaxZ; iZ++) {

					// Zelle (x,y,z) laden
					oCell = labyrinth[iY][iX][iZ];

					// da CellHandler zurueckgesetzt worden
					// ist -> Zelle dem CellHandler
					// hinzufuegen!
					String sCellTypeName = oCellHandler.addCell( oCell );

					m_asCellTypes[iY][iX][iZ] = sCellTypeName;

					elCell = doc.createElement(LevelXmlSyntax.TAG_CELL);
					elCell.setAttribute(LevelXmlSyntax.ATTR_TYPE, sCellTypeName);

					elCurrentFloor.appendChild(elCell);
				}
			}

			// den aktuellen Flur in die Flurliste packen
			Comment cmCurrentFloor = doc.createComment(" cells for floor " + iY + " ");
			elFloors.appendChild(cmCurrentFloor);
			elFloors.appendChild(elCurrentFloor);
		}

		// aktuellen CellHandler dumpen
		Element elCellTypes = oCellHandler.saveToDOM(doc);
		if (elCellTypes != null) { elGameLevel.appendChild(elCellTypes); }

		// aktuellen ItemHandler dumpen
		ItemHandler oItemHandler = ItemHandler.getInstance();
		Element elItemTypes = oItemHandler.saveToDOM(doc);
		if (elItemTypes != null) { elGameLevel.appendChild(elItemTypes); }

		// aktuellen ResourceHandler dumpen
		ResourceHandler oResourceHandler = pacman3d.util.ResourceHandler.getInstance();
		Element elResources = oResourceHandler.saveToDOM(doc);
		if (elResources != null) { elGameLevel.appendChild(elResources); }

		// anschlieend schreiben wir das alles in die angegebene Datei
		try {

			PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(sLevelFileName)));
			out.print("<?xml version=\"1.0\"?>"); // encoding=\"UTF-8\"?>");
			out.print(elGameLevel.toString());
			out.close();

			/*org.apache.xml.serialize.OutputFormat oOut =
			  new org.apache.xml.serialize.OutputFormat(doc);
			org.apache.xml.serialize.XMLSerializer oXMLSer =
			  new org.apache.xml.serialize.XMLSerializer(
			    new BufferedWriter(new FileWriter(sLevelFileName)),oOut);
			oXMLSer.serialize(doc);
			oXMLSer.endDocument();*/

		} catch (java.io.IOException ex) {
			Debug.out (this.getClass().getName(), "+ cannot save level to url: " + ex, Debug.LEVEL_CRITICAL);
			return;
		}

	} catch (Throwable t) {
		t.printStackTrace ();
	}

  }


}