package pacman3d.intro;

import java.awt.BorderLayout;
import java.awt.Canvas;
import java.awt.GraphicsConfiguration;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import javax.swing.JFrame;

import java.net.InetAddress;
import java.net.UnknownHostException;

import javax.media.j3d.BranchGroup;
import javax.media.j3d.Canvas3D;
import javax.media.j3d.Switch;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;

import javax.vecmath.Point3f;
import javax.vecmath.Vector3d;
import javax.vecmath.Vector3f;

import pacman3d.Game;
import pacman3d.net.NetworkListener;
import pacman3d.net.NetworkMessage;
import pacman3d.net.NetworkService;
import pacman3d.net.TCPClient;
import pacman3d.net.TCPServer;
import pacman3d.util.Debug;
import pacman3d.util.LevelMetaInfo;

import com.mnstarfire.loaders3d.Inspector3DS;
import com.sun.j3d.utils.universe.SimpleUniverse;
import ncsa.j3d.loaders.*;
import com.sun.j3d.loaders.*;

/**
 * <b>Title:</b> Java3D-Praktikum - Pacman3D</br>
 * <b>Description:</b><br>
 *
 * Die Hauptklasse des Intropakets: Anzeigen der Eingangssequenz, Men&uuml;funktionen und
 * Konfiguration der Netzwerkschnittstelle.
 * <br>
 *
 * <b>Copyright:</b>	Copyright (c) 2002<br>
 *
 * @author              Netz+Intro<br>
 */
public class Intro implements KeyListener, NetworkListener {


	/*
	 * 		I N T R O P H A S E
	 */

	// hauptSwitch zum umschalten: entering oder menu-room
	private PacmanRoomScene room_scene = null;
	private EnteringRoomScene entering_scene = null;
	private TransformGroup menueTG = null;
	private Switch mainSW = null;
	private BranchGroup introBranch = null;

	// 	navigationscodes (in welchem status befindet sich das intro)
	private static final int NAV_ENTERING = 1;
	private static final int NAV_MENU = 2;
	private static final int NAV_CREDITS = 3;
	private static final int NAV_LEAVING = 4;
	private static final int NAV_MOVINGTOCREDITS = 5;
	private static final int NAV_MOVING_CREDITS_TO_MENU = 6;
	// so gehts los
	private int currentState = NAV_ENTERING;

	/*
	 * 		M E N U E S
	 */

	// menu-container
	private MenuContainer menu = new MenuContainer();

	// navigation: hoch, runter
	private static final boolean UP = true;
	private static final boolean DOWN = false;

	// MenueSlices - Generell
	private int MENU_HAUPT;
	private int MENU_SPIELTYP;
	private int MENU_MULTIPLAYER;
	private int MENU_OPTIONS;

	// Multiplayer - Client
	private int MENU_CLIENT_SERVERIP;
	private int MENU_CLIENT_CONNECT;
	private int MENU_CLIENT_NOIP;
	private int MENU_CLIENT_CRASHED;
	private int MENU_CLIENT_PACMAN;

	// Multiplayer - Server
	private int MENU_SERVER_LEVEL;
	private int MENU_SERVER_CRASHED;
	private int MENU_SERVER_CONNECT;
	private int MENU_SERVER_STARTED;
	private int MENU_SERVER_PACMAN;
	private int MENU_SERVER_SPIELMODUS;
	private int MENU_SERVER_IPSELECT;

	// Singleplayer
	private int MENU_SINGLE_LEVEL;
	private int MENU_SINGLE_PACMAN;
	private int MENU_SINGLE_SPIELMODUS;

	// Besondere MenuEntry-Objekte im Menue
	private MenuEntry_IPInputField ip_C_Entry = null;
	private MenuEntry_LevelInfo lvl_Entry = null;
	private MenuEntry_SimpleText server_host_Entry = null;
	private MenuEntry_JaNeinSimpleText cleanup_Entry = null;
	private MenuEntry_SelectPacman sel_pacman_Entry = null;

	//host-liste bei server-spiel
	private String[] server_hosts = new String[5];
	
	// soll das Intro waehrend des Spiels im Speicher bleiben ?
	private boolean inMemory = true;


	/*
	 * 	G A M E - A N B I N D U N G
	 */

	// Haelt Referenz auf Game-Objekt
	private Game game = null;
	
	// Der "Hauptframe" des Spiels
	private JFrame frame = null;

	// Instanz der Netzwerkschnittstelle
	private NetworkService networkService = null;

	// Java3D - basics
	private static GraphicsConfiguration config =
		SimpleUniverse.getPreferredConfiguration();
	private Canvas3D canvas = null;
	private SimpleUniverse universe = null;

	/*
	 * 						S C H N I T T S T E L L E
	 */

	/**
	 * Standard-Konstruktor.
	 * 
	 * @param game Instanz der Einstiegsklasse von Pacman3D
	 */
	public Intro(Game game) {
		if (game == null)
			throw new IllegalArgumentException("uups. game is null.");
		this.game = game;
		this.frame = game.getGameFrame();
	}


	/**
	 * Methode regelt das Laden der Ressourcen, die Anzeige des Splash-Screens
	 * und den Introstart durch das Starten der Eingangsequenz.
	 */
	public void showIntro() {

		Debug.out(getClass().getName(), "showIntro() called.", Debug.LEVEL_NOTICE);

		// wie lange brauchen wir zum laden...
		long sTime = System.currentTimeMillis();

		/* STUFE 1 - laden des menues, hosts checken */

		// suche brauchbare ip (stelle die ersten fuenf gefundenen zur auswahl)
		checkServerHosts();

		// an dieser stelle schon mal menue-tg laden
		if (menueTG == null)
			menueTG = createMenuEntries();

		/* STUFE 2 - anzeigen des loading-screens  */

		// splash-screen
		Canvas loading = new LoadingCanvas();
		loading.setSize(frame.getWidth(), frame.getHeight());
		frame.getContentPane().setLayout(new BorderLayout());
		frame.getContentPane().add("Center", loading);
		frame.setVisible(true);

		/* STUFE 3 - scene laden */

		// universe + canvas fuer aufbau
		canvas = new Canvas3D(config);
		universe = new SimpleUniverse(canvas);

		// scene laden
		if (room_scene == null)
			room_scene = new PacmanRoomScene(universe, this);

		if (entering_scene == null)
			entering_scene = new EnteringRoomScene(universe, this);

		if (introBranch == null) {

			introBranch = new BranchGroup();
			// umschalter entering_scene - room_scene
			mainSW = new Switch(0);
			mainSW.setCapability(Switch.ALLOW_SWITCH_WRITE);
			mainSW.setCapability(Switch.ALLOW_SWITCH_READ);
			mainSW.setBounds(Defaults.BIG_BOUNDS);
			// child 0
			mainSW.addChild(entering_scene.createEnteringScene());
			// child 1
			TransformGroup tg = new TransformGroup();
			tg.addChild(room_scene.createRoom());
			tg.addChild(menueTG);
			mainSW.addChild(tg);
			// switch hinzufuegen
			introBranch.addChild(mainSW);
			introBranch.compile();
		}

		// building universe
		universe.addBranchGraph(introBranch);

		/* STUFE 4 - scene anzeigen */

		// splash-screen weg und introsequenz init
		frame.getContentPane().remove(loading);
		loading = null;
		frame.getContentPane().add("Center", canvas);

		// keyListener anbinden
		frame.addKeyListener(this);
		canvas.addKeyListener(this);

		// startposition setzen
		frame.setVisible(true);

		// introsequenz: position setzen und starten
		Transform3D universeTransform = new Transform3D();
		universeTransform.set(new Vector3d(-1.3d, 0.0d, 10.0d));
		universe
			.getViewingPlatform()
			.getMultiTransformGroup()
			.getTransformGroup(0)
			.setTransform(universeTransform);

		// ladezeit ausgeben
		long eTime = System.currentTimeMillis();
		Debug.out(
			getClass().getName(),
			"Introladenzeit : " + ((eTime - sTime) / 1000) + " Sekunden\n",
			Debug.LEVEL_NOTICE);

		// introsequenz: starten
		entering_scene.setEnable(true);

	}

	/**
	 * R&auml;mt in Abh&auml;nigkeit von <i>inMemory</i> den Speicher frei und
	 * entfernt Intro-Frame-Inhalt.
	 */
	public void hideIntro() {

		Debug.out(
			getClass().getName(),
			"hideIntro() called. inMemory = " + (inMemory ? "true" : "false"),
			Debug.LEVEL_NOTICE);

		// networkListener-Aufruf deaktivieren (ab hier sind wir nicht mehr zustaendig)
		if (networkService != null)
			networkService.removeNetworkListener(this);

		// keyListener entfernen
		canvas.removeKeyListener(this);
		frame.removeKeyListener(this);

		// weg mit canvas
		frame.getContentPane().remove(canvas);
		frame = null;

		// im intro referenzen loeschen
		ip_C_Entry = null;
		server_host_Entry = null;
		cleanup_Entry = null;
		menu = null;
		universe = null;
		canvas = null;

		// falls intensiv aufraeumen...
		if (!inMemory) {
			room_scene = null;
			entering_scene = null;
			introBranch = null;
			menueTG = null;
		}

		// freimachen
		Runtime.getRuntime().gc();

	}

	/**
	 * Ist cleanUp gesetzt, erfolgen gr&uuml;ndliche Aufr&auml;marbeiten, um
	 * Arbeitsspeicherbedarf zu verringern.
	 * 
	 * @param inMemory bestimmt, ob das Intro w&auml;hrend des Spiels im Speicher
	 * gehalten werden soll
	 */
	public void setCleanUp(boolean inMemory) {
		this.inMemory = inMemory;
	}

	/**
	 * Gibt LevelMetaInfo-Instanz mit ge&auml;hlten Level zur&uuuml;ck.
	 * 
	 * @return eine Klasse, welche das ausgew&auml;hlte Level kapselt
	 */
	public LevelMetaInfo getSelectedLevel() {
		return lvl_Entry.getLevelInfo();
	}

	/**
	 * Ausgew&auml;hltes Pacman Model.
	 * 
	 * @return das ausgew&auml;hlte Pacman-Model in String-From
	 */
	public String getSelectedPacmanModel() {
		return sel_pacman_Entry.getSelectedPacmanModel();
	}
	
	/**
	 * Ausgew&auml;hlter Spielmodus.
	 * 
	 * @return Pacman-Spielmodus
	 */
	public boolean getSelectedSpielModus() {
		return sel_pacman_Entry.getSelectedSpielModus();
	}

	/*
	 * 						S Z E N E N - A U F B A U
	 */

	/**
	 * Erstellt die Java3D-Texte im Men&uuml;.
	 * 
	 * @return eine <i>TransformGroup</i> mit den Men&uuml;textobjekten.
	 */
	private TransformGroup createMenuEntries() {

		// Menuposition festlegen
		TransformGroup objRoot = new TransformGroup();
		Transform3D t3d = new Transform3D();
		t3d.setTranslation(new Vector3f(-6.0f, 12.0f, -70.0f));
		objRoot.setTransform(t3d);

		// Menues generieren...
		// Hauptmenue
		MENU_HAUPT = menu.addSlice();
		menu.addItem(MENU_HAUPT, "Neues Spiel starten ", new Point3f(0.0f, 0.0f, 0.0f));
		menu.addItem(MENU_HAUPT, "Intro wiederholen ", new Point3f(0.0f, -1.5f, 0.0f));
		menu.addItem(MENU_HAUPT, "Credits ", new Point3f(0.0f, -3.0f, 0.0f));
		menu.addItem(MENU_HAUPT, "Optionen ", new Point3f(0.0f, -4.5f, 0.0f));
		menu.addItem(MENU_HAUPT, "Beenden ", new Point3f(0.0f, -6.5f, 0.0f));

		// SpielTyp-Menue
		MENU_SPIELTYP = menu.addSlice();
		menu.addEntry(
			MENU_SPIELTYP,
			new MenuEntry_SimpleText("Neues Spiel ", new Point3f(7.0f, 3.0f, 0.0f)));
		menu.addItem(MENU_SPIELTYP, "Singleplayer ", new Point3f(0.0f, 0.0f, 0.0f));
		menu.addItem(MENU_SPIELTYP, "Multiplayer ", new Point3f(0.0f, -1.5f, 0.0f));
		menu.addItem(MENU_SPIELTYP, "Zurck ", new Point3f(0.0f, -3.5f, 0.0f));

		// Optionen
		MENU_OPTIONS = menu.addSlice();

		cleanup_Entry =
			new MenuEntry_JaNeinSimpleText(new Point3f(10.0f, -1.5f, 0.0f), false);
		menu.addEntry(MENU_OPTIONS, cleanup_Entry);
		menu.addEntry(
			MENU_OPTIONS,
			new MenuEntry_SimpleText("Optionen ", new Point3f(6.0f, 3.0f, 0.0f)));
		if (cleanup_Entry.getState() != inMemory)
			cleanup_Entry.changeState();
		menu.addItem(
			MENU_OPTIONS,
			"Intro im Speicher halten ",
			new Point3f(-1.5f, -1.5f, 0.0f));
		menu.addItem(MENU_OPTIONS, "Weiter ", new Point3f(-1.5f, -6.0f, 0.0f));
		menu.addItem(MENU_OPTIONS, "Abbruch ", new Point3f(-1.5f, -7.5f, 0.0f));

		// Multiplayer-Menue
		MENU_MULTIPLAYER = menu.addSlice();
		menu.addEntry(
			MENU_MULTIPLAYER,
			new MenuEntry_SimpleText("Spieltyp whlen ", new Point3f(6.5f, 3.0f, 0.0f)));
		menu.addItem(
			MENU_MULTIPLAYER,
			"Client anmelden ",
			new Point3f(0.0f, 0.0f, 0.0f));
		menu.addItem(
			MENU_MULTIPLAYER,
			"Server starten ",
			new Point3f(0.0f, -1.5f, 0.0f));
		menu.addItem(MENU_MULTIPLAYER, "Zurck ", new Point3f(0.0f, -3.5f, 0.0f));

		// ServerIP-Menu (neues spiel - an server anmelden)
		MENU_CLIENT_SERVERIP = menu.addSlice();
		ip_C_Entry = new MenuEntry_IPInputField(15, new Point3f(1.0f, -3.0f, 0.0f));
		menu.addEntry(MENU_CLIENT_SERVERIP, ip_C_Entry);
		menu.addEntry(
			MENU_CLIENT_SERVERIP,
			new MenuEntry_SimpleText(
				"Bitte Server-Adresse eingeben ",
				new Point3f(-1.0f, -1.0f, 0.0f)));
		menu.addItem(
			MENU_CLIENT_SERVERIP,
			"Annehmen ",
			new Point3f(-1.0f, -7.5f, 0.0f));
		menu.addItem(MENU_CLIENT_SERVERIP, "Lschen ", new Point3f(5.0f, -7.5f, 0.0f));
		menu.addItem(MENU_CLIENT_SERVERIP, "Zurck ", new Point3f(10.0f, -7.5f, 0.0f));

		// keine ip adresse (neues spiel - an server anmelden - ip-eingabe)
		MENU_CLIENT_NOIP = menu.addSlice();
		menu.addEntry(
			MENU_CLIENT_NOIP,
			new MenuEntry_SimpleText(
				"Bitte korrekte IP eingeben ",
				new Point3f(0.5f, -2.0f, 0.0f)));
		menu.addItem(MENU_CLIENT_NOIP, "Zurck ", new Point3f(4.5f, -4.0f, 0.0f));

		// level-auswahl (bei singleplayer-spiel)
		MENU_SINGLE_LEVEL = menu.addSlice();
		lvl_Entry = new MenuEntry_LevelInfo();
		menu.addEntry(MENU_SINGLE_LEVEL, lvl_Entry);
		menu.addEntry(
			MENU_SINGLE_LEVEL,
			new MenuEntry_SimpleText("Bitte Level whlen ", new Point3f(6.0f, 3.0f, 0.0f)));
		menu.addItem(MENU_SINGLE_LEVEL, "Naechstes ", new Point3f(-2.5f, -7.5f, 0.0f));
		menu.addItem(MENU_SINGLE_LEVEL, "Vorheriges ", new Point3f(3.0f, -7.5f, 0.0f));
		menu.addItem(MENU_SINGLE_LEVEL, "OK ", new Point3f(8.5f, -7.5f, 0.0f));
		menu.addItem(MENU_SINGLE_LEVEL, "Zurck ", new Point3f(11.0f, -7.5f, 0.0f));

		// an server anmelden (bei client-spiel)
		MENU_CLIENT_CONNECT = menu.addSlice();
		menu.addEntry(
			MENU_CLIENT_CONNECT,
			new MenuEntry_SimpleText(
				"Verbinden zum Netzwerk... ",
				new Point3f(0.0f, -2.0f, 0.0f)));
		menu.addEntry(
			MENU_CLIENT_CONNECT,
			new MenuEntry_SimpleText("ESC - Abbruch ", new Point3f(2.5f, -4.0f, 0.0f)));

		// keine verbindung zum server
		MENU_CLIENT_CRASHED = menu.addSlice();
		menu.addEntry(
			MENU_CLIENT_CRASHED,
			new MenuEntry_SimpleText(
				"Keine Verbindungsaufbau mglich ",
				new Point3f(-1.5f, -2.0f, 0.0f)));
		menu.addItem(MENU_CLIENT_CRASHED, "Zurck ", new Point3f(4.5f, -4.0f, 0.0f));

		// server gestartet-meldung
		MENU_SERVER_STARTED = menu.addSlice();
		server_host_Entry =
			new MenuEntry_SimpleText("test-text", new Point3f(5.0f, -2.0f, 0.0f));
		menu.addEntry(MENU_SERVER_STARTED, server_host_Entry);
		menu.addEntry(
			MENU_SERVER_STARTED,
			new MenuEntry_SimpleText("Server gestartet ", new Point3f(6.0f, 3.0f, 0.0f)));
		menu.addEntry(
			MENU_SERVER_STARTED,
			new MenuEntry_SimpleText("Host ist", new Point3f(1.0f, -2.0f, 0.0f)));
		menu.addItem(MENU_SERVER_STARTED, "Weiter ", new Point3f(1.5f, -4.0f, 0.0f));
		menu.addItem(MENU_SERVER_STARTED, "Abbruch ", new Point3f(6.0f, -4.0f, 0.0f));

		// pacman waehlen
		MENU_SINGLE_PACMAN = menu.addSlice();
		sel_pacman_Entry = new MenuEntry_SelectPacman();
		menu.addEntry(MENU_SINGLE_PACMAN, sel_pacman_Entry);
		menu.addEntry(
			MENU_SINGLE_PACMAN,
			new MenuEntry_SimpleText(
				"Bitte Pacman whlen ",
				new Point3f(6.0f, 3.0f, 0.0f)));
		menu.addItem(MENU_SINGLE_PACMAN, "Naechstes ", new Point3f(-2.5f, -7.5f, 0.0f));
		menu.addItem(MENU_SINGLE_PACMAN, "Vorheriges ", new Point3f(3.0f, -7.5f, 0.0f));
		menu.addItem(MENU_SINGLE_PACMAN, "OK ", new Point3f(8.5f, -7.5f, 0.0f));
		menu.addItem(MENU_SINGLE_PACMAN, "Zurck ", new Point3f(11.0f, -7.5f, 0.0f));

		// Spielmodus-menu
		MENU_SERVER_SPIELMODUS = menu.addSlice();
		menu.addEntry(
			MENU_SERVER_SPIELMODUS,
			new MenuEntry_SimpleText("Spielmodus ", new Point3f(7.0f, 3.0f, 0.0f)));
		menu.addItem(
			MENU_SERVER_SPIELMODUS,
			"Miteinander ",
			new Point3f(0.0f, 0.0f, 0.0f));
		menu.addItem(
			MENU_SERVER_SPIELMODUS,
			"Gegeneinander ",
			new Point3f(0.0f, -1.5f, 0.0f));
		menu.addItem(MENU_SERVER_SPIELMODUS, "Zurck ", new Point3f(0.0f, -3.5f, 0.0f));

		// als server level waehlen
		MENU_SERVER_LEVEL = menu.cloneSlice(MENU_SINGLE_LEVEL);

		// server connect
		MENU_SERVER_CONNECT = menu.cloneSlice(MENU_CLIENT_CONNECT);

		// server crashed
		MENU_SERVER_CRASHED = menu.cloneSlice(MENU_CLIENT_CRASHED);

		// als server pacman waehlen
		MENU_SERVER_PACMAN = menu.cloneSlice(MENU_SINGLE_PACMAN);

		// pacman bei client-spiel waehlen
		MENU_CLIENT_PACMAN = menu.cloneSlice(MENU_SINGLE_PACMAN);

		// server ip bestimmen, falls mehrere aktiv
		MENU_SERVER_IPSELECT = menu.addSlice();
		menu.addEntry(
			MENU_SERVER_IPSELECT,
			new MenuEntry_SimpleText("Host-IP whlen ", new Point3f(6.0f, 3.0f, 0.0f)));
		menu.addItem(MENU_SERVER_IPSELECT, "Zurck ", new Point3f(0.0f, -8.0f, 0.0f));
		for (int i = 0; i < 5; i++) {
			if (server_hosts[i] != null) {
				Debug.out(
					getClass().getName(),
					"found host " + server_hosts[i] + ". adding to server_hosts.",
					Debug.LEVEL_NOTICE);
				menu.addItem(
					MENU_SERVER_IPSELECT,
					server_hosts[i],
					new Point3f(0.0f, (-i * 1.5f), 0.0f));
			}

		}

		//java3d-menu holen
		objRoot.addChild(menu.getMenu());
		return objRoot;
	}

	/*
	 * 						S U P P O R T
	 */

	/**
	 * L&auml;dt ein 3DS-Model.
	 * 
	 * @param file Dateiname
	 * @return Model als <i>TransformGroup</i>
	 */
	public static TransformGroup loadTGfrom3DS(String file) {
		Debug.out(
			Intro.class.getName(),
			new String("Loading: " + Defaults.BASE_PATH + file),
			Debug.LEVEL_NOTICE);

		TransformGroup modelTG = null;
		Inspector3DS loader = new Inspector3DS(Defaults.BASE_PATH + file);
		loader.setTextureLightingOn();
		loader.parseIt();
		return loader.getModel();
	}

	/**
	 * Setzt die gew&uml;schte Betrachterposition in der Szene.
	 * 
	 * @param value Indexnummer der Position
	 */
	private void setViewPosition(int value) {
		Transform3D basic = new Transform3D();
		Transform3D ext = new Transform3D();
		switch (value) {

			case NAV_ENTERING :
				basic.setTranslation(new Vector3d(0.0d, 0.0d, 2.5d));
				break;

			case NAV_MENU :
				basic.setTranslation(new Vector3f(0.0f, 10.0f, -38.0f));
				break;

			case NAV_CREDITS :
				basic.setTranslation(new Vector3f(0.0f, 10.0f, -38.0f));
				ext.rotY(Math.PI / 180 * 70);
				basic.mul(ext);
				break;

			case NAV_LEAVING :
				break;
		}
		universe.getViewingPlatform().getViewPlatformTransform().setTransform(basic);
	}

	/**
	 * Hostadressen des LocalHost ermitteln.
	 */
	private void checkServerHosts() {

		InetAddress[] localHosts = null;
		int count = 0;
		try {
			localHosts =
				InetAddress.getAllByName(InetAddress.getLocalHost().getHostAddress());
			// anzahl hosts begrenzen
			if (localHosts.length > 5)
				count = 5;
			else if (localHosts.length > 0)
				count = localHosts.length;
			// falls count > 0 dann setzen, sonst deafult, wei bei exception
			if (count > 0)
				for (int i = 0; i < count; i++) {
					server_hosts[i] = new String(localHosts[i].getHostAddress());
				} else
				server_hosts[0] = new String(InetAddress.getLocalHost().getHostAddress());
		} catch (Exception ex) {
			Debug.out(
				Intro.class.getName(),
				ex + " in checkServerHosts().\nSetting default.",
				Debug.LEVEL_NOTICE);
			server_hosts = new String[5];
			try {
				server_hosts[0] = new String(InetAddress.getLocalHost().getHostAddress());
			} catch (UnknownHostException ex2) {
				server_hosts[0] = new String("127.0.0.1");
			}
		}

	}

	/*
	 *					 		N E T W O R K - L I S T E N E R
	 */

	/**
	 * Wird vom <i>NetworkService</i> aufgerufen.
	 * 
	 * @param message Erhaltene <i>NetworkMessage</i>.
	 */
	public void networkMessageReceived(NetworkMessage message) {
	}

	/**
	 * Liefert den Statuscode der Netzwerkschicht.
	 * 
	 * @param status der Netzwerkschichtstatus
	 * @see NetworkListener
	 */
	public void networkStatus(int status) {
		if (networkService != null) {
			// server
			if (networkService.isServer()) {
				if (status == NetworkListener.STATUS_CRASHED)
					menu.setActiveSlice(MENU_SERVER_CRASHED);
				else if (status == NetworkListener.STATUS_CONNECTED) {

					// richtige Instanz holen
					server_host_Entry =
						(MenuEntry_SimpleText) menu.getEntryByClass(
							MENU_SERVER_STARTED,
							MenuEntry_SimpleText.class);

					server_host_Entry.setString(networkService.getHost());
					menu.setActiveSlice(MENU_SERVER_STARTED);
				}
				// client
			} else {
				if (status == NetworkListener.STATUS_CRASHED)
					menu.setActiveSlice(MENU_CLIENT_CRASHED);
				else if (status == NetworkListener.STATUS_CONNECTED) {
					// okay zur pacman-wahl (multiplayer - client)
						((MenuEntry_SelectPacman) menu.getEntryByClass(
						MENU_CLIENT_PACMAN,
						MenuEntry_SelectPacman.class)).initialize();
					menu.setActiveSlice(MENU_CLIENT_PACMAN);
				}

			}

		}
	}

	/*
	 * 							N E T W O R K - S U P P O R T
	 */

	/**
	 * Starten der Netzwerkschicht.
	 * 
	 * @param isServer wird Server oder Client erzeugt
	 * @param ip die IP-Adresse, unter der der Netzwerkdienst gestartet wird
	 * 
	 */
	private void initializeNetwork(boolean isServer, String ip) {
		if (isServer)
			networkService = new TCPServer(ip);
		else
			networkService = new TCPClient(ip);
		networkService.addNetworkListener(this);
		networkService.startNetwork();
	}

	/**
	 * Terminiert die Netzwerkschicht.
	 */
	private void terminateNetwork() {
		if (networkService != null) {
			networkService.removeNetworkListener(this);
			networkService.closeNetwork();
		}
		networkService = null;
	}

	/*
	 * 						K E Y - L I S T E N E R
	 */

	public void keyTyped(KeyEvent k_event) {
	}
	public void keyReleased(KeyEvent k_event) {
	}

	/**
	 * Abfangen der Tastaturevents.
	 */
	public void keyPressed(KeyEvent k_event) {

		// welche taste wurde gedrueckt ?
		int key = k_event.getKeyCode();

		// globale tastatursteuerung
		switch (currentState) {

			case NAV_ENTERING :
				enteringHandler(key);
				break;

			case NAV_MENU :
				menuHandler(key);
		// nk testing
				// menuViewHandler(key);
				break;

			case NAV_CREDITS :
				creditHandler(key);
				break;
			
			// nk testing	
			case NAV_MOVINGTOCREDITS :
				creditInHandler(key);
				break;
				
			case NAV_MOVING_CREDITS_TO_MENU:
				//creditHandler(key);
				//menuViewHandler(key);
				toMenuHandler(key);
				break;

		}

	}

	/*
	 * 						N A V I G A T I O N
	 */

	
	/**
	 * Umschalten von Pacman-Animation zum Men&uuml;.
	 * 
	 * @param key Tastaturcode aus dem KeyEvent
	 */
	private void menuViewHandler(int key) {
		if (key == 27 || key == 112) {
			currentState = NAV_MENU;
			room_scene.setAllDisable();
			room_scene.setViewPos(2);
		}
		
	}


	/**
	 * Weiterleitung zum Men&uuml;- oder Credit-Handler.
	 * 
	 * @param key Tastaturcode aus dem KeyEvent
	 */	
	private void toMenuHandler(int key) {
		// ESC
		if (key == 27) {
			menuHandler(key);
		}
		// F1
		if (key == 112) {
			creditInHandler(key);
		}
		
	}
	
	
	/**
	 * TastaturEvent-Steuerung w&auml;hrend Anzeige der Credits.
	 * 
	 * @param key Tastaturcode aus dem KeyEvent
	 */
	private void creditInHandler(int key){
		// ESC
		if (key == 27) {
			currentState = NAV_MOVING_CREDITS_TO_MENU;
			room_scene.setAllDisable();
			room_scene.setViewPos(3);
		} 
		// F1
		if (key == 112) {
			currentState = NAV_MENU;
			room_scene.setDisable(1);
			room_scene.setDisable(3);	
			room_scene.setEnable(2);
		}	
	}


	/**
	 * Umschalten von Credits zum Men&uuml;.
	 * 
	 * @param key Tastaturcode aus dem KeyEvent
	 */
	private void creditHandler(int key) {
		if (key == 27) {
			currentState = NAV_MENU;
			room_scene.setAllDisable();
			room_scene.setViewPos(2);
			
		}
	}

	/**
	 * Umschalten von Eingangssequenz zum Men&uuml;.
	 * 
	 * @param key Tastaturcode aus dem KeyEvent
	 */
	private void enteringHandler(int key) {
		if (key == 27) {
			Debug.out(
				getClass().getName(),
				"enteringHandler(): switch to menu.",
				Debug.LEVEL_NOTICE);
			enteringSceneDone();
		}
	}

	/**
	 * Schaltet von Eingangsequenz zur Men&uuml;ansicht.
	 */
	protected void enteringSceneDone() {
		entering_scene.setEnable(false);
		mainSW.setWhichChild(1);
		currentState = NAV_MENU;
		room_scene.setEnable(1);
	}


	/**
	 * Behandelt Men&uuml;navigation mit Tastatur.
	 * 
	 * @param key Tastaturcode aus dem KeyEvent
	 */
	private void handleUPDOWN(int key) {
		if (key == 38)
			menu.changeEntry(UP);
		else if (key == 40)
			menu.changeEntry(DOWN);
	}


	/**
	 * Behandelt Men&uuml;navigation mit Tastatur.
	 * 
	 * @param key Tastaturcode aus dem KeyEvent
	 */
	private void handleLEFTRIGHT(int key) {
		if (key == 37)
			menu.changeEntry(UP);
		else if (key == 39)
			menu.changeEntry(DOWN);
	}

	/**
	 * Methode zum Behandeln von KeyEvents. Anhand der Tastatureingaben werden
	 * die Men&uuml;eintra&auml;ge angepasst.
	 * 
	 * @param key Tastaturcode aus dem KeyEvent
	 */
	private void menuHandler(int key) {
		
		int as = menu.getActiveSlice();
		int ae = menu.getActiveEntry();

		if (as == MENU_HAUPT) {
			// hauptmenu
			// hoch runter okay
			handleUPDOWN(key);

			if (key == 10) {

				// neues spiel
				if (ae == 0) {
					menu.setActiveSlice(MENU_SPIELTYP);
				} else

					// intro wiederholen
					if (ae == 1) {
						mainSW.setWhichChild(0);
						entering_scene.setEnable(true);
					} else

						// credits
						if (ae == 2) {
							// nk testing
							currentState = NAV_MOVINGTOCREDITS;
							//printModus();
							//currentState = NAV_CREDITS;
							// //setViewPosition(NAV_CREDITS);
							// //cba:25.2.02
							room_scene.setEnable(3);
						} else

							// optionen
							if (ae == 3) {
								menu.setActiveSlice(MENU_OPTIONS);
							} else

								// beenden
								if (ae == 4) {
									hideIntro();
									// temp
									if (game != null)
										game.quit();
									else
										System.exit(0);
								}
			}

		} else if (as == MENU_SPIELTYP) {
			//spieltyp

			handleUPDOWN(key);

			if (key == 10) {

				// singleplayer
				if (ae == 0) {
					// richtige Instanz holen
					lvl_Entry =
						(MenuEntry_LevelInfo) menu.getEntryByClass(
							MENU_SINGLE_LEVEL,
							MenuEntry_LevelInfo.class);
					lvl_Entry.initialize();
					menu.setActiveSlice(MENU_SINGLE_LEVEL);

				} else

					// multiplayer
					if (ae == 1) {
						menu.setActiveSlice(MENU_MULTIPLAYER);
					} else

						// zurueck
						if (ae == 2) {
							menu.setActiveSlice(MENU_HAUPT);
						}
			}

		} else if (as == MENU_MULTIPLAYER) {
			//multiplayer

			handleUPDOWN(key);

			if (key == 10) {

				// client
				if (ae == 0) {
					menu.setActiveSlice(MENU_CLIENT_SERVERIP);
				} else

					// server
					if (ae == 1) {
						menu.setActiveSlice(MENU_SERVER_IPSELECT);
						menu.setActiveEntry(1);
					} else

						// zurueck
						if (ae == 2) {
							menu.setActiveSlice(MENU_SPIELTYP);
						}
			}
		} else if (as == MENU_CLIENT_SERVERIP) {

			handleLEFTRIGHT(key);

			// richtige Instanz holen
			ip_C_Entry =
				(MenuEntry_IPInputField) menu.getEntryByClass(
					MENU_CLIENT_SERVERIP,
					MenuEntry_IPInputField.class);

			if (((key > 47) && (key < 58)) || (key == 46)) {

				ip_C_Entry.addKey(key);
			} else if (key == 8) {
				ip_C_Entry.removeKey();
			} else if (key == 10) {
				// annehmen
				if (ae == 0) {
					if (ip_C_Entry.isIPString(ip_C_Entry.getString())) {
						menu.setActiveSlice(MENU_CLIENT_CONNECT);
						initializeNetwork(false, ip_C_Entry.getString());
					} else
						menu.setActiveSlice(MENU_CLIENT_NOIP);

				} else
					// loeschen
					if (ae == 1) {
						ip_C_Entry.clearString();
						menu.changeEntry(UP);
					} else
						// zurueck
						if (ae == 2) {
							menu.setActiveSlice(MENU_MULTIPLAYER);
						}

			}

		} else if (as == MENU_SINGLE_LEVEL) {
			// levelangabe bei singleplayer

			handleLEFTRIGHT(key);

			if (key == 10) {

				// richtige Instanz holen
				lvl_Entry =
					(MenuEntry_LevelInfo) menu.getEntryByClass(
						MENU_SINGLE_LEVEL,
						MenuEntry_LevelInfo.class);
				// naechstes
				if (ae == 0) {
					lvl_Entry.forward();
				} else

					// vorheriges
					if (ae == 1) {
						lvl_Entry.backward();
					} else

						// annehmen
						if (ae == 2) {
							// richtige Instanz holen
							((MenuEntry_SelectPacman) menu
								.getEntryByClass(MENU_SINGLE_PACMAN, MenuEntry_SelectPacman.class))
								.initialize();
							menu.setActiveSlice(MENU_SINGLE_PACMAN);

						} else

							// zurueck
							if (ae == 3) {
								menu.setActiveSlice(MENU_SPIELTYP);
							}

			}
		} else if (as == MENU_SERVER_LEVEL) {
			// levelangabe bei multiplayer

			handleLEFTRIGHT(key);

			if (key == 10) {

				lvl_Entry =
					(MenuEntry_LevelInfo) menu.getEntryByClass(
						MENU_SERVER_LEVEL,
						MenuEntry_LevelInfo.class);
				// naechstes
				if (ae == 0) {
					lvl_Entry.forward();
				} else

					// vorheriges
					if (ae == 1) {
						lvl_Entry.backward();
					} else

						// annehmen
						if (ae == 2) {
							// richtige Instanz holen
							((MenuEntry_SelectPacman) menu
								.getEntryByClass(MENU_SERVER_PACMAN, MenuEntry_SelectPacman.class))
								.initialize();
							menu.setActiveSlice(MENU_SERVER_PACMAN);

						} else

							// zurueck
							if (ae == 3) {
								terminateNetwork();
								menu.setActiveSlice(MENU_MULTIPLAYER);
							}

			}
		} else if (as == MENU_CLIENT_NOIP) {
			// falsche ip-eingabe

			if (key == 10) {

				// zurueck
				if (ae == 0) {
					menu.setActiveSlice(MENU_CLIENT_SERVERIP);
				}
			}
		} else if (as == MENU_CLIENT_CRASHED) {
			// client crashed

			if (key == 10) {

				// zurueck
				if (ae == 0) {
					menu.setActiveSlice(MENU_CLIENT_SERVERIP);
				}
			}
		} else if (as == MENU_SERVER_CRASHED) {
			// server crashed

			if (key == 10) {

				// zurueck
				if (ae == 0) {
					menu.setActiveSlice(MENU_MULTIPLAYER);
				}
			}
		} else if (as == MENU_SERVER_STARTED) {
			// server started

			handleLEFTRIGHT(key);

			if (key == 10) {

				// weiter
				if (ae == 0) {
					// richtige Instanz holen
					lvl_Entry =
						(MenuEntry_LevelInfo) menu.getEntryByClass(
							MENU_SERVER_LEVEL,
							MenuEntry_LevelInfo.class);
					lvl_Entry.initialize();

					menu.setActiveSlice(MENU_SERVER_LEVEL);
				} else

					// abbruch
					if (ae == 1) {
						terminateNetwork();
						menu.setActiveSlice(MENU_SPIELTYP);
					}

			}

		} else if (as == MENU_SERVER_SPIELMODUS) {
			//spielmodus

			handleUPDOWN(key);

			if (key == 10) {

				// richtige Instanz holen
				sel_pacman_Entry =
					(MenuEntry_SelectPacman) menu.getEntryByClass(
						MENU_SERVER_PACMAN,
						MenuEntry_SelectPacman.class);

				// miteinander
				if (ae == 0) {
					// setzt den spielmodus
					sel_pacman_Entry.selectSpielModus(true);
					// okay zum spiel (miteinander - server)
					hideIntro();
					game.startGame(networkService);
				} else

					// gegeneinander
					if (ae == 1) {
						// setzt den spielmodus
						sel_pacman_Entry.selectSpielModus(false);
						// okay zum spiel (gegeneinander - server)
						hideIntro();
						game.startGame(networkService);
					} else

						// zurueck
						if (ae == 2) {
							menu.setActiveSlice(MENU_SERVER_PACMAN);
						}
			}

		} else if (as == MENU_SINGLE_PACMAN) {
			// singleplayer waehlt pacman
			handleLEFTRIGHT(key);
			if (key == 10) {

				// richtige Instanz holen
				sel_pacman_Entry =
					(MenuEntry_SelectPacman) menu.getEntryByClass(
						MENU_SINGLE_PACMAN,
						MenuEntry_SelectPacman.class);
				// naechstes
				if (ae == 0) {
					sel_pacman_Entry.forward();
				} else

					// vorheriges
					if (ae == 1) {
						sel_pacman_Entry.backward();
					} else

						// annehmen
						if (ae == 2) {
							//menu.setActiveSlice(MENU_C_SPIELMODUS);
							// okay zum spiel (gegeneinander)
							hideIntro();
							game.startGame();

						} else

							// zurueck
							if (ae == 3) {
								menu.setActiveSlice(MENU_SINGLE_LEVEL);
							}

			}

		} else if (as == MENU_SERVER_PACMAN) {

			// singleplayer waehlt pacman
			handleLEFTRIGHT(key);
			if (key == 10) {

				// richtige Instanz holen
				sel_pacman_Entry =
					(MenuEntry_SelectPacman) menu.getEntryByClass(
						MENU_SERVER_PACMAN,
						MenuEntry_SelectPacman.class);
				// naechstes
				if (ae == 0) {
					sel_pacman_Entry.forward();
				} else

					// vorheriges
					if (ae == 1) {
						sel_pacman_Entry.backward();
					} else

						// annehmen
						if (ae == 2) {
							menu.setActiveSlice(MENU_SERVER_SPIELMODUS);
						} else

							// zurueck
							if (ae == 3) {
								menu.setActiveSlice(MENU_SERVER_LEVEL);
							}
			}

		} else if (as == MENU_CLIENT_PACMAN) {

			// singleplayer waehlt pacman
			handleLEFTRIGHT(key);
			if (key == 10) {

				// richtige Instanz holen
				sel_pacman_Entry =
					(MenuEntry_SelectPacman) menu.getEntryByClass(
						MENU_CLIENT_PACMAN,
						MenuEntry_SelectPacman.class);
				// naechstes
				if (ae == 0) {
					sel_pacman_Entry.forward();
				} else

					// vorheriges
					if (ae == 1) {
						sel_pacman_Entry.backward();
					} else

						// annehmen
						if (ae == 2) {
							Debug.out(
								getClass().getName(),
								"Client connected. hideIntro(). startGame().",
								Debug.LEVEL_NOTICE);
							hideIntro();
							game.startGame(networkService);
						} else

							// zurueck
							if (ae == 3) {
								menu.setActiveSlice(MENU_CLIENT_SERVERIP);
							}
			}

		} else if (as == MENU_OPTIONS) {
			// optionen

			handleUPDOWN(key);

			if (key == 10) {
				cleanup_Entry =
					(MenuEntry_JaNeinSimpleText) menu.getEntryByClass(
						MENU_OPTIONS,
						MenuEntry_JaNeinSimpleText.class);

				// intro im speicher halten
				if (ae == 0) {
					cleanup_Entry.changeState();
				} else

					// weiter
					if (ae == 1) {
						inMemory = cleanup_Entry.getState();
						menu.setActiveSlice(MENU_HAUPT);

					} else

						// abbruch
						if (ae == 2) {
							menu.setActiveSlice(MENU_HAUPT);
						}

			}

		} else if (as == MENU_SERVER_IPSELECT) {
			// server ip auswaehlen

			handleUPDOWN(key);

			if (key == 10) {

				// zurueck
				if (ae == 0) {
					menu.setActiveSlice(MENU_SPIELTYP);
				} else
					// host gewaehlt
					if (ae > 0 && ae < 6) {
						menu.setActiveSlice(MENU_SERVER_CONNECT);
						initializeNetwork(true, server_hosts[ae - 1]);
					}
			}

		} else if (as == MENU_CLIENT_CONNECT) {
			// client connect dauert zu lange - dann raus mit ESC
			if (key == 27) {
				terminateNetwork();
				menu.setActiveSlice(MENU_CLIENT_SERVERIP);
			}
		}

	}
	
} // end of class